<?php

/**
 * BebanStore Settings Class
 * 
 * Handles all settings-related functionality for the BebanStore plugin.
 * This includes registering settings, rendering the settings page,
 * and managing option dependencies.
 * 
 * @package BebanStore
 * @since 1.0.0
 */
class BebanStore_Settings {
    /**
     * Plugin options
     * 
     * @var array
     * @access private
     */
    private $options;

    /**
     * Features container
     * 
     * @var array
     * @access private
     */
    private $features;

    /**
     * Settings register instance
     * 
     * @var BebanStore_Settings_Register
     * @access private
     */
    private $register;

    /**
     * Settings render instance
     * 
     * @var BebanStore_Settings_Render
     * @access private
     */
    private $render;

    /**
     * Initialize the class
     * 
     * Sets up the settings page and registers all necessary hooks.
     * 
     * @param array $features Plugin features
     * @since 1.0.0
     */
    public function __construct($features = array()) {
        // Store features
        $this->features = $features;
        
        // Load plugin options
        $this->options = get_option('bebanstore_options', array());

        // Initialize components
        $this->init_components();

        // Add admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Register settings
        add_action('admin_init', array($this, 'register_settings'));

        // Add admin scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));
    }

    /**
     * Initialize components
     */
    private function init_components() {
        // Initialize access manager
        require_once BEBANSTORE_PLUGIN_DIR . 'admin/settings/class-feature-access-manager.php';
        
        // Initialize register component
        require_once BEBANSTORE_PLUGIN_DIR . 'admin/settings/class-settings-register.php';
        $this->register = new BebanStore_Settings_Register($this->options, $this->features);

        // Initialize render component
        require_once BEBANSTORE_PLUGIN_DIR . 'admin/settings/class-settings-render.php';
        $this->render = new BebanStore_Settings_Render($this->options, $this->features);
    }

    /**
     * Enqueue admin scripts and styles
     */
    public function admin_scripts() {
        // Enqueue admin styles
        wp_enqueue_style('bebanstore-admin', BEBANSTORE_PLUGIN_URL . 'assets/css/admin.css', array(), BEBANSTORE_VERSION);
        
        // Enqueue WordPress media scripts
        wp_enqueue_media();
        
        // Enqueue admin scripts
        wp_enqueue_script('bebanstore-admin', BEBANSTORE_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), BEBANSTORE_VERSION, true);
        
        // Localize script
        wp_localize_script('bebanstore-admin', 'bebanstoreAdmin', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('bebanstore-admin-nonce')
        ));
    }

    /**
     * Add admin menu
     * 
     * Adds the plugin's menu item to the WordPress admin menu.
     * 
     * @since 1.0.0
     */
    public function add_admin_menu() {
        add_menu_page(
            'تنظیمات ببان استور',
            'ببان استور',
            'manage_options',
            'bebanstore-settings',
            array($this->render, 'render_settings_page'),
            'dashicons-store',
            59
        );
    }

    /**
     * Register plugin settings
     * 
     * Registers all settings, sections, and fields for the plugin.
     * 
     * @since 1.0.0
     */
    public function register_settings() {
        $this->register->register_settings();
    }

    /**
     * AJAX handler for saving settings
     */
    public function ajax_save_settings() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'bebanstore_save_settings')) {
            wp_send_json_error('Invalid nonce');
            return;
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
            return;
        }

        // Get current options
        $current_options = get_option('bebanstore_options', array());

        // Process form data
        $new_options = array();
        foreach ($_POST as $key => $value) {
            // Skip non-option fields
            if (in_array($key, array('action', 'nonce'))) {
                continue;
            }

            // Handle checkbox fields
            if (strpos($key, 'bebanstore_') === 0) {
                $field_name = substr($key, strlen('bebanstore_'));
                $new_options[$field_name] = $value;
            }
        }

        // Merge with current options
        $merged_options = array_merge($current_options, $new_options);

        // Sanitize options
        $sanitized_options = $this->register->sanitize_options($merged_options);

        // Update options
        $updated = update_option('bebanstore_options', $sanitized_options);

        if ($updated) {
            wp_send_json_success('تنظیمات با موفقیت ذخیره شد.');
        } else {
            wp_send_json_error('خطا در ذخیره تنظیمات.');
        }
    }

    /**
     * Cleanup feature resources
     */
    public function cleanup() {
        remove_action('admin_menu', array($this, 'add_admin_menu'));
        remove_action('admin_init', array($this, 'register_settings'));
        remove_action('admin_enqueue_scripts', array($this, 'admin_scripts'));
    }
} 