<?php
/**
 * Feature Settings Factory Class
 * 
 * @package BebanStore
 * @since 1.0.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class BebanStore_Feature_Settings_Factory {
    /**
     * Default settings container
     * 
     * @var array
     * @access private
     */
    private $default_settings = array();

    /**
     * Initialize the class
     */
    public function __construct() {
        $this->load_default_settings();
    }

    /**
     * Load default settings from all feature settings files
     */
    private function load_default_settings() {
        $features = array(
            'ajax_add_to_cart',
            'sales_progress',
            'custom_quantity_control',
            'variable_price',
            'countdown',
            'express_shipping',
            'tracking_code',
            'product_attributes',
            'product_discount',
            'discount_badges',
            'order_count',
            'order_view',
            'checkout_fields_manager',
            'cart_confirmation',
            'your_savings',
            'wishlist',
            'custom_font',
            'add_to_cart_button_text',
            'save_for_later_cart', // Pro feature preview
            'toast_customization', // Pro feature preview
            'product_mentions_in_content' // Pro feature preview
        );

        // Allow Pro plugin to add features
        $features = apply_filters('bebanstore_feature_settings_factory_features', $features);

        foreach ($features as $feature_id) {
            $settings = $this->get_settings($feature_id);
            if ($settings['has_settings']) {
                $this->default_settings[$feature_id] = $this->extract_default_values($settings['settings']);
            }
        }
    }

    /**
     * Extract default values from settings array
     * 
     * @param array $settings Settings array
     * @return array Default values
     */
    private function extract_default_values($settings) {
        $defaults = array();
        foreach ($settings as $field_id => $field) {
            if (isset($field['default'])) {
                $defaults[$field_id] = $field['default'];
            }
        }
        return $defaults;
    }

    /**
     * Get default settings
     * 
     * @return array Default settings
     */
    public function get_default_settings() {
        return array(
            'feature_settings' => $this->default_settings
        );
    }

    /**
     * Get feature settings
     * 
     * @param string $feature_id Feature ID
     * @return array Feature settings
     */
    public function get_settings($feature_id) {
        // Convert feature_id to file name format
        $file_name = str_replace('_', '-', $feature_id);
        
        // Check if Pro version is available and has this feature
        $pro_settings_file = apply_filters('bebanstore_pro_settings_file_path', '', $feature_id, $file_name);
        if (!empty($pro_settings_file) && file_exists($pro_settings_file)) {
            // Use Pro version if available
            $settings_file = $pro_settings_file;
        } else {
            // Use free version
            $settings_file = BEBANSTORE_PLUGIN_DIR . "admin/settings/features/{$file_name}-settings.php";
            if (!file_exists($settings_file)) {
                return array('has_settings' => false);
            }
        }

        // Reset the settings array
        $settings = array();

        try {
            // Include the settings file
            $settings = include $settings_file;

            if (!is_array($settings)) {
                return array('has_settings' => false);
            }

            // Ensure has_settings is set
            if (!isset($settings['has_settings'])) {
                $settings['has_settings'] = true;
            }

            // Ensure settings array exists
            if (!isset($settings['settings'])) {
                $settings['settings'] = array();
            }

            // Track whether this feature provides Pro-only settings
            $has_pro_settings_flag = isset($settings['has_pro_settings']) ? (bool) $settings['has_pro_settings'] : false;
            $has_separate_pro_settings = isset($settings['pro_only_settings']) && is_array($settings['pro_only_settings']) && !empty($settings['pro_only_settings']);
            if ($has_separate_pro_settings) {
                $has_pro_settings_flag = true;
            }

            // Always merge pro_only_settings if it exists (regardless of Pro activation)
            // This allows showing disabled fields when Pro is not active
            if ($has_separate_pro_settings) {
                $settings['settings'] = array_merge($settings['settings'], $settings['pro_only_settings']);
                // Remove pro_only_settings after merging
                unset($settings['pro_only_settings']);
            }

            // Persist flag so renderer can show Freemium badge
            $settings['has_pro_settings'] = $has_pro_settings_flag;

            // Allow other plugins to modify settings
            return apply_filters("bebanstore_{$feature_id}_settings", $settings);
        } catch (Exception $e) {
            return array('has_settings' => false);
        }
    }

    /**
     * Get all features settings
     * 
     * @return array All features settings
     */
    public function get_all_settings() {
        $features = array(
            'ajax_add_to_cart',
            'sales_progress',
            'custom_quantity_control',
            'variable_price',
            'countdown',
            'express_shipping',
            'tracking_code',
            'product_attributes',
            'product_discount',
            'discount_badges',
            'order_count',
            'order_view',
            'checkout_fields_manager',
            'cart_confirmation',
            'your_savings',
            'wishlist',
            'custom_font',
            'add_to_cart_button_text',
            'save_for_later_cart', // Pro feature preview
            'toast_customization', // Pro feature preview
            'product_mentions_in_content' // Pro feature preview
        );

        // Allow Pro plugin to add features
        $features = apply_filters('bebanstore_feature_settings_factory_features', $features);

        $settings = array();
        foreach ($features as $feature_id) {
            $settings[$feature_id] = $this->get_settings($feature_id);
        }

        return $settings;
    }
} 