<?php
/**
 * کلاس مدیریت وضعیت ویژگی‌ها و فیلدها
 */
class BebanStore_Feature_Status {
    /**
     * وضعیت‌های تعریف شده
     */
    private static $statuses = array(
        'coming_soon' => array(
            'text' => 'به زودی',
            'class' => 'coming-soon',
            'color' => '#f0ad4e',
            'icon' => 'dashicons-clock'
        ),
        'deprecated' => array(
            'text' => 'منسوخ شده',
            'class' => 'deprecated',
            'color' => '#d9534f',
            'icon' => 'dashicons-warning'
        ),
        'beta' => array(
            'text' => 'نسخه آزمایشی',
            'class' => 'beta',
            'color' => '#5bc0de',
            'icon' => 'dashicons-testimonial'
        ),
        'pro' => array(
            'text' => '',
            'class' => 'pro',
            'color' => '#ff6b6b',
            'icon' => 'dashicons-star-filled',
            'tooltip' => 'این ویژگی فقط در نسخه Pro در دسترس است.'
        ),
        'freemium' => array(
            'text' => '',
            'class' => 'freemium',
            'color' => '#f4b400',
            'icon' => 'dashicons-admin-generic',
            'tooltip' => 'این ویژگی شامل تنظیمات اختصاصی نسخه Pro است.'
        )
    );

    /**
     * Get status badge HTML
     * 
     * @param string $status Status key
     * @return string HTML badge
     */
    public static function get_status_badge($status) {
        if (!isset(self::$statuses[$status])) {
            return '';
        }

        $status_data = self::$statuses[$status];
        $icon = '';

        // تعیین آیکون بر اساس وضعیت
        switch ($status) {
            case 'coming_soon':
                $icon = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M15.24 2H8.76004C5.00004 2 4.71004 5.38 6.74004 7.22L17.26 16.78C19.29 18.62 19 22 15.24 22H8.76004C5.00004 22 4.71004 18.62 6.74004 16.78L17.26 7.22C19.29 5.38 19 2 15.24 2Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                </svg>';
                break;
            case 'deprecated':
                $icon = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M9.16998 14.83L14.83 9.17004" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                    <path d="M14.83 14.83L9.16998 9.17004" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                    <path d="M9 22H15C20 22 22 20 22 15V9C22 4 20 2 15 2H9C4 2 2 4 2 9V15C2 20 4 22 9 22Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                </svg>';
                break;
            case 'beta':
                $icon = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M12 9V14" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                    <path d="M12.0001 21.41H5.94005C2.47005 21.41 1.02005 18.93 2.70005 15.9L5.82006 10.28L8.76006 5.00003C10.5401 1.79003 13.4601 1.79003 15.2401 5.00003L18.1801 10.29L21.3001 15.91C22.9801 18.94 21.5201 21.42 18.0601 21.42H12.0001V21.41Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                    <path d="M11.9945 17H12.0035" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                </svg>';
                break;
            case 'pro':
                $icon = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M16.7 18.98H7.30002C6.88002 18.98 6.41002 18.65 6.27002 18.25L2.13002 6.66999C1.54002 5.00999 2.23002 4.49999 3.65002 5.51999L7.55002 8.30999C8.20002 8.75999 8.94002 8.52999 9.22002 7.79999L10.98 3.10999C11.54 1.60999 12.47 1.60999 13.03 3.10999L14.79 7.79999C15.07 8.52999 15.81 8.75999 16.45 8.30999L20.11 5.69999C21.67 4.57999 22.42 5.14999 21.78 6.95999L17.74 18.27C17.59 18.65 17.12 18.98 16.7 18.98Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                    <path d="M6.5 22H17.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                    <path d="M9.5 14H14.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                </svg>';
                break;
            case 'freemium':
                $icon = '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M19 9C19 10.45 18.57 11.78 17.83 12.89C16.75 14.49 15.04 15.62 13.05 15.91C12.71 15.97 12.36 16 12 16C11.64 16 11.29 15.97 10.95 15.91C8.96 15.62 7.25 14.49 6.17 12.89C5.43 11.78 5 10.45 5 9C5 5.13 8.13 2 12 2C15.87 2 19 5.13 19 9Z" stroke="#B26A00" stroke-opacity="0.827451" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M21.25 18.47L19.6 18.86C19.23 18.95 18.94 19.23 18.86 19.6L18.51 21.07C18.32 21.87 17.3 22.11 16.77 21.48L12 16L7.22996 21.49C6.69996 22.12 5.67996 21.88 5.48996 21.08L5.13996 19.61C5.04996 19.24 4.75996 18.95 4.39996 18.87L2.74996 18.48C1.98996 18.3 1.71996 17.35 2.26996 16.8L6.16996 12.9C7.24996 14.5 8.95996 15.63 10.95 15.92C11.29 15.98 11.64 16.01 12 16.01C12.36 16.01 12.71 15.98 13.05 15.92C15.04 15.63 16.75 14.5 17.83 12.9L21.73 16.8C22.28 17.34 22.01 18.29 21.25 18.47Z" stroke="#B26A00" stroke-opacity="0.827451" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M12.58 5.98L13.17 7.15999C13.25 7.31999 13.46 7.48 13.65 7.51L14.72 7.68999C15.4 7.79999 15.56 8.3 15.07 8.79L14.24 9.61998C14.1 9.75998 14.02 10.03 14.07 10.23L14.31 11.26C14.5 12.07 14.07 12.39 13.35 11.96L12.35 11.37C12.17 11.26 11.87 11.26 11.69 11.37L10.69 11.96C9.96997 12.38 9.53997 12.07 9.72997 11.26L9.96997 10.23C10.01 10.04 9.93997 9.75998 9.79997 9.61998L8.96997 8.79C8.47997 8.3 8.63997 7.80999 9.31997 7.68999L10.39 7.51C10.57 7.48 10.78 7.31999 10.86 7.15999L11.45 5.98C11.74 5.34 12.26 5.34 12.58 5.98Z" stroke="#B26A00" stroke-opacity="0.827451" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
</svg>
';
                break;
        }

        // استفاده از tooltip سفارشی اگر موجود باشد، در غیر این صورت از text استفاده کن
        $tooltip_text = isset($status_data['tooltip']) ? $status_data['tooltip'] : $status_data['text'];
        
        $label = trim(esc_html($status_data['text']));
        $text_html = $label !== '' ? ' ' . $label : '';

        return sprintf(
            '<span class="bebanstore-badge bebanstore-status-%s" data-tooltip="%s">%s%s</span>',
            esc_attr($status),
            esc_attr($tooltip_text),
            $icon,
            $text_html
        );
    }

    /**
     * اعمال وضعیت به فیلد
     */
    public static function apply_status_to_field($field, $status) {
        if (!isset(self::$statuses[$status])) {
            return $field;
        }

        $config = self::$statuses[$status];
        
        // اضافه کردن کلاس‌های وضعیت به wrapper
        $field['wrapper_class'] = isset($field['wrapper_class']) ? $field['wrapper_class'] . ' ' : '';
        $field['wrapper_class'] .= 'bebanstore-status-' . $config['class'];

        // غیرفعال کردن فیلد اگر وضعیت coming_soon یا deprecated باشد
        if (in_array($status, array('coming_soon', 'deprecated'))) {
            $field['disabled'] = true;
        }

        // اضافه کردن برچسب وضعیت به برچسب فیلد
        if (isset($field['label'])) {
            $field['label'] = $field['label'] . ' ' . self::get_status_badge($status);
        }

        return $field;
    }

    /**
     * اعمال وضعیت به ویژگی
     */
    public static function apply_status_to_feature($feature_id, $status) {
        if (!isset(self::$statuses[$status])) {
            return;
        }

        $config = self::$statuses[$status];
        
        // اضافه کردن کلاس وضعیت به ردیف ویژگی
        add_filter('bebanstore_feature_row_class', function($classes, $id) use ($feature_id, $config) {
            if ($id === $feature_id) {
                $classes[] = 'bebanstore-status-' . $config['class'];
            }
            return $classes;
        }, 10, 2);

        // اضافه کردن برچسب وضعیت به عنوان ویژگی
        add_filter('bebanstore_feature_status', function($status_badge, $id) use ($feature_id, $status) {
            if ($id === $feature_id) {
                return self::get_status_badge($status);
            }
            return $status_badge;
        }, 10, 2);
    }

    /**
     * دریافت وضعیت ویژگی
     * 
     * @param string $feature_id شناسه ویژگی
     * @return string وضعیت ویژگی
     */
    public static function get_feature_status($feature_id) {
        $factory = new BebanStore_Feature_Settings_Factory();
        $feature_settings = $factory->get_settings($feature_id);
        return isset($feature_settings['status']) ? $feature_settings['status'] : 'active';
    }
    
    /**
     * بررسی اینکه آیا چک‌باکس ویژگی باید فعال باشد
     * 
     * @param string $feature_id شناسه ویژگی
     * @return bool
     */
    public static function should_enable_checkbox($feature_id) {
        $status = self::get_feature_status($feature_id);
        
        // Check if Pro plugin is active for pro features
        if ($status === 'pro') {
            return apply_filters('bebanstore_pro_available', false);
        }
        
        // Pro features should not have enabled checkbox in free version
        return in_array($status, array('active', 'beta'));
    }
    
    /**
     * بررسی اینکه آیا ویژگی باید فعال باشد
     * 
     * @param string $feature_id شناسه ویژگی
     * @return bool
     */
    public static function should_enable_feature($feature_id) {
        $status = self::get_feature_status($feature_id);
        // Pro features should be enabled for display but not for functionality
        return in_array($status, array('active', 'beta', 'pro'));
    }
    
    /**
     * بررسی اینکه آیا دکمه تنظیمات باید نمایش داده شود
     * 
     * @param string $feature_id شناسه ویژگی
     * @return bool
     */
    public static function should_show_settings_button($feature_id) {
        $status = self::get_feature_status($feature_id);
        
        // Check if Pro plugin is active for pro features
        if ($status === 'pro') {
            return apply_filters('bebanstore_pro_available', false);
        }
        
        // Pro features should not show settings button in free version
        return in_array($status, array('active', 'beta'));
    }
    
    /**
     * دریافت پیام وضعیت
     * 
     * @param string $status وضعیت
     * @return string پیام
     */
    public static function get_status_message($status) {
        $messages = array(
            'coming_soon' => 'این ویژگی در آپدیت های بعدی فعال می‌شود',
            'pro' => 'این ویژگی فقط در نسخه Pro در دسترس است',
            'deprecated' => 'این ویژگی بدلایلی منسوخ شده است'
        );
        
        return isset($messages[$status]) ? $messages[$status] : '';
    }
    
    /**
     * دریافت badge PRO برای ویژگی‌های فعال پرو
     * 
     * @param bool $is_active آیا ویژگی فعال است؟
     * @return string HTML badge
     */
    public static function get_active_pro_badge($is_active = true) {
        $icon = '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
            <path d="M16.7 18.98H7.30002C6.88002 18.98 6.41002 18.65 6.27002 18.25L2.13002 6.66999C1.54002 5.00999 2.23002 4.49999 3.65002 5.51999L7.55002 8.30999C8.20002 8.75999 8.94002 8.52999 9.22002 7.79999L10.98 3.10999C11.54 1.60999 12.47 1.60999 13.03 3.10999L14.79 7.79999C15.07 8.52999 15.81 8.75999 16.45 8.30999L20.11 5.69999C21.67 4.57999 22.42 5.14999 21.78 6.95999L17.74 18.27C17.59 18.65 17.12 18.98 16.7 18.98Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
            <path d="M6.5 22H17.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
            <path d="M9.5 14H14.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
        </svg>';
        
        // تغییر tooltip بر اساس وضعیت فعال بودن
        $tooltip = $is_active 
            ? 'این ویژگی از نسخه Pro است' 
            : 'این ویژگی فقط در نسخه Pro در دسترس است';
        
        return sprintf(
            '<span class="bebanstore-badge bebanstore-status-pro" data-tooltip="%s">%s</span>',
            esc_attr($tooltip),
            $icon
        );
    }

    /**
     * دریافت badge ویژه تنظیمات Pro در ویژگی‌های رایگان
     *
     * @return string
     */
    public static function get_freemium_badge() {
        return self::get_status_badge('freemium');
    }
} 