<?php

require_once plugin_dir_path(__FILE__) . 'class-feature-status.php';

/**
 * BebanStore Settings Register Class
 * 
 * Handles registration of all settings, sections, and fields.
 * 
 * @package BebanStore
 * @since 1.0.0
 */
class BebanStore_Settings_Register {
    /**
     * Plugin options
     * 
     * @var array
     * @access private
     */
    private $options;

    /**
     * Features container
     * 
     * @var array
     * @access private
     */
    private $features;

    /**
     * Feature settings factory
     * 
     * @var BebanStore_Feature_Settings_Factory
     * @access private
     */
    private $feature_settings;

    /**
     * Initialize the class
     * 
     * @param array $options Plugin options
     * @param array $features Plugin features
     */
    public function __construct($options, $features) {
        $this->options = $options;
        $this->features = $features;
        $this->feature_settings = new BebanStore_Feature_Settings_Factory();
    }

    /**
     * Register all settings
     */
    public function register_settings() {
        // Hook before registering settings
        do_action('bebanstore_before_register_settings');

        // Register main settings group
        register_setting(
            'bebanstore_options',
            'bebanstore_options',
            array(
                'type' => 'array',
                'sanitize_callback' => array($this, 'sanitize_options'),
                'default' => array(
                    'ajax_add_to_cart' => false,
                    'cart_confirmation' => false,
                    'custom_quantity_control' => false,
                    'sales_progress' => false,
                    'variable_price' => false,
                    'countdown' => false,
                    'express_shipping' => false,
                    'tracking_code' => false,
                    'product_attributes' => false,
                    'discount_badges' => false,
                    'product_discount' => false,
                    'order_count' => false,
                    'order_view' => false,
                    'checkout_fields_manager' => false,
                    'your_savings' => false,
                    'wishlist' => false,
                    'custom_font' => false,
                    'add_to_cart_button_text' => false,
                    'toast_customization' => false,
                    'product_mentions_in_content' => false,
                    'elementor_widgets_enabled' => array(),
                    'feature_settings' => $this->feature_settings->get_default_settings()['feature_settings']
                )
            )
        );

        // Hook after registering settings
        do_action('bebanstore_after_register_settings');
    }


    /**
     * Sanitize options
     * 
     * @param array $input Input options
     * @return array Sanitized options
     */
    public function sanitize_options($input) {
        // Get existing options
        $existing_options = get_option('bebanstore_options', array());
        
        // Ensure input is an array
        if (!is_array($input)) {
            return $existing_options;
        }

        // Initialize sanitized options with existing options
        $sanitized = $existing_options;

        // Process checkbox fields
        $checkbox_fields = array(
            'ajax_add_to_cart',
            'cart_confirmation',
            'custom_quantity_control',
            'sales_progress',
            'variable_price',
            'countdown',
            'express_shipping',
            'tracking_code',
            'product_attributes',
            'product_discount',
            'discount_badges',
            'order_count',
            'order_view',
            'checkout_fields_manager',
            'your_savings',
            'wishlist',
            'custom_font',
            'add_to_cart_button_text',
            'toast_customization',
            'product_mentions_in_content'
        );

        // Allow Pro plugin to add checkbox fields
        $checkbox_fields = apply_filters('bebanstore_pro_checkbox_fields', $checkbox_fields);

        foreach ($checkbox_fields as $field) {
            $sanitized[$field] = isset($input[$field]) ? (bool) $input[$field] : false;
        }

        // Process elementor widgets enabled checkboxes
        // Get all features that have elementor widget
        $all_settings = $this->feature_settings->get_all_settings();
        $elementor_widgets_enabled = isset($existing_options['elementor_widgets_enabled']) 
            ? $existing_options['elementor_widgets_enabled'] 
            : array();
        
        // Check each feature that has elementor widget
        foreach ($all_settings as $feature_id => $feature_settings) {
            // Check if feature has elementor widget
            if (isset($feature_settings['elementor_widget']) && !empty($feature_settings['elementor_widget'])) {
                // If in input, use input value (true if checkbox was checked, false otherwise)
                // If not in input, it means checkbox was unchecked, so set to false
                if (isset($input['elementor_widgets_enabled'][$feature_id])) {
                    $elementor_widgets_enabled[$feature_id] = (bool) $input['elementor_widgets_enabled'][$feature_id];
                } else {
                    // Checkbox was not in input, meaning it was unchecked
                    $elementor_widgets_enabled[$feature_id] = false;
                }
            }
        }
        
        $sanitized['elementor_widgets_enabled'] = $elementor_widgets_enabled;

        // Process feature settings
        if (isset($input['feature_settings']) && is_array($input['feature_settings'])) {
            $feature_settings = $this->get_feature_settings();
            
            foreach ($input['feature_settings'] as $feature_id => $settings) {
                if (!isset($feature_settings[$feature_id]) || !$feature_settings[$feature_id]['has_settings']) {
                    continue;
                }

                // Initialize feature settings if not exists
                if (!isset($sanitized['feature_settings'][$feature_id])) {
                    $sanitized['feature_settings'][$feature_id] = array();
                }

                foreach ($feature_settings[$feature_id]['settings'] as $setting_id => $setting) {
                    // For checkboxes, if not set in input, it means unchecked
                    if ($setting['type'] === 'checkbox') {
                        $checkbox_value = isset($settings[$setting_id]) ? (bool) $settings[$setting_id] : false;
                        $sanitized['feature_settings'][$feature_id][$setting_id] = $checkbox_value;
                        continue;
                    }

                    // For other field types, only process if value exists in input
                    if (isset($settings[$setting_id])) {
                        switch ($setting['type']) {
                            case 'text':
                                $sanitized['feature_settings'][$feature_id][$setting_id] = sanitize_text_field($settings[$setting_id]);
                                break;

                            case 'textarea':
                                $sanitized['feature_settings'][$feature_id][$setting_id] = sanitize_textarea_field($settings[$setting_id]);
                                break;

                            case 'number':
                                $sanitized['feature_settings'][$feature_id][$setting_id] = absint($settings[$setting_id]);
                                if (isset($setting['min'])) {
                                    $sanitized['feature_settings'][$feature_id][$setting_id] = max($setting['min'], $sanitized['feature_settings'][$feature_id][$setting_id]);
                                }
                                if (isset($setting['max'])) {
                                    $sanitized['feature_settings'][$feature_id][$setting_id] = min($setting['max'], $sanitized['feature_settings'][$feature_id][$setting_id]);
                                }
                                break;

                            case 'select':
                                if (isset($setting['options'][$settings[$setting_id]])) {
                                    $sanitized['feature_settings'][$feature_id][$setting_id] = sanitize_text_field($settings[$setting_id]);
                                } else {
                                    $sanitized['feature_settings'][$feature_id][$setting_id] = $setting['default'];
                                }
                                break;

                            case 'color':
                                // Support both hex colors (#ffffff) and rgba colors (rgba(255, 255, 255, 1))
                                $color_value = sanitize_text_field($settings[$setting_id]);
                                // If it's a hex color, validate it
                                if (strpos($color_value, '#') === 0) {
                                    $sanitized_color = sanitize_hex_color($color_value);
                                    if ($sanitized_color) {
                                        $color_value = $sanitized_color;
                                    }
                                }
                                $sanitized['feature_settings'][$feature_id][$setting_id] = $color_value;
                                break;

                            case 'media_uploader':
                                $sanitized['feature_settings'][$feature_id][$setting_id] = esc_url_raw($settings[$setting_id]);
                                break;

                            case 'date':
                            case 'time':
                                $sanitized['feature_settings'][$feature_id][$setting_id] = sanitize_text_field($settings[$setting_id]);
                                break;
                        }
                    }
                }
            }
        }

        // Sanitize custom CSS
        if (isset($input['custom_css'])) {
            $sanitized['custom_css'] = wp_strip_all_tags($input['custom_css']);
        }
        
        // Sanitize CSS location
        if (isset($input['css_location'])) {
            $allowed_locations = array('inline_head', 'inline_body', 'file_head', 'file_body');
            $sanitized['css_location'] = in_array($input['css_location'], $allowed_locations) ? $input['css_location'] : 'inline_head';
        }
        
        return $sanitized;
    }


    /**
     * Get feature settings
     * 
     * @return array
     */
    public function get_feature_settings() {
        $factory = new BebanStore_Feature_Settings_Factory();
        return $factory->get_all_settings();
    }
} 