<?php

require_once plugin_dir_path(__FILE__) . 'class-feature-status.php';

/**
 * BebanStore Settings Render Class
 * 
 * Handles rendering of the settings page and its components.
 * 
 * @package BebanStore
 * @since 1.0.0
 */
class BebanStore_Settings_Render {
    /**
     * Plugin options
     * 
     * @var array
     * @access private
     */
    private $options;

    /**
     * Features container
     * 
     * @var array
     * @access private
     */
    private $features;

    /**
     * Active tab
     * 
     * @var string
     * @access private
     */
    private $active_tab;

    /**
     * Track if a section accordion is currently open
     * 
     * @var bool
     * @access private
     */
    private $section_open = false;

    /**
     * Initialize the class
     * 
     * @param array $options Plugin options
     * @param array $features Plugin features
     */
    public function __construct($options, $features) {
        $this->options = $options;
        $this->features = $features;
        $this->active_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'general';
    }

    /**
     * Render the settings page
     */
    public function render_settings_page() {
        // Hook before rendering settings page
        do_action('bebanstore_before_render_settings_page');
        
        // Make features available to the view
        $features = $this->features;
        
        // Include the settings page template
        include BEBANSTORE_PLUGIN_DIR . 'admin/views/settings-page.php';
        
        // Hook after rendering settings page
        do_action('bebanstore_after_render_settings_page');
    }

    /**
     * Render tabs
     */
    private function render_tabs() {
        $tabs = $this->get_tabs();
        if (empty($tabs)) {
            return;
        }
        ?>
        <nav class="nav-tab-wrapper bebanstore-tabs">
            <!-- Plugin Info Tab -->
            <div class="bebanstore-plugin-info">
                <div class="bebanstore-plugin-icon">
                <svg width="376" height="512" viewBox="0 0 376 512" fill="none" xmlns="http://www.w3.org/2000/svg">
<g clip-path="url(#clip0_34_29)">
<g clip-path="url(#clip1_34_29)">
<path fill-rule="evenodd" clip-rule="evenodd" d="M0.864746 137.667H111.295V324.508C111.295 339.655 115.794 354.462 124.222 367.056C132.651 379.651 144.63 389.467 158.646 395.263C172.662 401.06 188.085 402.576 202.964 399.621C217.844 396.666 231.511 389.372 242.239 378.662C252.966 367.951 260.271 354.305 263.231 339.449C266.191 324.593 264.672 309.195 258.866 295.201C253.061 281.207 243.229 269.246 230.615 260.831C218.001 252.416 203.171 247.924 188 247.924H138.903V137.667H188C225.012 137.667 261.193 148.625 291.967 169.156C322.741 189.686 346.727 218.867 360.891 253.007C375.054 287.148 378.76 324.716 371.54 360.959C364.319 397.203 346.496 430.495 320.325 456.625C294.153 482.755 260.809 500.55 224.508 507.759C188.208 514.969 150.581 511.269 116.386 497.127C82.192 482.986 52.9654 459.038 32.4027 428.312C11.84 397.586 0.864746 361.462 0.864746 324.508L0.864746 137.667Z" fill="#333333"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M111.295 110.103H0.864746V-0.15332H56.0799C70.7114 -0.112685 84.732 5.70852 95.078 16.0383C105.424 26.3681 111.254 40.3666 111.295 54.9751V110.103Z" fill="#8D9BFF"/>
</g>
</g>
<defs>
<clipPath id="clip0_34_29">
<rect width="374.271" height="512.307" fill="white" transform="translate(0.864746 -0.15332)"/>
</clipPath>
<clipPath id="clip1_34_29">
<rect width="374.271" height="511.503" fill="white" transform="translate(0.864746 -0.15332)"/>
</clipPath>
</defs>
</svg>

                </div>
                <div class="bebanstore-plugin-details">
                    <h2 class="bebanstore-plugin-title"><?php _e('ببان استور', 'bebanstore'); ?></h2>
                    <p class="bebanstore-plugin-version"><?php printf(__('نسخه %s', 'bebanstore'), BEBANSTORE_VERSION); ?></p>
                </div>
            </div>

            <?php foreach ($tabs as $tab_id => $tab) : ?>
                <a href="?page=bebanstore-settings&tab=<?php echo esc_attr($tab_id); ?>" 
                   class="nav-tab <?php echo $this->active_tab === $tab_id ? 'nav-tab-active' : ''; ?>">
                    <span class="bebanstore-tab-icon"><?php echo $tab['icon']; ?></span>
                    <div class="beban-tab-detail">
                        <span class="bebanstore-tab-title"><?php echo esc_html($tab['title']); ?></span>
                        <?php if (!empty($tab['description'])) : ?>
                            <span class="bebanstore-tab-description"><?php echo esc_html($tab['description']); ?></span>
                        <?php endif; ?>
                    </div>
                </a>
            <?php endforeach; ?>
        </nav>
        <?php
    }

    /**
     * Get available tabs
     */
    private function get_tabs() {
        return apply_filters('bebanstore_settings_tabs', array(
            'general' => array(
                'title' => __('همگانی', 'bebanstore'),
                'description' => __('تنظیمات عمومی و پایه افزونه', 'bebanstore'),
                'icon' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M19 22V11" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M19 7V2" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M12 22V17" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M12 13V2" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M5 22V11" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M5 7V2" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M3 11H7" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M17 11H21" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M10 13H14" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
</svg>'
            ),
            'product' => array(
                'title' => __('محصولات', 'bebanstore'),
                'description' => __('تنظیمات مربوط به محصولات', 'bebanstore'),
                'icon' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M2 2L3.74001 2C4.82001 2 5.67 2.93 5.58 4L4.75 13.96C4.61 15.59 5.89999 16.99 7.53999 16.99L18.19 16.99C19.63 16.99 20.89 15.81 21 14.38L21.54 6.88C21.66 5.22 20.4 3.87 18.73 3.87L5.82001 3.87" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M16.25 22C16.9404 22 17.5 21.4404 17.5 20.75C17.5 20.0596 16.9404 19.5 16.25 19.5C15.5596 19.5 15 20.0596 15 20.75C15 21.4404 15.5596 22 16.25 22Z" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M8.25 22C8.94036 22 9.5 21.4404 9.5 20.75C9.5 20.0596 8.94036 19.5 8.25 19.5C7.55964 19.5 7 20.0596 7 20.75C7 21.4404 7.55964 22 8.25 22Z" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M9 8L21 8" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
</svg>'
            ),
            'cart' => array(
                'title' => __('سبد خرید', 'bebanstore'),
                'description' => __('مدیریت تنظیمات سبد خرید', 'bebanstore'),
                'icon' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M8.5 14.25C8.5 16.17 10.08 17.75 12 17.75C13.92 17.75 15.5 16.17 15.5 14.25" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M8.81 2L5.19 5.63" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M15.19 2L18.81 5.63" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M2 7.84998C2 5.99998 2.99 5.84998 4.22 5.84998H19.78C21.01 5.84998 22 5.99998 22 7.84998C22 9.99998 21.01 9.84998 19.78 9.84998H4.22C2.99 9.84998 2 9.99998 2 7.84998Z" stroke="#292D32" stroke-width="1.5"/>
<path d="M3.5 10L4.91 18.64C5.23 20.58 6 22 8.86 22H14.89C18 22 18.46 20.64 18.82 18.76L20.5 10" stroke="#292D32" stroke-width="1.5" stroke-linecap="round"/>
</svg>'
            ),
            'checkout' => array(
                'title' => __('تسویه حساب', 'bebanstore'),
                'description' => __('تنظیمات مربوط به صفحه تسویه حساب', 'bebanstore'),
                'icon' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M8.5 19H8C4 19 2 18 2 13V8C2 4 4 2 8 2H16C20 2 22 4 22 8V13C22 17 20 19 16 19H15.5C15.19 19 14.89 19.15 14.7 19.4L13.2 21.4C12.54 22.28 11.46 22.28 10.8 21.4L9.3 19.4C9.14 19.18 8.77 19 8.5 19Z" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M7 8H17" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M7 13H13" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
</svg>'
            ),
            'thank-you' => array(
                'title' => __('صفحه تشکر', 'bebanstore'),
                'description' => __('تنظیمات مربوط به صفحه تشکر', 'bebanstore'),
                'icon' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M23 17.91C23.02 18.66 22.82 19.37 22.46 19.98C22.26 20.34 21.99 20.6701 21.69 20.9401C21 21.5801 20.09 21.9701 19.08 22.0001C17.62 22.0301 16.33 21.2801 15.62 20.1301C15.24 19.5401 15.01 18.8301 15 18.0801C14.97 16.8201 15.53 15.6801 16.43 14.9301C17.11 14.3701 17.97 14.0201 18.91 14.0001C21.12 13.9501 22.95 15.7 23 17.91Z" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M17.44 18.03L18.45 18.99L20.54 16.97" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M3.17004 7.43994L12 12.5499L20.77 7.46991" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M12 21.6099V12.5399" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M21.61 9.17V14.83C21.61 14.88 21.61 14.92 21.6 14.97C20.9 14.36 20 14 19 14C18.06 14 17.19 14.33 16.5 14.88C15.58 15.61 15 16.74 15 18C15 18.75 15.21 19.46 15.58 20.06C15.67 20.22 15.78 20.37 15.9 20.51L14.07 21.52C12.93 22.16 11.07 22.16 9.92999 21.52L4.59 18.56C3.38 17.89 2.39001 16.21 2.39001 14.83V9.17C2.39001 7.79 3.38 6.11002 4.59 5.44002L9.92999 2.48C11.07 1.84 12.93 1.84 14.07 2.48L19.41 5.44002C20.62 6.11002 21.61 7.79 21.61 9.17Z" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
</svg>'
            ),
            'account' => array(
                'title' => __('حساب کاربری', 'bebanstore'),
                'description' => __('مدیریت تنظیمات حساب کاربری', 'bebanstore'),
                'icon' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M12 12C14.7614 12 17 9.76142 17 7C17 4.23858 14.7614 2 12 2C9.23858 2 7 4.23858 7 7C7 9.76142 9.23858 12 12 12Z" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M20.5899 22C20.5899 18.13 16.7399 15 11.9999 15C7.25991 15 3.40991 18.13 3.40991 22" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
</svg>'
            ),
            'advanced' => array(
                'title' => __('پیشرفته', 'bebanstore'),
                'description' => __('تنظیمات پیشرفته ', 'bebanstore'),
                'icon' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M20 14C21.1046 14 22 13.1046 22 12C22 10.8954 21.1046 10 20 10C18.8954 10 18 10.8954 18 12C18 13.1046 18.8954 14 20 14Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M20 6C21.1046 6 22 5.10457 22 4C22 2.89543 21.1046 2 20 2C18.8954 2 18 2.89543 18 4C18 5.10457 18.8954 6 20 6Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M20 22C21.1046 22 22 21.1046 22 20C22 18.8954 21.1046 18 20 18C18.8954 18 18 18.8954 18 20C18 21.1046 18.8954 22 20 22Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M4 14C5.10457 14 6 13.1046 6 12C6 10.8954 5.10457 10 4 10C2.89543 10 2 10.8954 2 12C2 13.1046 2.89543 14 4 14Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M6 12H18" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M18 4H14C12 4 11 5 11 7V17C11 19 12 20 14 20H18" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
</svg>'
            ),
            'about' => array(
                'title' => __('اطلاعات', 'bebanstore'),
                'description' => __('اطلاعات و پشتیبانی افزونه', 'bebanstore'),
                'icon' => '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M12 7.75V13" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M21.08 8.58003V15.42C21.08 16.54 20.48 17.58 19.51 18.15L13.57 21.58C12.6 22.14 11.4 22.14 10.42 21.58L4.47998 18.15C3.50998 17.59 2.90997 16.55 2.90997 15.42V8.58003C2.90997 7.46003 3.50998 6.41999 4.47998 5.84999L10.42 2.42C11.39 1.86 12.59 1.86 13.57 2.42L19.51 5.84999C20.48 6.41999 21.08 7.45003 21.08 8.58003Z" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
<path d="M12 16.2V16.2999" stroke="#292D32" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
</svg>'
            )
        ));
    }

    /**
     * Render checkbox field
     */
    public function render_checkbox_field($field_id, $label) {
        $value = isset($this->options[$field_id]) ? $this->options[$field_id] : false;
        ?>
            <div class="bebanstore-checkbox-wrapper">
                <label class="bebanstore-checkbox">
                    <input class="bebanstore-checkbox__trigger visuallyhidden" 
                           type="checkbox" 
                           name="bebanstore_options[<?php echo esc_attr($field_id); ?>]" 
                           value="1" 
                           <?php checked($value, true); ?>>
                    <span class="bebanstore-checkbox__symbol">
                        <svg aria-hidden="true" class="bebanstore-checkbox__icon" width="28px" height="28px" viewBox="0 0 28 28" version="1" xmlns="http://www.w3.org/2000/svg">
                            <path d="M4 14l8 7L24 7"></path>
                        </svg>
                    </span>
                </label>
            </div>
        <?php
    }

    /**
     * Render field based on its type
     * 
     * @param string $field_name Full field name
     * @param array $field Field configuration
     * @param array $value Current value
     */
    public function render_field($field_name, $field, $value = '') {
        $type = isset($field['type']) ? $field['type'] : 'text';
        $label = isset($field['label']) ? $field['label'] : '';
        $description = isset($field['description']) ? $field['description'] : '';
        $placeholder = isset($field['placeholder']) ? $field['placeholder'] : '';
        $options = isset($field['options']) ? $field['options'] : array();
        $min = isset($field['min']) ? $field['min'] : '';
        $max = isset($field['max']) ? $field['max'] : '';
        $step = isset($field['step']) ? $field['step'] : '';
        $default = isset($field['default']) ? $field['default'] : '';

        if (empty($value) && $default !== '') {
            $value = $default;
        }

        // اعمال وضعیت به فیلد
        if (isset($field['status'])) {
            $field = BebanStore_Feature_Status::apply_status_to_field($field, $field['status']);
        }

        // ===== Field Types جدید برای Settings Grouping System =====
        
        // 1. Section Header (با قابلیت Accordion)
        if ($type === 'section_header') {
            // اگر section قبلی باز بود، بستنش
            if ($this->section_open) {
                echo '</div></div>'; // بستن content و wrapper section قبلی
            }
            
            $section_id = 'section_' . sanitize_key($field_name . '_' . (isset($field['title']) ? $field['title'] : ''));
            // پیش‌فرض: همه بسته هستند مگر اینکه 'open' => true تنظیم شده باشد
            $open = isset($field['open']) && $field['open'];
            $active_class = $open ? ' active' : '';
            
            echo '<div class="bebanstore-section-accordion' . $active_class . '">';
            echo '<div class="bebanstore-section-accordion-header" data-accordion="' . $section_id . '">';
            
            echo '<div class="bebanstore-section-header-content">';
            if (!empty($field['title'])) {
                echo '<h4 class="bebanstore-section-title">' . esc_html($field['title']) . '</h4>';
            }
            if (!empty($field['description'])) {
                echo '<p class="bebanstore-section-description">' . esc_html($field['description']) . '</p>';
            }
            echo '</div>';
            
            echo '<span class="bebanstore-section-accordion-icon"></span>';
            echo '</div>';
            
            // باز کردن wrapper برای محتوا
            echo '<div class="bebanstore-section-accordion-content">';
            
            $this->section_open = true;
            return;
        }
        
        // 2. Column Layout
        if ($type === 'columns') {
            $columns = isset($field['columns']) ? $field['columns'] : 2;
            echo '<div class="bebanstore-columns-wrapper bebanstore-columns-' . $columns . '">';
            
            if (isset($field['fields']) && is_array($field['fields'])) {
                foreach ($field['fields'] as $sub_field_id => $sub_field) {
                    // استخراج feature_id از field_name برای ساخت field_name صحیح
                    preg_match('/\[feature_settings\]\[([^\]]+)\]/', $field_name, $matches);
                    $feature_id = isset($matches[1]) ? $matches[1] : '';
                    
                    // ساخت field_name برای sub field
                    if (!empty($feature_id)) {
                        $sub_field_name = 'bebanstore_options[feature_settings][' . $feature_id . '][' . $sub_field_id . ']';
                    } else {
                        $sub_field_name = $field_name . '[' . $sub_field_id . ']';
                    }
                    
                    // گرفتن value برای sub field
                    $sub_value = '';
                    if (is_array($value) && isset($value[$sub_field_id])) {
                        $sub_value = $value[$sub_field_id];
                    } elseif (isset($sub_field['default'])) {
                        $sub_value = $sub_field['default'];
                    }
                    
                    echo '<div class="bebanstore-column">';
                    $this->render_field($sub_field_name, $sub_field, $sub_value);
                    echo '</div>';
                }
            }
            
            echo '</div>';
            return;
        }
        
        // 3. Accordion Group
        if ($type === 'accordion_group') {
            $accordion_id = 'accordion_' . sanitize_key($field_name);
            $collapsed = isset($field['collapsed']) && $field['collapsed'];
            $active_class = $collapsed ? '' : ' active';
            
            echo '<div class="bebanstore-accordion-group' . $active_class . '">';
            echo '<div class="bebanstore-accordion-header" data-accordion="' . $accordion_id . '">';
            if (!empty($field['title'])) {
                echo '<h4>' . esc_html($field['title']) . '</h4>';
            }
            echo '<span class="bebanstore-accordion-icon"></span>';
            echo '</div>';
            echo '<div class="bebanstore-accordion-content" id="' . $accordion_id . '">';
            
            if (isset($field['fields']) && is_array($field['fields'])) {
                foreach ($field['fields'] as $sub_field_id => $sub_field) {
                    // استخراج feature_id از field_name
                    preg_match('/\[feature_settings\]\[([^\]]+)\]/', $field_name, $matches);
                    $feature_id = isset($matches[1]) ? $matches[1] : '';
                    
                    // ساخت field_name برای sub field
                    if (!empty($feature_id)) {
                        $sub_field_name = 'bebanstore_options[feature_settings][' . $feature_id . '][' . $sub_field_id . ']';
                    } else {
                        $sub_field_name = $field_name . '[' . $sub_field_id . ']';
                    }
                    
                    // گرفتن value برای sub field
                    $sub_value = '';
                    if (is_array($value) && isset($value[$sub_field_id])) {
                        $sub_value = $value[$sub_field_id];
                    } elseif (isset($sub_field['default'])) {
                        $sub_value = $sub_field['default'];
                    }
                    
                    $this->render_field($sub_field_name, $sub_field, $sub_value);
                }
            }
            
            echo '</div>';
            echo '</div>';
            return;
        }

        // Add field wrapper
        echo '<div class="bebanstore-field">';
        echo '<div class="bebanstore-field-content">';
        echo '<div class="bebanstore-field-layout">';
        
        // Add right side (label + input)
        echo '<div class="bebanstore-field-right">';
        
        // Add label if exists
        if (!empty($label)) {
            echo '<div class="bebanstore-field-label">';
            echo '<span>' . wp_kses_post($label) . '</span>';
            echo '</div>';
        }
        
        // Add input wrapper
        echo '<div class="bebanstore-field-input">';

        // اگر فیلد غیرفعال است، فقط نمایش بده و مقدار رو در فیلد مخفی ذخیره کن
        if (isset($field['disabled']) && $field['disabled']) {
            // نمایش وضعیت فعلی
            switch ($type) {
                case 'checkbox':
                    echo '<div class="bebanstore-disabled-field">';
                    if (isset($field['status']) && $field['status'] === 'coming_soon') {
                        echo '<span class="bebanstore-disabled-value">' . __('این ویژگی در آپدیت های بعدی فعال می‌شود', 'bebanstore') . '</span>';
                    } else {
                        echo '<span class="bebanstore-disabled-value">' . ($value ? __('فعال', 'bebanstore') : __('غیرفعال', 'bebanstore')) . '</span>';
                    }
                    echo '</div>';
                    break;
                    
                case 'select':
                    $selected_label = isset($options[$value]) ? $options[$value] : $value;
                    echo '<div class="bebanstore-disabled-field">';
                    if (isset($field['status']) && $field['status'] === 'coming_soon') {
                        echo '<span class="bebanstore-disabled-value">' . __('این ویژگی در آپدیت های بعدی فعال می‌شود', 'bebanstore') . '</span>';
                    } else {
                        echo '<span class="bebanstore-disabled-value">' . esc_html($selected_label) . '</span>';
                    }
                    echo '</div>';
                    break;
                    
                default:
                    echo '<div class="bebanstore-disabled-field">';
                    if (isset($field['status']) && $field['status'] === 'coming_soon') {
                        echo '<span class="bebanstore-disabled-value">' . __('این ویژگی در آپدیت های بعدی فعال می‌شود', 'bebanstore') . '</span>';
                    } else {
                        echo '<span class="bebanstore-disabled-value">' . esc_html($value) . '</span>';
                    }
                    echo '</div>';
            }
            
            // فیلد مخفی برای حفظ مقدار
            printf(
                '<input type="hidden" name="%s" value="%s">',
                esc_attr($field_name),
                esc_attr($value)
            );
        } else {
            // Render field based on type
            switch ($type) {
                case 'text':
                    printf(
                        '<input type="text" name="%s" value="%s" class="regular-text" placeholder="%s" data-default-value="%s">',
                        esc_attr($field_name),
                        esc_attr($value),
                        esc_attr($placeholder),
                        esc_attr($default)
                    );
                    break;

                case 'textarea':
                    printf(
                        '<textarea name="%s" class="large-text" rows="5" placeholder="%s" data-default-value="%s">%s</textarea>',
                        esc_attr($field_name),
                        esc_attr($placeholder),
                        esc_attr($default),
                        esc_textarea($value)
                    );
                    break;

                case 'select':
                    echo '<select name="' . esc_attr($field_name) . '" data-default-value="' . esc_attr($default) . '">';
                    foreach ($options as $option_value => $option_label) {
                        printf(
                            '<option value="%s" %s>%s</option>',
                            esc_attr($option_value),
                            selected($value, $option_value, false),
                            esc_html($option_label)
                        );
                    }
                    echo '</select>';
                    break;

                case 'checkbox':
                    ?>
                    <div class="bebanstore-checkbox-wrapper">
                        <label class="bebanstore-checkbox">
                            <input class="bebanstore-checkbox__trigger visuallyhidden" 
                                   type="checkbox" 
                                   name="<?php echo esc_attr($field_name); ?>" 
                                   value="1" 
                                   <?php checked($value, true); ?>
                                   data-default-value="<?php echo esc_attr($default ? 'true' : 'false'); ?>">
                            <span class="bebanstore-checkbox__symbol">
                                <svg aria-hidden="true" class="bebanstore-checkbox__icon" width="28px" height="28px" viewBox="0 0 28 28" version="1" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M4 14l8 7L24 7"></path>
                                </svg>
                            </span>
                        </label>
                    </div>
                    <?php
                    break;

                case 'number':
                    printf(
                        '<input type="number" name="%s" value="%s" min="%s" max="%s" step="%s" data-default-value="%s">',
                        esc_attr($field_name),
                        esc_attr($value),
                        esc_attr($min),
                        esc_attr($max),
                        esc_attr($step),
                        esc_attr($default)
                    );
                    break;

                case 'color':
                    printf(
                        '<input type="color" name="%s" value="%s" data-default-value="%s">',
                        esc_attr($field_name),
                        esc_attr($value),
                        esc_attr($default)
                    );
                    break;

                case 'media_uploader':
                    $button_text = isset($field['button_text']) ? $field['button_text'] : __('انتخاب تصویر', 'bebanstore');
                    $remove_text = isset($field['remove_text']) ? $field['remove_text'] : __('حذف تصویر', 'bebanstore');
                    ?>
                    <div class="bebanstore-media-uploader">
                        <input type="hidden" 
                               name="<?php echo esc_attr($field_name); ?>" 
                               value="<?php echo esc_attr($value); ?>" 
                               class="bebanstore-media-input">
                        <div class="bebanstore-media-preview">
                            <?php if (!empty($value)): ?>
                                <img src="<?php echo esc_url($value); ?>" alt="">
                            <?php endif; ?>
                        </div>
                        <button type="button" class="button bebanstore-media-upload" data-field="<?php echo esc_attr($field_name); ?>">
                            <?php echo esc_html($button_text); ?>
                        </button>
                        <button type="button" class="button bebanstore-media-remove" data-field="<?php echo esc_attr($field_name); ?>" <?php echo empty($value) ? 'style="display:none;"' : ''; ?>>
                            <?php echo esc_html($remove_text); ?>
                        </button>
                    </div>
                    <?php
                    break;

                default:
                    printf(
                        '<input type="text" name="%s" value="%s" class="regular-text">',
                        esc_attr($field_name),
                        esc_attr($value)
                    );
            }
        }

        echo '</div>'; // End field input
        
        // Add description if exists
        if (!empty($description)) {
            echo '<p class="bebanstore-field-description">' . esc_html($description) . '</p>';
        }
        
        echo '</div>'; // End field right
        
        // Add tooltips if exists
        if (isset($field['tooltips']) && is_array($field['tooltips']) && !empty($field['tooltips'])) {
            echo '<div class="bebanstore-tooltips-container">';
            foreach ($field['tooltips'] as $tooltip) {
                $tooltip_type = isset($tooltip['type']) ? $tooltip['type'] : 'info';
                $tooltip_text = isset($tooltip['text']) ? $tooltip['text'] : '';
                $tooltip_class = 'tooltip-' . $tooltip_type;
                
                if (!empty($tooltip_text)) {
                    echo '<span class="bebanstore-tooltip-icon ' . esc_attr($tooltip_class) . '" data-tooltip="' . esc_attr($tooltip_text) . '">';
                    
                    if ($tooltip_type === 'warning') {
                        // Warning icon
                        echo '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">';
                        echo '<path d="M12 9V14" stroke="#F59E0B" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>';
                        echo '<path d="M12.0001 21.41H5.94005C2.47005 21.41 1.02005 18.93 2.70005 15.9L5.82006 10.28L8.76006 5.00003C10.5401 1.79003 13.4601 1.79003 15.2401 5.00003L18.1801 10.29L21.3001 15.91C22.9801 18.94 21.5201 21.42 18.0601 21.42H12.0001V21.41Z" stroke="#F59E0B" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>';
                        echo '<path d="M11.9945 17H12.0035" stroke="#F59E0B" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';
                        echo '</svg>';
                    } else {
                        // Info icon (default)
                        echo '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">';
                        echo '<path d="M12 22C17.5 22 22 17.5 22 12C22 6.5 17.5 2 12 2C6.5 2 2 6.5 2 12C2 17.5 6.5 22 12 22Z" stroke="#333333" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>';
                        echo '<path d="M12 8V13" stroke="#333333" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>';
                        echo '<path d="M11.9945 16H12.0035" stroke="#333333" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';
                        echo '</svg>';
                    }
                    
                    echo '</span>';
                }
            }
            echo '</div>';
        } elseif (isset($field['tooltip']) && !empty($field['tooltip'])) {
            // Backward compatibility for single tooltip
            $tooltip_type = isset($field['tooltip_type']) ? $field['tooltip_type'] : 'info';
            $tooltip_class = 'tooltip-' . $tooltip_type;
            
            echo '<div class="bebanstore-tooltips-container">';
            echo '<span class="bebanstore-tooltip-icon ' . esc_attr($tooltip_class) . '" data-tooltip="' . esc_attr($field['tooltip']) . '">';
            
            if ($tooltip_type === 'warning') {
                // Warning icon
                echo '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">';
                echo '<path d="M12 9V14" stroke="#F59E0B" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>';
                echo '<path d="M12.0001 21.41H5.94005C2.47005 21.41 1.02005 18.93 2.70005 15.9L5.82006 10.28L8.76006 5.00003C10.5401 1.79003 13.4601 1.79003 15.2401 5.00003L18.1801 10.29L21.3001 15.91C22.9801 18.94 21.5201 21.42 18.0601 21.42H12.0001V21.41Z" stroke="#F59E0B" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>';
                echo '<path d="M11.9945 17H12.0035" stroke="#F59E0B" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';
                echo '</svg>';
            } else {
                // Info icon (default)
                echo '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">';
                echo '<path d="M12 22C17.5 22 22 17.5 22 12C22 6.5 17.5 2 12 2C6.5 2 2 6.5 2 12C2 17.5 6.5 22 12 22Z" stroke="#2563EB" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>';
                echo '<path d="M12 8V13" stroke="#2563EB" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>';
                echo '<path d="M11.9945 16H12.0035" stroke="#2563EB" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';
                echo '</svg>';
            }
            
            echo '</span>';
            echo '</div>';
        }
        
        echo '</div>'; // End field layout
        echo '</div>'; // End field content
        echo '</div>'; // End field wrapper
    }

    /**
     * Close any open section accordion
     * 
     * @return void
     */
    private function close_open_section() {
        if ($this->section_open) {
            echo '</div></div>'; // بستن content و wrapper section
            $this->section_open = false;
        }
    }

    /**
     * Check if field should be displayed based on conditions
     * 
     * @param array $field Field configuration
     * @param array $all_values All field values for the feature
     * @return bool Whether the field should be displayed
     */
    private function should_display_field($field, $all_values) {
        if (!isset($field['condition'])) {
            return true;
        }
        
        $condition = $field['condition'];
        
        if (!isset($condition['field']) || !isset($condition['value'])) {
            return true;
        }
        
        $condition_field = $condition['field'];
        $condition_value = $condition['value'];
        
        // Check if condition field value matches
        if (!is_array($all_values) || !isset($all_values[$condition_field])) {
            return false;
        }
        
        return $all_values[$condition_field] == $condition_value;
    }

    /**
     * Render reset button
     *
     * @param string $feature_id Feature ID
     * @param bool $has_settings Whether the feature has settings
     * @return void
     */
    public function render_reset_button($feature_id, $has_settings) {
        if (!$has_settings) {
            return;
        }
        ?>
        <div class="bebanstore-reset-button-wrapper">
            <button type="button" class="bebanstore-reset-settings button" data-feature="<?php echo esc_attr($feature_id); ?>">
                <?php _e('بازنشانی تنظیمات', 'bebanstore'); ?>
            </button>
        </div>
        <?php
    }

    /**
     * Render additional settings button
     *
     * @param string $feature_id Feature ID
     * @param array $feature_settings Feature settings
     * @param array $options Current options
     * @return void
     */
    public function render_additional_settings_button($feature_id, $feature_settings, $options) {
        // دکمه فقط زمانی غیرفعال است که چک‌باکس ویژگی فعال نباشد
        // چون حتی اگر ویژگی تنظیمات content/style نداشته باشد، تب شورتکد و پیشرفته وجود دارد
        $is_disabled = !isset($options[$feature_id]) || !$options[$feature_id];
        ?>
        <button type="button" class="bebanstore-show-settings button <?php echo $is_disabled ? 'disabled' : ''; ?>" title="<?php _e('تنظیمات اضافی', 'bebanstore'); ?>">
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M12 15C13.6569 15 15 13.6569 15 12C15 10.3431 13.6569 9 12 9C10.3431 9 9 10.3431 9 12C9 13.6569 10.3431 15 12 15Z" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
                <path d="M2 12.8799V11.1199C2 10.0799 2.85 9.21994 3.9 9.21994C5.71 9.21994 6.45 7.93994 5.54 6.36994C5.02 5.46994 5.33 4.29994 6.24 3.77994L7.97 2.78994C8.76 2.31994 9.78 2.59994 10.25 3.38994L10.36 3.57994C11.26 5.14994 12.74 5.14994 13.65 3.57994L13.76 3.38994C14.23 2.59994 15.25 2.31994 16.04 2.78994L17.77 3.77994C18.68 4.29994 18.99 5.46994 18.47 6.36994C17.56 7.93994 18.3 9.21994 20.11 9.21994C21.15 9.21994 22.01 10.0699 22.01 11.1199V12.8799C22.01 13.9199 21.16 14.7799 20.11 14.7799C18.3 14.7799 17.56 16.0599 18.47 17.6299C18.99 18.5399 18.68 19.6999 17.77 20.2199L16.04 21.2099C15.25 21.6799 14.23 21.3999 13.76 20.6099L13.65 20.4199C12.75 18.8499 11.27 18.8499 10.36 20.4199L10.25 20.6099C9.78 21.3999 8.76 21.6799 7.97 21.2099L6.24 20.2199C5.33 19.6999 5.02 18.5299 5.54 17.6299C6.45 16.0599 5.71 14.7799 3.9 14.7799C2.85 14.7799 2 13.9199 2 12.8799Z" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
            </svg>
        </button>
        <?php
    }

    /**
     * Render help section
     *
     * @param string $feature_id Feature ID
     * @param array $feature_settings Feature settings
     * @return void
     */
    public function render_help_section($feature_id, $feature_settings) {
        if (!isset($feature_settings[$feature_id]['help'])) {
            return;
        }

        $help = $feature_settings[$feature_id]['help'];
        ?>
        <div class="bebanstore-help-section">
            <div class="bebanstore-help-header">
                <h4 class="bebanstore-help-title">
                    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M13.26 3.59997L5.04997 12.29C4.73997 12.62 4.43997 13.27 4.37997 13.72L4.00997 16.96C3.87997 18.13 4.71997 18.93 5.87997 18.73L9.09997 18.18C9.54997 18.1 10.18 17.77 10.49 17.43L18.7 8.73997C20.12 7.23997 20.76 5.52997 18.55 3.43997C16.35 1.36997 14.68 2.09997 13.26 3.59997Z" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
                        <path d="M11.89 5.05005C12.32 7.81005 14.56 9.92005 17.34 10.2" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
                        <path d="M3 22H21" stroke="#292D32" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
                    </svg>
                    <?php echo esc_html($help['title']); ?>
                </h4>
            </div>
            <div class="bebanstore-help-content">
                <?php if (is_array($help['content'])): ?>
                    <ul class="bebanstore-help-list">
                        <?php foreach ($help['content'] as $item): ?>
                            <li><?php echo wp_kses_post($item); ?></li>
                        <?php endforeach; ?>
                    </ul>
                <?php else: ?>
                    <p><?php echo wp_kses_post($help['content']); ?></p>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

    /**
     * Render warning section
     *
     * @param string $feature_id Feature ID
     * @param array $feature_settings Feature settings
     * @return void
     */
    public function render_warning_section($feature_id, $feature_settings) {
        if (!isset($feature_settings[$feature_id]['warning'])) {
            return;
        }

        $warning = $feature_settings[$feature_id]['warning'];
        ?>
        <div class="bebanstore-warning-section">
            <div class="bebanstore-warning-header">
                <h4 class="bebanstore-warning-title">
                    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M12 9V13" stroke="#DC3545" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                        <path d="M12 17H12.01" stroke="#DC3545" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                        <path d="M12 22C17.5 22 22 17.5 22 12C22 6.5 17.5 2 12 2C6.5 2 2 6.5 2 12C2 17.5 6.5 22 12 22Z" stroke="#DC3545" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                    </svg>
                    <?php echo esc_html($warning['title']); ?>
                </h4>
            </div>
            <div class="bebanstore-warning-content">
                <?php if (is_array($warning['content'])): ?>
                    <ul class="bebanstore-warning-list">
                        <?php foreach ($warning['content'] as $item): ?>
                            <li><?php echo wp_kses_post($item); ?></li>
                        <?php endforeach; ?>
                    </ul>
                <?php else: ?>
                    <p><?php echo wp_kses_post($warning['content']); ?></p>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

    /**
     * Render Elementor widget toggle (checkbox)
     * 
     * @param string $feature_id Feature ID
     * @param array $feature_settings Feature settings
     * @param array $options Current options
     */
    private function render_elementor_widget_toggle($feature_id, $feature_settings, $options, $disabled = false) {
        $elementor_data = isset($feature_settings['elementor_widget']) 
            ? $feature_settings['elementor_widget'] 
            : array();
        
        // بررسی وضعیت فعال بودن المان المنتور
        $elementor_widgets_enabled = isset($options['elementor_widgets_enabled']) 
            ? $options['elementor_widgets_enabled'] 
            : array();
        
        $is_enabled = isset($elementor_widgets_enabled[$feature_id]) 
            ? $elementor_widgets_enabled[$feature_id] 
            : false;
        
        // استفاده از render_field برای نمایش checkbox با استایل یکسان با سایر فیلدها
        $field_name = 'bebanstore_options[elementor_widgets_enabled][' . $feature_id . ']';
        $field_config = array(
            'type' => 'checkbox',
            'label' => isset($elementor_data['title']) 
                ? $elementor_data['title'] 
                : __('فعال‌سازی المان المنتور', 'bebanstore'),
            'tooltips' => array(
                array(
                    'type' => 'info',
                    'text' => isset($elementor_data['description']) 
                        ? $elementor_data['description'] 
                        : __('با فعال کردن این گزینه، المان المنتور این ویژگی در ویرایشگر المنتور در دسترس خواهد بود.', 'bebanstore')
                )
            ),
            'default' => false,
            'disabled' => $disabled,
            'readonly' => $disabled
        );
        
        $this->render_field($field_name, $field_config, $is_enabled);
    }

    /**
     * Check if Pro is activated
     * 
     * @return bool Whether Pro is activated
     */
    public function is_pro_activated() {
        // Check if Pro plugin class exists and is activated
        if (class_exists('BebanStore_Pro')) {
            return BebanStore_Pro::is_activated();
        }
        
        // Check via filter for custom activation logic
        return apply_filters('bebanstore_pro_activation_check', false);
    }

    /**
     * Render Pro badge
     * 
     * @param string $feature_id Feature ID
     * @return void
     */
    public function render_pro_badge($feature_id) {
        if (!$this->is_pro_activated()) {
            echo BebanStore_Feature_Status::get_status_badge('premium');
        }
    }

    /**
     * Render feature badges (Pro / Freemium)
     *
     * @param string $feature_id Feature ID
     * @param array  $feature_setting Feature configuration
     * @return void
     */
    public function render_feature_badges($feature_id, $feature_setting) {
        $is_pro_feature = isset($feature_setting['is_pro_feature']) && $feature_setting['is_pro_feature'];

        if ($is_pro_feature) {
            echo BebanStore_Feature_Status::get_active_pro_badge(true);
            return;
        }

        $has_pro_settings = isset($feature_setting['has_pro_settings']) && $feature_setting['has_pro_settings'];
        if ($has_pro_settings) {
            echo BebanStore_Feature_Status::get_freemium_badge();
        }
    }

    /**
     * Check if Pro plugin is active
     * 
     * @return bool
     */
    private function has_pro_plugin_active() {
        return class_exists('BebanStore_Pro');
    }

    /**
     * Get tab content title
     * 
     * @param string $tab_key Tab key
     * @return string Tab content title
     */
    private function get_tab_content_title($tab_key) {
        $titles = array(
            'content' => __('تنظیمات', 'bebanstore'),
            'style' => __('تنظیمات استایل', 'bebanstore'),
            'shortcode' => __('شورتکد', 'bebanstore'),
            'advanced' => __('تنظیمات پیشرفته', 'bebanstore'),
            'help' => __('روش استفاده', 'bebanstore'),
            'pro' => __('تنظیمات پرو', 'bebanstore')
        );
        
        return isset($titles[$tab_key]) ? $titles[$tab_key] : '';
    }

    /**
     * Group settings by tab
     * 
     * @param array $settings Settings array
     * @return array Grouped settings
     */
    private function group_settings_by_tab($settings) {
        $grouped = array(
            'content' => array(),
            'style' => array(),
            'advanced' => array(),
            'pro' => array()
        );
        
        foreach ($settings as $key => $setting) {
            $tab = isset($setting['tab']) ? $setting['tab'] : 'content';
            $grouped[$tab][$key] = $setting;
        }
        
        // Remove empty tabs
        foreach ($grouped as $tab_key => $tab_settings) {
            if (empty($tab_settings)) {
                unset($grouped[$tab_key]);
            }
        }
        
        return $grouped;
    }

    /**
     * Render tabbed settings
     * 
     * @param string $feature_id Feature ID
     * @param array $feature_settings Feature settings configuration
     * @param array $feature_options Current feature options
     * @return void
     */
    public function render_tabbed_settings($feature_id, $feature_settings, $feature_options) {
        $has_pro_settings = isset($feature_settings['has_pro_settings']) && $feature_settings['has_pro_settings'];
        $settings = isset($feature_settings['settings']) ? $feature_settings['settings'] : array();
        
        // Group settings by tab
        $grouped_settings = $this->group_settings_by_tab($settings);
        
        // Tab labels and order
        $tab_labels = array(
            'content' => __('محتوا', 'bebanstore'),
            'style' => __('استایل', 'bebanstore'),
            'help' => __('روش استفاده', 'bebanstore'),
            'pro' => __('پرو', 'bebanstore'),
            'advanced' => __('پیشرفته', 'bebanstore'),
            'shortcode' => __('شورتکد', 'bebanstore')
        );
        
        // Determine which tabs to show
        $tabs_to_show = array();
        
        // Check tabs that have settings
        foreach ($grouped_settings as $tab_key => $tab_settings) {
            if (!empty($tab_settings)) {
                $tabs_to_show[] = $tab_key;
            }
        }
        
        // Check if feature has shortcode
        $has_shortcode = false;
        $shortcode_data = array();
        if (isset($this->features[$feature_id])) {
            $feature = $this->features[$feature_id];
            if ($feature instanceof BebanStore_Shortcode_Feature) {
                $has_shortcode = true;
                $shortcode_data = array(
                    'title' => method_exists($feature, 'get_title') ? $feature->get_title() : '',
                    'shortcode' => method_exists($feature, 'get_shortcode') ? $feature->get_shortcode() : '',
                    'description' => method_exists($feature, 'get_usage_instructions') ? $feature->get_usage_instructions() : ''
                );
            }
        }
        
        // Check if feature has shortcode in settings
        if (!$has_shortcode && isset($feature_settings['shortcode']) && !empty($feature_settings['shortcode'])) {
            $has_shortcode = true;
            $shortcode_data = array(
                'title' => isset($feature_settings['shortcode']['title']) ? $feature_settings['shortcode']['title'] : '',
                'shortcode' => isset($feature_settings['shortcode']['shortcode']) ? $feature_settings['shortcode']['shortcode'] : '',
                'description' => isset($feature_settings['shortcode']['description']) ? $feature_settings['shortcode']['description'] : ''
            );
        }
        
        // Check if feature has help or warning
        $has_help_content = (isset($feature_settings['help']) && !empty($feature_settings['help'])) || 
                           (isset($feature_settings['warning']) && !empty($feature_settings['warning']));
        $has_pro_elementor = isset($feature_settings['elementor_widget']) && !empty($feature_settings['elementor_widget']);
        
        // Show help tab (روش استفاده) if has shortcode, help/warning, or elementor widget (after style)
        if (($has_shortcode || $has_help_content || $has_pro_elementor) && !in_array('help', $tabs_to_show)) {
            $tabs_to_show[] = 'help';
        }
        
        // Add Pro tab if feature has pro settings or elementor widget (after help, before Advanced)
        if (($has_pro_settings || $has_pro_elementor) && !in_array('pro', $tabs_to_show)) {
            $tabs_to_show[] = 'pro';
        }
        
        // Always add Advanced tab (for reset button) - last
        if (!in_array('advanced', $tabs_to_show)) {
            $tabs_to_show[] = 'advanced';
        }
        
        // Build available tabs in correct order
        $available_tabs = array();
        foreach ($tab_labels as $tab_key => $tab_label) {
            if (in_array($tab_key, $tabs_to_show)) {
                $available_tabs[$tab_key] = $tab_label;
            }
        }
        
        // اگه فقط تب Advanced وجود داره و help هم نداره، بدون تب نمایش بده
        $has_help_for_display = ($has_shortcode || $has_help_content || $has_pro_elementor);
        if (count($available_tabs) === 1 && isset($available_tabs['advanced']) && !$has_help_for_display) {
            echo '<div class="bebanstore-reset-section-simple">';
            echo '<p class="bebanstore-reset-section-description">' . __('با استفاده از این دکمه، تمام تنظیمات این ویژگی به مقادیر پیش‌فرض بازگردانده می‌شود.', 'bebanstore') . '</p>';
            
            if (isset($feature_settings['has_settings']) && $feature_settings['has_settings']) {
                echo '<button type="button" class="bebanstore-reset-settings button" data-feature="' . esc_attr($feature_id) . '">';
                echo '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">';
                echo '<path d="M14.89 5.08005C18.75 5.57005 21.8 8.89005 21.8 12.91C21.8 17.26 18.26 20.8 13.91 20.8C9.56001 20.8 6.02001 17.26 6.02001 12.91C6.02001 10.54 7.03001 8.42005 8.66001 6.97005" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>';
                echo '<path d="M12.08 7.47998L15.15 4.13998" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>';
                echo '<path d="M12.08 7.48L15.34 10.37" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>';
                echo '</svg>';
                echo '<span>' . __('بازنشانی تنظیمات', 'bebanstore') . '</span>';
                echo '</button>';
            }
            echo '</div>';
            return;
        }
        
        // If only one tab (other than advanced and help), render without tabs
        if (count($available_tabs) <= 1 && !$has_help_for_display) {
            echo '<div class="bebanstore-flelds-subsettings">';
            foreach ($settings as $setting_id => $setting) {
                $value = isset($feature_options[$feature_id][$setting_id]) 
                    ? $feature_options[$feature_id][$setting_id] 
                    : (isset($setting['default']) ? $setting['default'] : '');
                
                $field_name = 'bebanstore_options[feature_settings][' . $feature_id . '][' . $setting_id . ']';
                $this->render_field($field_name, $setting, $value);
            }
            echo '</div>';
            return;
        }
        
        // Render tabs
        echo '<div class="bebanstore-feature-tabs-container">';
        
        // Render tab navigation
        echo '<div class="bebanstore-feature-settings-tabs">';
        $first_tab = true;
        foreach ($available_tabs as $tab_key => $tab_label) {
            $active_class = $first_tab ? ' active' : '';
            
            echo '<button type="button" class="bebanstore-feature-settings-tab' . $active_class . '" data-tab="' . esc_attr($tab_key) . '">';
            echo esc_html($tab_label);
            echo '</button>';
            
            $first_tab = false;
        }
        echo '</div>';
        
        // Render tab content
        $first_tab = true;
        foreach ($available_tabs as $tab_key => $tab_label) {
            $active_class = $first_tab ? ' active' : '';
            
            // Reset section_open برای هر tab
            $this->section_open = false;
            
            echo '<div class="bebanstore-feature-tab-content' . $active_class . '" data-tab="' . esc_attr($tab_key) . '">';
            
            // Check if pro tab
            if ($tab_key === 'pro') {
                // Check if there are actual pro settings defined
                $has_pro_settings_defined = isset($grouped_settings['pro']) && !empty($grouped_settings['pro']);
                $has_pro_elementor = isset($feature_settings['has_pro_elementor']) && $feature_settings['has_pro_elementor'];
                
                if ($this->has_pro_plugin_active()) {
                    // Pro plugin is active
                    // 1. Render pro settings (if exists)
                    if ($has_pro_settings_defined) {
                        echo '<div class="bebanstore-flelds-subsettings">';
                        foreach ($grouped_settings['pro'] as $setting_id => $setting) {
                            $all_values = isset($feature_options[$feature_id]) ? $feature_options[$feature_id] : array();
                            
                            $value = isset($feature_options[$feature_id][$setting_id]) 
                                ? $feature_options[$feature_id][$setting_id] 
                                : (isset($setting['default']) ? $setting['default'] : '');
                            
                            $field_name = 'bebanstore_options[feature_settings][' . $feature_id . '][' . $setting_id . ']';
                            
                            // برای columns و accordion، $all_values رو به عنوان value پاس بده
                            $setting_type = isset($setting['type']) ? $setting['type'] : 'text';
                            if (in_array($setting_type, array('columns', 'accordion_group'))) {
                                $this->render_field($field_name, $setting, $all_values);
                            } else {
                                $this->render_field($field_name, $setting, $value);
                            }
                        }
                        
                        // بستن آخرین section اگر باز بود
                        $this->close_open_section();
                        
                        echo '</div>';
                        
                        // جداکننده اگر المان المنتور هم هست
                        if ($has_pro_elementor) {
                            echo '<div class="bebanstore-pro-section-divider"></div>';
                        }
                    }
                    
                    // 2. Render Elementor widget toggle (if exists)
                    if ($has_pro_elementor) {
                        $options = get_option('bebanstore_options', array());
                        $this->render_elementor_widget_toggle($feature_id, $feature_settings, $options);
                    }
                    
                    // 3. If neither exists, show coming soon
                    if (!$has_pro_settings_defined && !$has_pro_elementor) {
                        echo '<div class="bebanstore-pro-coming-soon">';
                        echo '<div class="bebanstore-pro-coming-soon-icon">';
                        echo '<svg width="64" height="64" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">';
                        echo '<path d="M12 2L15.09 8.26L22 9.27L17 14.14L18.18 21.02L12 17.77L5.82 21.02L7 14.14L2 9.27L8.91 8.26L12 2Z" fill="#8D9BFF"/>';
                        echo '</svg>';
                        echo '</div>';
                        echo '<h3>' . __('تنظیمات پیشرفته پرو', 'bebanstore') . '</h3>';
                        echo '<p>' . __('تنظیمات پیشرفته این ویژگی در نسخه بعدی پلاگین پرو اضافه خواهد شد.', 'bebanstore') . '</p>';
                        echo '<p class="bebanstore-pro-coming-soon-note">' . __('شما می‌توانید از تمام ویژگی‌های فعلی نسخه پرو استفاده کنید.', 'bebanstore') . '</p>';
                        echo '</div>';
                    }
                } else {
                    // Pro plugin is not active - show compact notice and disabled fields
                    echo '<div class="bebanstore-pro-tab-notice">';
                    echo '<div class="bebanstore-pro-tab-notice-content">';
                    echo '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">';
                    echo '<path d="M12 2L15.09 8.26L22 9.27L17 14.14L18.18 21.02L12 17.77L5.82 21.02L7 14.14L2 9.27L8.91 8.26L12 2Z" fill="#8D9BFF"/>';
                    echo '</svg>';
                    echo '<div class="bebanstore-pro-tab-notice-text">';
                    echo '<strong>' . __('این تنظیمات در نسخه پرو قابل دسترسی است', 'bebanstore') . '</strong>';
                    echo '<p>' . __('با ارتقا به نسخه پرو، می‌توانید از تنظیمات پیشرفته این ویژگی استفاده کنید.', 'bebanstore') . '</p>';
                    echo '</div>';
                    echo '<a href="https://beban.ir/" class="bebanstore-pro-tab-upgrade-btn" target="_blank">';
                    echo __('ارتقا به نسخه پرو', 'bebanstore');
                    echo '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">';
                    echo '<path d="M5 12h14M12 5l7 7-7 7" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>';
                    echo '</svg>';
                    echo '</a>';
                    echo '</div>';
                    echo '</div>';
                    
                    // Render pro settings as disabled if they exist
                    if ($has_pro_settings_defined) {
                        echo '<div class="bebanstore-pro-settings-disabled">';
                        foreach ($grouped_settings['pro'] as $setting_id => $setting) {
                            $all_values = isset($feature_options[$feature_id]) ? $feature_options[$feature_id] : array();
                            
                            $value = isset($feature_options[$feature_id][$setting_id]) 
                                ? $feature_options[$feature_id][$setting_id] 
                                : (isset($setting['default']) ? $setting['default'] : '');
                            
                            $field_name = 'bebanstore_options[feature_settings][' . $feature_id . '][' . $setting_id . ']';
                            
                            // Mark setting as disabled
                            $setting['disabled'] = true;
                            $setting['readonly'] = true;
                            
                            // Force section headers to be open when Pro is not active
                            if (isset($setting['type']) && $setting['type'] === 'section_header') {
                                $setting['open'] = true;
                            }
                            
                            // برای columns و accordion، $all_values رو به عنوان value پاس بده
                            $setting_type = isset($setting['type']) ? $setting['type'] : 'text';
                            if (in_array($setting_type, array('columns', 'accordion_group'))) {
                                $this->render_field($field_name, $setting, $all_values);
                            } else {
                                $this->render_field($field_name, $setting, $value);
                            }
                        }
                        
                        // بستن آخرین section اگر باز بود
                        $this->close_open_section();
                        
                    echo '</div>';
                    }
                    
                    // Render Elementor widget toggle as disabled (if exists)
                    if ($has_pro_elementor) {
                        echo '<div class="bebanstore-pro-settings-disabled">';
                        $options = get_option('bebanstore_options', array());
                        $this->render_elementor_widget_toggle($feature_id, $feature_settings, $options, true); // true = disabled
                    echo '</div>';
                    }
                }
            } elseif ($tab_key === 'help') {
                // Render help tab content (روش استفاده)
                echo '<div class="bebanstore-help-content">';
                
                // Tab title
                $tab_title = $this->get_tab_content_title($tab_key);
                if (!empty($tab_title)) {
                    echo '<h3 class="bebanstore-tab-content-title">' . esc_html($tab_title) . '</h3>';
                }
                
                // 1. Shortcode code display (if has shortcode)
                if ($has_shortcode && !empty($shortcode_data['shortcode'])) {
                    echo '<div class="bebanstore-shortcode-display">';
                    echo '<div class="bebanstore-shortcode-code-wrapper">';
                    echo '<code class="bebanstore-shortcode-code-text">' . esc_html($shortcode_data['shortcode']) . '</code>';
                    echo '<button type="button" class="bebanstore-copy-shortcode-tab button">';
                    echo '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">';
                    echo '<path d="M16 12.9V17.1C16 20.6 14.6 22 11.1 22H6.9C3.4 22 2 20.6 2 17.1V12.9C2 9.4 3.4 8 6.9 8H11.1C14.6 8 16 9.4 16 12.9Z" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>';
                    echo '<path d="M22 6.9V11.1C22 14.6 20.6 16 17.1 16H16V12.9C16 9.4 14.6 8 11.1 8H8V6.9C8 3.4 9.4 2 12.9 2H17.1C20.6 2 22 3.4 22 6.9Z" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>';
                    echo '</svg>';
                    echo '<span>' . __('کپی شورتکد', 'bebanstore') . '</span>';
                    echo '</button>';
                    echo '</div>';
                    echo '</div>';
                }
                
                // 2. Shortcode description (if has shortcode)
                if ($has_shortcode && !empty($shortcode_data['description'])) {
                    echo '<div class="bebanstore-shortcode-description">';
                    echo '<h4>' . __('راهنمای استفاده:', 'bebanstore') . '</h4>';
                    echo '<p>' . wp_kses_post(nl2br($shortcode_data['description'])) . '</p>';
                    echo '</div>';
                }
                
                // 3. Render help section (if has help)
                if (isset($feature_settings['help']) && !empty($feature_settings['help'])) {
                    echo '<div class="bebanstore-help-section">';
                    if (isset($feature_settings['help']['title'])) {
                        echo '<h3 class="bebanstore-help-title">' . esc_html($feature_settings['help']['title']) . '</h3>';
                    }
                    if (isset($feature_settings['help']['content'])) {
                        if (is_array($feature_settings['help']['content'])) {
                            echo '<ul class="bebanstore-help-list">';
                            foreach ($feature_settings['help']['content'] as $help_item) {
                                echo '<li>' . wp_kses_post($help_item) . '</li>';
                            }
                            echo '</ul>';
                        } else {
                            echo '<p>' . wp_kses_post($feature_settings['help']['content']) . '</p>';
                        }
                    }
                    echo '</div>';
                }
                
                // 4. Render warning section (if has warning)
                if (isset($feature_settings['warning']) && !empty($feature_settings['warning'])) {
                    echo '<div class="bebanstore-warning-section">';
                    if (isset($feature_settings['warning']['title'])) {
                        echo '<h3 class="bebanstore-warning-title">' . esc_html($feature_settings['warning']['title']) . '</h3>';
                    }
                    if (isset($feature_settings['warning']['content'])) {
                        if (is_array($feature_settings['warning']['content'])) {
                            echo '<ul class="bebanstore-warning-list">';
                            foreach ($feature_settings['warning']['content'] as $warning_item) {
                                echo '<li>' . wp_kses_post($warning_item) . '</li>';
                            }
                            echo '</ul>';
                        } else {
                            echo '<p>' . wp_kses_post($feature_settings['warning']['content']) . '</p>';
                        }
                    }
                    echo '</div>';
                }
                
                // 5. Show Elementor widget upgrade notice (if has elementor widget)
                $has_pro_elementor = isset($feature_settings['elementor_widget']) && !empty($feature_settings['elementor_widget']);
                if ($has_pro_elementor) {
                    echo '<div class="bebanstore-elementor-upgrade-notice" style="margin-top: 20px; padding: 16px; background-color: #f0f7ff; border-radius: 4px;">';
                    echo '<p style="margin: 0; color: #333; font-size: 14px; line-height: 1.6;">';
                    echo '<strong>' . __(' نکته:', 'bebanstore') . '</strong> ';
                    echo __('با ارتقا افزونه به نسخه پرو، به جای شورت‌کد می‌توانی از المان المنتور با دسترسی‌های بیشتر برای استایل‌دهی استفاده کنی.', 'bebanstore');
                    echo '</p>';
                    echo '</div>';
                }
                
                echo '</div>';
            } elseif ($tab_key === 'advanced') {
                // Render advanced tab content (reset button)
                echo '<div class="bebanstore-advanced-tab-content">';
                
                // Tab title
                $tab_title = $this->get_tab_content_title($tab_key);
                if (!empty($tab_title)) {
                    echo '<h3 class="bebanstore-tab-content-title">' . esc_html($tab_title) . '</h3>';
                }
                
                // Advanced settings section
                echo '<div class="bebanstore-advanced-settings-section">';
                
                // Render advanced settings if exists
                if (isset($grouped_settings['advanced']) && !empty($grouped_settings['advanced'])) {
                    echo '<div class="bebanstore-flelds-subsettings">';
                    foreach ($grouped_settings['advanced'] as $setting_id => $setting) {
                        $all_values = isset($feature_options[$feature_id]) ? $feature_options[$feature_id] : array();
                        
                        $value = isset($feature_options[$feature_id][$setting_id]) 
                            ? $feature_options[$feature_id][$setting_id] 
                            : (isset($setting['default']) ? $setting['default'] : '');
                        
                        $field_name = 'bebanstore_options[feature_settings][' . $feature_id . '][' . $setting_id . ']';
                        
                        // برای columns و accordion، $all_values رو به عنوان value پاس بده
                        $setting_type = isset($setting['type']) ? $setting['type'] : 'text';
                        if (in_array($setting_type, array('columns', 'accordion_group'))) {
                            $this->render_field($field_name, $setting, $all_values);
                        } else {
                            $this->render_field($field_name, $setting, $value);
                        }
                    }
                    
                    // بستن آخرین section اگر باز بود
                    $this->close_open_section();
                    
                    echo '</div>';
                }
                
                echo '</div>';
                
                // Reset settings section
                echo '<div class="bebanstore-reset-section">';
                echo '<h4 class="bebanstore-reset-section-title">' . __('بازنشانی تنظیمات', 'bebanstore') . '</h4>';
                echo '<p class="bebanstore-reset-section-description">' . __('با استفاده از این دکمه، تمام تنظیمات این ویژگی به مقادیر پیش‌فرض بازگردانده می‌شود. این عملیات قابل بازگشت نیست.', 'bebanstore') . '</p>';
                
                // Render reset button
                if (isset($feature_settings['has_settings']) && $feature_settings['has_settings']) {
                    echo '<button type="button" class="bebanstore-reset-settings button" data-feature="' . esc_attr($feature_id) . '">';
                    echo '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">';
                    echo '<path d="M14.89 5.08005C18.75 5.57005 21.8 8.89005 21.8 12.91C21.8 17.26 18.26 20.8 13.91 20.8C9.56001 20.8 6.02001 17.26 6.02001 12.91C6.02001 10.54 7.03001 8.42005 8.66001 6.97005" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>';
                    echo '<path d="M12.08 7.47998L15.15 4.13998" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>';
                    echo '<path d="M12.08 7.48L15.34 10.37" stroke="#292D32" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>';
                    echo '</svg>';
                    echo '<span>' . __('بازنشانی تنظیمات', 'bebanstore') . '</span>';
                    echo '</button>';
                }
                
                echo '</div>';
                
                echo '</div>';
            } else {
                // Render regular tab settings
                echo '<div class="bebanstore-regular-tab-content">';
                
                // Tab title
                $tab_title = $this->get_tab_content_title($tab_key);
                if (!empty($tab_title)) {
                    echo '<h3 class="bebanstore-tab-content-title">' . esc_html($tab_title) . '</h3>';
                }
                
                echo '<div class="bebanstore-flelds-subsettings">';
                if (isset($grouped_settings[$tab_key]) && !empty($grouped_settings[$tab_key])) {
                    foreach ($grouped_settings[$tab_key] as $setting_id => $setting) {
                        // بررسی شرط نمایش (Conditional Fields)
                        $all_values = isset($feature_options[$feature_id]) ? $feature_options[$feature_id] : array();
                        if (!$this->should_display_field($setting, $all_values)) {
                            continue;
                        }
                        
                        $value = isset($feature_options[$feature_id][$setting_id]) 
                            ? $feature_options[$feature_id][$setting_id] 
                            : (isset($setting['default']) ? $setting['default'] : '');
                        
                        $field_name = 'bebanstore_options[feature_settings][' . $feature_id . '][' . $setting_id . ']';
                        
                        // برای columns و accordion، $all_values رو به عنوان value پاس بده
                        $setting_type = isset($setting['type']) ? $setting['type'] : 'text';
                        if (in_array($setting_type, array('columns', 'accordion_group'))) {
                            $this->render_field($field_name, $setting, $all_values);
                        } else {
                            $this->render_field($field_name, $setting, $value);
                        }
                    }
                    
                    // بستن آخرین section اگر باز بود
                    $this->close_open_section();
                }
                echo '</div>';
                
                echo '</div>';
            }
            
            echo '</div>';
            $first_tab = false;
        }
        
        echo '</div>';
    }

} 