/**
 * Ajax Add to Cart Feature
 * 
 * این فایل مسئول پیاده‌سازی قابلیت افزودن به سبد خرید با Ajax است.
 * این قابلیت به کاربران اجازه می‌دهد بدون بارگذاری مجدد صفحه، محصولات را به سبد خرید اضافه کنند.
 * 
 * @package BebanStore
 * @since 1.0.0
 */

jQuery(document).ready(function($) {
    // Check if WooCommerce parameters are available
    if (typeof wc_cart_fragments_params === 'undefined') {
        return;
    }

    // Check if BebanStore parameters are available
    if (typeof bebanstore_ajax_add_to_cart === 'undefined') {
        return;
    }

    /**
     * ذخیره متن اصلی دکمه‌های افزودن به سبد خرید
     * این کار برای بازگرداندن متن اصلی دکمه پس از اتمام انیمیشن ضروری است
     */
    $('.single_add_to_cart_button').each(function() {
        var $button = $(this);
        if (!$button.data('original-text')) {
            $button.data('original-text', $button.text().trim());
        }
    });

    // Function to show toast notification
    function showToast(message, type = 'success') {
        if (typeof BebanStoreToast !== 'undefined') {
            BebanStoreToast[type](message);
        } else {
            // Fallback to simple alert if toast is not available
            alert(message);
        }
    }

    // Function to refresh cart fragments
    var refreshCartFragments = {
        url: wc_cart_fragments_params.wc_ajax_url.toString().replace('%%endpoint%%', 'get_refreshed_fragments'),
        type: 'POST',
        success: function(data) {
            if (data && data.fragments) {
                $.each(data.fragments, function(key, value) {
                    $(key).replaceWith(value);
                });
                $(document.body).trigger('wc_fragments_refreshed');
            }
        }
    };

    /**
     * مدیریت کلیک روی دکمه افزودن به سبد خرید
     * این بخش اصلی ترین منطق افزودن به سبد خرید را پیاده‌سازی می‌کند
     */
    $(document).on('click', '.single_add_to_cart_button', function(e) {
        e.preventDefault();

        const $form = $(this).closest('form.cart');
        const $button = $(this);
        const isVariableProduct = $form.hasClass('variations_form');
        
        // Get product data based on product type
        let productData = {};
        let selectedVariations = [];
        
        if (isVariableProduct) {
            // Handle variable product
            const variationId = $form.find('input[name="variation_id"]').val();
            
            if (!variationId) {
                return;
            }
            
            // Collect selected variations
            $form.find('select[name^="attribute_"]').each(function() {
                const $select = $(this);
                const attributeName = $select.attr('name').replace('attribute_', '');
                const attributeLabel = $select.closest('.variations').find('label[for="' + $select.attr('id') + '"]').text().replace(':', '').trim();
                const attributeValue = $select.find('option:selected').text();
                if (attributeValue) {
                    selectedVariations.push({
                        name: attributeLabel || attributeName,
                        value: attributeValue
                    });
                }
            });
            
            productData = {
                product_id: $form.find('input[name="product_id"]').val(),
                variation_id: variationId,
                quantity: $form.find('input[name="quantity"]').val() || 1,
                variation: {}
            };

            // Get all variation attributes
            $form.find('select[name^="attribute_"]').each(function() {
                const attributeName = $(this).attr('name');
                const attributeValue = $(this).val();
                if (attributeValue) {
                    productData.variation[attributeName] = attributeValue;
                }
            });
        } else {
            /**
             * مدیریت محصولات ساده
             * این بخش برای محصولاتی که متغیر ندارند استفاده می‌شود
             */
            const productId = $button.val();
            
            if (!productId) {
                return;
            }

            // آماده‌سازی داده‌های محصول ساده
            productData = {
                product_id: productId,
                quantity: $form.find('input[name="quantity"]').val() || 1
            };
        }

        /**
         * نمایش حالت بارگذاری با سیستم یکپارچه
         */
        if (typeof BebanStoreLoader !== 'undefined') {
            // استفاده از سیستم یکپارچه loader با پشتیبانی از انواع انیمیشن
            const animationType = bebanstore_ajax_add_to_cart.animation_type || 'default';
            const overlayColor = bebanstore_ajax_add_to_cart.loader_overlay_color || '#EF4056';
            const animationColor = bebanstore_ajax_add_to_cart.loader_animation_color || '#ffffff';
            
            // انیمیشن‌های بصری: spinner, dots, text, یا default
            const customText = animationType === 'text' ? bebanstore_ajax_add_to_cart.i18n.adding : null;
            BebanStoreLoader.showLoaderWithCustomColors($button[0], animationType, animationColor, overlayColor, customText);
        } else {
            // Fallback به روش قدیمی
            $button.addClass('loading');
            
            if (bebanstore_ajax_add_to_cart.animation_type === 'spinner') {
                $button.attr('data-animation', 'spinner');
                $button.append('<span class="bebanstore-spinner"></span>');
            } else if (bebanstore_ajax_add_to_cart.animation_type === 'text') {
                $button.text(bebanstore_ajax_add_to_cart.i18n.adding);
            } else if (bebanstore_ajax_add_to_cart.animation_type === 'dots') {
                $button.attr('data-animation', 'dots');
                $button.append('<span class="bebanstore-dots"></span>');
            }
        }

        /**
         * ارسال درخواست Ajax
         * این بخش درخواست افزودن به سبد خرید را به سرور ارسال می‌کند
         */
        $.ajax({
            url: bebanstore_ajax_add_to_cart.ajax_url,
            type: 'POST',
            data: {
                action: 'bebanstore_add_to_cart',
                nonce: bebanstore_ajax_add_to_cart.nonce,
                product_id: productData.product_id,
                variation_id: productData.variation_id || 0,
                quantity: productData.quantity,
                variation: productData.variation || {}
            },
            success: function(response) {
                if (response.success) {
                    /**
                     * بروزرسانی قطعات سبد خرید
                     * این بخش المان‌های مرتبط با سبد خرید را بروزرسانی می‌کند
                     */
                    $.ajax(refreshCartFragments);

                    /**
                     * نمایش پیام موفقیت
                     * این بخش بر اساس تنظیمات کاربر، پیام مناسب را نمایش می‌دهد
                     */
                    if (bebanstore_ajax_add_to_cart.success_message_type === 'cart') {
                        // نمایش پاپ‌آپ سبد خرید
                        const $popup = $('#bebanstoreCartPopup');
                        const $popupContent = $popup.find('.bebanstore-cart-popup-content');
                        
                        // Update popup content
                        $popupContent.find('.bebanstore-cart-popup-product-title').text(response.data.title);
                        $popupContent.find('.bebanstore-cart-popup-product-price').html(response.data.price);
                        $popupContent.find('.bebanstore-cart-popup-image').attr('src', response.data.image);
                        
                        // Update variation attributes if available
                        const $variationsContainer = $popupContent.find('.bebanstore-cart-popup-product-variations');
                        $variationsContainer.empty();
                        
                        if (isVariableProduct && selectedVariations.length > 0) {
                            selectedVariations.forEach(function(variation) {
                                const $variationTag = $('<div class="bebanstore-cart-popup-product-variation">')
                                    .append($('<span class="bebanstore-cart-popup-product-variation-name">').text(variation.name + ':'))
                                    .append($('<span class="bebanstore-cart-popup-product-variation-value">').text(variation.value));
                                $variationsContainer.append($variationTag);
                            });
                        }
                        
                        // Show popup
                        $popup.fadeIn(200);
                    } else if (bebanstore_ajax_add_to_cart.success_message_type === 'simple') {
                        // Show toast notification using BebanStoreToast
                        showToast('محصول به سبد خرید اضافه شد', 'success');
                    }
                } else {
                    // Show info toast for stock quantity messages, error for other issues
                    const message = response.data && response.data.message 
                        ? response.data.message 
                        : 'خطا در افزودن محصول به سبد خرید';
                    
                    // Check if it's a stock quantity message
                    if (message.includes('حداکثر تعداد مجاز این محصول')) {
                        showToast(message, 'info');
                    } else {
                        showToast(message, 'error');
                    }
                }
            },
            error: function(xhr, status, error) {
                // Show error toast
                const errorMessage = 'خطا در اتصال به سرور. لطفاً دوباره تلاش کنید.';
                showToast(errorMessage, 'error');
            },
            complete: function() {
                /**
                 * Restore button state with integrated system
                 */
                if (typeof BebanStoreLoader !== 'undefined') {
                    // Hide loader for all animation types
                    BebanStoreLoader.hideLoader($button[0]);
                } else {
                    // Fallback to old method
                    $button.removeClass('loading');
                    if (bebanstore_ajax_add_to_cart.animation_type === 'spinner') {
                        $button.removeAttr('data-animation');
                        $button.find('.bebanstore-spinner').remove();
                    } else if (bebanstore_ajax_add_to_cart.animation_type === 'dots') {
                        $button.removeAttr('data-animation');
                        $button.find('.bebanstore-dots').remove();
                    }
                }
            }
        });
    });

    /**
     * Close popup by clicking outside
     * This section allows users to close the popup by clicking on the background
     */
    $(document).on('click', '#bebanstoreCartPopup', function(e) {
        if ($(e.target).is('#bebanstoreCartPopup')) {
            $(this).fadeOut(200);
        }
    });

    /**
     * Close popup by clicking close button
     * This section allows users to close the popup by clicking the close button
     */
    $(document).on('click', '.bebanstore-cart-popup-close', function() {
        $('#bebanstoreCartPopup').fadeOut(200);
    });

    /**
     * Handle WooCommerce's native add_to_cart action
     * Show toast notification when product is added via WooCommerce's default add to cart
     */
    $(document.body).on('added_to_cart', function(event, fragments, cart_hash, button) {
        // Show toast notification for WooCommerce's add_to_cart
        if (typeof BebanStoreToast !== 'undefined') {
            BebanStoreToast.success('محصول به سبد خرید اضافه شد');
        }
    });
}); 