(function($) {
    'use strict';

    class ProductCountdown {
        constructor(element) {
            this.element = element;
            this.productId = element.dataset.productId;
            
            // Get settings from data attributes
            this.settings = {
                countdownColor: element.dataset.countdownColor || '#000000',
                separatorColor: element.dataset.separatorColor || '#000000',
                labelColor: element.dataset.labelColor || '#666666',
                showLabels: element.dataset.showLabels === 'true',
                numberFontSize: element.dataset.numberFontSize || '20',
                separatorFontSize: element.dataset.separatorFontSize || '16',
                labelFontSize: element.dataset.labelFontSize || '12',
                boxPadding: element.dataset.boxPadding || '8',
                labelGap: element.dataset.labelGap || '8',
                cardBackgroundColor: element.dataset.cardBackgroundColor || '#ffffff',
                cardBorderColor: element.dataset.cardBorderColor || '#ededed',
                cardBorderWidth: element.dataset.cardBorderWidth || '1',
                cardBorderRadius: element.dataset.cardBorderRadius || '8',
                cardGap: element.dataset.cardGap || '4',
                displayStyle: element.dataset.displayStyle || 'card',
                dayLabel: 'روز',
                hourLabel: 'ساعت',
                minuteLabel: 'دقیقه',
                secondLabel: 'ثانیه',
            };
            
            // Parse dates with error checking
            const endDateStr = element.dataset.endDate;
            const endTimeStr = element.dataset.endTime;
            
            if (!endDateStr) {
                return;
            }

            try {
                this.endDate = new Date(endDateStr);
                if (isNaN(this.endDate.getTime())) {
                    throw new Error('Invalid end date format');
                }
            } catch (error) {
                return;
            }

            this.timer = null;
            this.init();
        }

        init() {
            if (!this.endDate || this.isExpired()) {
                this.element.innerHTML = '<div class="countdown-placeholder">تخفیف به پایان رسیده است</div>';
                return;
            }

            // ابتدا HTML را بسازیم
            this.updateTimer();
            // سپس تایمر را راه‌اندازی کنیم
            this.timer = setInterval(() => this.updateTimer(), 1000);
        }

        applyStyles() {
            // Apply colors to countdown elements
            const countdownElements = this.element.querySelectorAll('.bebanstore-countdown-number, .bebanstore-countdown-days, .bebanstore-countdown-hours, .bebanstore-countdown-minutes, .bebanstore-countdown-seconds');
            countdownElements.forEach(el => {
                el.style.color = this.settings.countdownColor;
                el.style.setProperty('font-size', this.settings.numberFontSize + 'px', 'important');
            });

            // Apply font size and color to separators
            const separatorElements = this.element.querySelectorAll('.bebanstore-countdown-separator');
            separatorElements.forEach(el => {
                el.style.setProperty('font-size', this.settings.separatorFontSize + 'px', 'important');
                el.style.setProperty('color', this.settings.separatorColor, 'important');
            });

            // Apply colors and font size to labels
            const labelElements = this.element.querySelectorAll('.bebanstore-countdown-label');
            labelElements.forEach(el => {
                el.style.setProperty('color', this.settings.labelColor, 'important');
                el.style.setProperty('font-size', this.settings.labelFontSize + 'px', 'important');
                // اضافه کردن فاصله عمودی برای برچسب‌ها در حالت کارتی
                if (this.settings.displayStyle === 'card') {
                    el.style.setProperty('margin-top', this.settings.labelGap + 'px', 'important');
                }
            });

            // Apply padding based on display style
            if (this.settings.displayStyle === 'inline') {
                // در حالت خطی: padding روی کل تایمر
                const timer = this.element.querySelector('.bebanstore-countdown-timer');
                if (timer) {
                    timer.style.setProperty('padding', this.settings.boxPadding + 'px', 'important');
                }
            } else if (this.settings.displayStyle === 'card') {
                // در حالت کارتی: اعمال gap و استایل‌های کادر
                const timer = this.element.querySelector('.bebanstore-countdown-timer');
                if (timer) {
                    timer.style.setProperty('gap', this.settings.cardGap + 'px', 'important');
                }
                
            const countdownItems = this.element.querySelectorAll('.bebanstore-countdown-item');
            countdownItems.forEach(item => {
                    item.style.setProperty('padding', this.settings.boxPadding + 'px', 'important');
                    item.style.setProperty('background-color', this.settings.cardBackgroundColor, 'important');
                    item.style.setProperty('border-color', this.settings.cardBorderColor, 'important');
                    item.style.setProperty('border-width', this.settings.cardBorderWidth + 'px', 'important');
                item.style.setProperty('border-style', 'solid', 'important');
                    item.style.setProperty('border-radius', this.settings.cardBorderRadius + 'px', 'important');
                });
            } else {
                // در حالت کلاسیک: فقط padding روی هر آیتم
                const countdownItems = this.element.querySelectorAll('.bebanstore-countdown-item');
                countdownItems.forEach(item => {
                item.style.setProperty('padding', this.settings.boxPadding + 'px', 'important');
            });
            }
        }

        isExpired() {
            const now = new Date();
            return now > this.endDate;
        }

        updateTimer() {
            const now = new Date();
            const timeLeft = this.endDate - now;

            if (timeLeft <= 0) {
                clearInterval(this.timer);
                this.element.innerHTML = '<div class="countdown-placeholder">تخفیف به پایان رسیده است</div>';
                return;
            }

            const days = Math.floor(timeLeft / (1000 * 60 * 60 * 24));
            const hours = Math.floor((timeLeft % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
            const minutes = Math.floor((timeLeft % (1000 * 60 * 60)) / (1000 * 60));
            const seconds = Math.floor((timeLeft % (1000 * 60)) / 1000);

            // ایجاد HTML بر اساس سبک نمایش
            let html = '<div class="bebanstore-countdown-timer">';
            
            if (this.settings.displayStyle === 'inline') {
                // سبک خطی - ترتیب اصلاح شده: برچسب قبل از عدد
                
                // Days
                if (this.settings.showLabels) {
                    html += `<span class="bebanstore-countdown-label bebanstore-countdown-day-label">${this.settings.dayLabel}</span>`;
                }
                html += `<span class="bebanstore-countdown-days">${String(days).padStart(2, '0')}</span>`;
                html += `<span class="bebanstore-countdown-separator">:</span>`;
                
                // Hours
                if (this.settings.showLabels) {
                    html += `<span class="bebanstore-countdown-label bebanstore-countdown-hour-label">${this.settings.hourLabel}</span>`;
                }
                html += `<span class="bebanstore-countdown-hours">${String(hours).padStart(2, '0')}</span>`;
                html += `<span class="bebanstore-countdown-separator">:</span>`;
                
                // Minutes
                if (this.settings.showLabels) {
                    html += `<span class="bebanstore-countdown-label bebanstore-countdown-minute-label">${this.settings.minuteLabel}</span>`;
                }
                html += `<span class="bebanstore-countdown-minutes">${String(minutes).padStart(2, '0')}</span>`;
                html += `<span class="bebanstore-countdown-separator">:</span>`;
                
                // Seconds
                if (this.settings.showLabels) {
                    html += `<span class="bebanstore-countdown-label bebanstore-countdown-second-label">${this.settings.secondLabel}</span>`;
                }
                html += `<span class="bebanstore-countdown-seconds">${String(seconds).padStart(2, '0')}</span>`;
                
                // آخرین جداکننده اضافه نمی‌شود
            } else {
                // سبک کلاسیک و کارتی
                // ساختار جدید: آیتم‌ها و جداکننده‌ها جدا از هم
                const items = [
                    { value: days, label: this.settings.dayLabel, show: days > 0 },
                    { value: hours, label: this.settings.hourLabel, show: true },
                    { value: minutes, label: this.settings.minuteLabel, show: true },
                    { value: seconds, label: this.settings.secondLabel, show: true }
                ];
                
                const visibleItems = items.filter(item => item.show);
                
                visibleItems.forEach((item, index) => {
                    html += `
                        <div class="bebanstore-countdown-item">
                            <span class="bebanstore-countdown-number">${String(item.value).padStart(2, '0')}</span>
                            ${this.settings.showLabels ? `<span class="bebanstore-countdown-label">${item.label}</span>` : ''}
                        </div>
                    `;
                    
                    // اضافه کردن جداکننده بین آیتم‌ها (نه در آخرین آیتم)
                    if (index < visibleItems.length - 1) {
                        html += `<span class="bebanstore-countdown-separator">:</span>`;
                    }
                });
            }
            
            html += '</div>';
            
            // اعمال HTML به المان
            this.element.innerHTML = html;
            
            // اعمال استایل‌ها
            this.applyStyles();
        }

        destroy() {
            if (this.timer) {
                clearInterval(this.timer);
            }
        }
    }

    // Initialize all countdowns on the page
    function initCountdowns() {
        const countdowns = document.querySelectorAll('.bebanstore-countdown');
        countdowns.forEach(countdown => {
            if (!countdown.dataset.initialized) {
                try {
                    new ProductCountdown(countdown);
                    countdown.dataset.initialized = 'true';
                } catch (error) {
                    // Silent fail
                }
            }
        });
    }

    // Cleanup function
    function cleanupCountdowns() {
        const countdowns = document.querySelectorAll('.bebanstore-countdown');
        countdowns.forEach(countdown => {
            if (countdown.countdownInstance) {
                countdown.countdownInstance.destroy();
            }
        });
    }

    // Initialize on page load
    document.addEventListener('DOMContentLoaded', initCountdowns);

    // Reinitialize on cart/checkout updates
    document.addEventListener('added_to_cart', initCountdowns);
    document.addEventListener('removed_from_cart', initCountdowns);
    document.addEventListener('updated_checkout', initCountdowns);

    // Reinitialize on product variation changes
    document.addEventListener('woocommerce_variation_has_changed', initCountdowns);
    document.addEventListener('found_variation', initCountdowns);
    document.addEventListener('reset_image', initCountdowns);

    // Cleanup before page unload
    window.addEventListener('beforeunload', cleanupCountdowns);

})(jQuery); 