<?php
/* 
Plugin Name: Beban Store
Plugin URI: https://beban.ir/academy/bebanstore/
Description: افزونه جامع و کاربردی برای بهبود فروشگاه ووکامرس با قابلیت‌های متنوع
Version: 1.0.7
Requires at least: 5.6
Requires PHP: 7.4
Author: BEBAN
Author URI: https://beban.ir/
License: GPL-2.0-or-later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: bebanstore
Domain Path: languages
*/

// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}

// Define plugin constants
define('BEBANSTORE_VERSION', '1.0.7');
define('BEBANSTORE_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('BEBANSTORE_PLUGIN_URL', plugin_dir_url(__FILE__));
define('BEBANSTORE_PLUGIN_NAME', 'ببان استور');
define('BEBANSTORE_PLUGIN_BASENAME', plugin_basename(__FILE__));
define('BEBANSTORE_PLUGIN_SLUG', 'bebanstore');

// Customize plugin menu icon and name
add_action('admin_menu', function() {
    global $menu;
    foreach ($menu as $key => $item) {
        if (isset($item[2]) && $item[2] === 'bebanstore-settings') {
            $menu[$key][0] = 'ببان استور';
            $menu[$key][6] = 'data:image/svg+xml;base64,' . base64_encode('<svg width="20" height="20" viewBox="0 0 376 512" fill="none" xmlns="http://www.w3.org/2000/svg"><g clip-path="url(#clip0_34_29)"><g clip-path="url(#clip1_34_29)"><path fill-rule="evenodd" clip-rule="evenodd" d="M0.864746 137.667H111.295V324.508C111.295 339.655 115.794 354.462 124.222 367.056C132.651 379.651 144.63 389.467 158.646 395.263C172.662 401.06 188.085 402.576 202.964 399.621C217.844 396.666 231.511 389.372 242.239 378.662C252.966 367.951 260.271 354.305 263.231 339.449C266.191 324.593 264.672 309.195 258.866 295.201C253.061 281.207 243.229 269.246 230.615 260.831C218.001 252.416 203.171 247.924 188 247.924H138.903V137.667H188C225.012 137.667 261.193 148.625 291.967 169.156C322.741 189.686 346.727 218.867 360.891 253.007C375.054 287.148 378.76 324.716 371.54 360.959C364.319 397.203 346.496 430.495 320.325 456.625C294.153 482.755 260.809 500.55 224.508 507.759C188.208 514.969 150.581 511.269 116.386 497.127C82.192 482.986 52.9654 459.038 32.4027 428.312C11.84 397.586 0.864746 361.462 0.864746 324.508L0.864746 137.667Z" fill="#2271b1"/><path fill-rule="evenodd" clip-rule="evenodd" d="M111.295 110.103H0.864746V-0.15332H56.0799C70.7114 -0.112685 84.732 5.70852 95.078 16.0383C105.424 26.3681 111.254 40.3666 111.295 54.9751V110.103Z" fill="#2271b1"/></g></g><defs><clipPath id="clip0_34_29"><rect width="374.271" height="512.307" fill="white" transform="translate(0.864746 -0.15332)"/></clipPath><clipPath id="clip1_34_29"><rect width="374.271" height="511.503" fill="white" transform="translate(0.864746 -0.15332)"/></clipPath></defs></svg>');
            break;
        }
    }
}, 999);

// Add custom icon to plugin action links
add_filter('plugin_action_links_' . BEBANSTORE_PLUGIN_BASENAME, function($links) {
    $settings_link = '<a href="' . admin_url('admin.php?page=bebanstore-settings') . '">' . __('تنظیمات', 'bebanstore') . '</a>';
    array_unshift($links, $settings_link);
    return $links;
});

// Load required files
// Base classes first
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-feature.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-shortcode-feature.php';

// Feature classes
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-ajax-add-to-cart.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-sales-progress.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-quantity-control.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-variable-price.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-cart-confirmation.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-countdown.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-express-shipping.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-tracking-code.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-product-attributes.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-product-discount.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-discount-badges.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-order-count.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-order-view.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-checkout-fields-manager.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-your-savings.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-wishlist.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-custom-font.php';
require_once BEBANSTORE_PLUGIN_DIR . 'includes/class-bebanstore-add-to-cart-button-text.php';

// Admin classes
require_once BEBANSTORE_PLUGIN_DIR . 'admin/class-bebanstore-settings.php';
require_once BEBANSTORE_PLUGIN_DIR . 'admin/settings/class-feature-settings-factory.php';
require_once BEBANSTORE_PLUGIN_DIR . 'admin/settings/class-settings-register.php';
require_once BEBANSTORE_PLUGIN_DIR . 'admin/settings/class-settings-render.php';

/**
 * Main plugin class
 */
class BebanStore {
    /**
     * Plugin options
     */
    private $options;

    /**
     * Settings instance
     */
    private $settings;

    /**
     * Features container
     * @var array
     */
    private $features = array();

    /**
     * Initialize the plugin
     */
    public function __construct() {
        // Load plugin options
        $this->options = get_option('bebanstore_options', array());
        
        // Initialize features
        $this->init_features();

        // Initialize settings with features
        $this->settings = new BebanStore_Settings($this->features);

        add_action('plugins_loaded', array($this, 'init'));
    }

    /**
     * Initialize plugin
     */
    public function init() {
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', array($this, 'woocommerce_missing_notice'));
            return;
        }
        
        // WordPress.org auto-loads translations for hosted plugins (no manual load required)
        
        // Add basic scripts and styles
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        
        // Add admin scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // Add custom CSS
        add_action('wp_head', array($this, 'output_custom_css'), 999);
        add_action('wp_footer', array($this, 'output_custom_css'), 999);
    }

    

    /**
     * Enqueue frontend scripts and styles
     */
    public function enqueue_scripts() {
        // Enqueue Toastify CSS
        wp_enqueue_style(
            'bebanstore-toastify',
            BEBANSTORE_PLUGIN_URL . 'assets/css/toastify.css',
            array(),
            BEBANSTORE_VERSION
        );

        // Enqueue Toastify JS
        wp_enqueue_script(
            'bebanstore-toastify',
            BEBANSTORE_PLUGIN_URL . 'assets/js/toastify.js',
            array(),
            BEBANSTORE_VERSION,
            true
        );

        // Enqueue BebanStore Loader CSS
        wp_enqueue_style(
            'bebanstore-loader',
            BEBANSTORE_PLUGIN_URL . 'assets/css/bebanstore-loader.css',
            array(),
            BEBANSTORE_VERSION
        );

        // Enqueue BebanStore Loader JS
        wp_enqueue_script(
            'bebanstore-loader',
            BEBANSTORE_PLUGIN_URL . 'assets/js/bebanstore-loader.js',
            array('jquery'),
            BEBANSTORE_VERSION,
            true
        );
    }

    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_admin_scripts() {
        // Enqueue Toastify CSS for admin
        wp_enqueue_style(
            'bebanstore-toastify-admin',
            BEBANSTORE_PLUGIN_URL . 'assets/css/toastify.css',
            array(),
            BEBANSTORE_VERSION
        );

        // Enqueue Toastify JS for admin
        wp_enqueue_script(
            'bebanstore-toastify-admin',
            BEBANSTORE_PLUGIN_URL . 'assets/js/toastify.js',
            array('jquery'),
            BEBANSTORE_VERSION,
            true
        );

    }

    /**
     * Display WooCommerce missing notice
     */
    public function woocommerce_missing_notice() {
        ?>
        <div class="notice notice-error">
            <p><?php _e('BebanStore requires WooCommerce to be installed and activated.', 'bebanstore'); ?></p>
        </div>
        <?php
    }

    /**
     * Initialize features
     */
    private function init_features() {
        $feature_classes = array(
            'ajax_add_to_cart' => 'BebanStore_Ajax_Add_To_Cart',
            'cart_confirmation' => 'BebanStore_Cart_Confirmation',
            'custom_quantity_control' => 'BebanStore_Quantity_Control',
            'sales_progress' => 'BebanStore_Sales_Progress',
            'variable_price' => 'BebanStore_Variable_Price',
            'countdown' => 'BebanStore_Countdown',
            'express_shipping' => 'BebanStore_Express_Shipping',
            'tracking_code' => 'BebanStore_Tracking_Code',
            'product_attributes' => 'BebanStore_Product_Attributes',
            'product_discount' => 'BebanStore_Product_Discount',
            'discount_badges' => 'BebanStore_Discount_Badges',
            'order_count' => 'BebanStore_Order_Count',
            'order_view' => 'BebanStore_Order_View',
            'checkout_fields_manager' => 'BebanStore_Checkout_Fields_Manager',
            'your_savings' => 'BebanStore_Your_Savings',
            'wishlist' => 'BebanStore_Wishlist',
            'custom_font' => 'BebanStore_Custom_Font',
            'add_to_cart_button_text' => 'BebanStore_Add_To_Cart_Button_Text'
        );

        // Allow Pro features to register via hooks
        $pro_feature_classes = apply_filters('bebanstore_pro_feature_classes', array());
        $feature_classes = array_merge($feature_classes, $pro_feature_classes);

        // Initialize features
        foreach ($feature_classes as $feature_id => $class_name) {
            if (class_exists($class_name)) {
                $this->features[$feature_id] = new $class_name();
            }
        }

        // Initialize Pro features separately
        foreach ($pro_feature_classes as $feature_id => $class_name) {
            if (class_exists($class_name)) {
                $this->features[$feature_id] = new $class_name();
            }
        }

        // Fire hook after all features are initialized
        do_action('bebanstore_features_initialized', $this->features);
    }

    /**
     * Initialize a specific feature
     * 
     * @param string $feature_id The feature ID to initialize
     * @return bool Whether the feature was successfully initialized
     */
    public function init_feature($feature_id) {
        if (!isset($this->features[$feature_id])) {
            return false;
        }

        $feature = $this->features[$feature_id];
        if (!$feature->is_enabled()) {
            return false;
        }

        return true;
    }

    /**
     * Check plugin dependencies
     * 
     * @return bool Whether all dependencies are satisfied
     */
    private function check_dependencies() {
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', array($this, 'woocommerce_missing_notice'));
            return false;
        }
        return true;
    }

    /**
     * Get all registered features
     * 
     * @return array All registered features
     */
    public function get_features() {
        return $this->features;
    }

    /**
     * Check if Pro features are available
     * 
     * @return bool Whether Pro features are available
     */
    public function is_pro_available() {
        return apply_filters('bebanstore_pro_available', false);
    }
    
    /**
     * Output custom CSS based on location setting
     */
    public function output_custom_css() {
        $options = get_option('bebanstore_options', array());
        $custom_css = isset($options['custom_css']) ? trim($options['custom_css']) : '';
        $css_location = isset($options['css_location']) ? $options['css_location'] : 'inline_head';
        
        if (empty($custom_css)) {
            return;
        }
        
        // Check current action to determine where we are
        $current_action = current_action();
        
        switch ($css_location) {
            case 'inline_head':
                if ($current_action === 'wp_head') {
                    $this->output_css_inline($custom_css, 'head');
                }
                break;
            case 'inline_body':
                if ($current_action === 'wp_footer') {
                    $this->output_css_inline($custom_css, 'body');
                }
                break;
            case 'file_head':
                if ($current_action === 'wp_head') {
                    $this->output_css_as_file($custom_css, 'head');
                }
                break;
            case 'file_body':
                if ($current_action === 'wp_footer') {
                    $this->output_css_as_file($custom_css, 'body');
                }
                break;
        }
    }
    
    /**
     * Output CSS inline (head or body)
     */
    private function output_css_inline($css, $location) {
        $location_text = ($location === 'head') ? 'Head' : 'Body';
        echo "\n<!-- BebanStore Custom CSS (Inline - $location_text) -->\n";
        echo '<style id="bebanstore-custom-css" type="text/css">' . "\n";
        echo wp_strip_all_tags($css);
        echo "\n</style>\n";
    }
    
    /**
     * Output CSS as separate file (head or body)
     */
    private function output_css_as_file($css, $location) {
        $upload_dir = wp_upload_dir();
        $css_file_path = $upload_dir['basedir'] . '/bebanstore-custom.css';
        $css_file_url = $upload_dir['baseurl'] . '/bebanstore-custom.css';
        
        // Write CSS to file
        file_put_contents($css_file_path, wp_strip_all_tags($css));
        
        // Output link tag directly (not using wp_enqueue_style)
        $file_time = filemtime($css_file_path);
        $location_text = ($location === 'head') ? 'Head' : 'Body';
        echo "\n<!-- BebanStore Custom CSS (File - $location_text) -->\n";
        echo '<link rel="stylesheet" id="bebanstore-custom-css" href="' . esc_url($css_file_url . '?v=' . $file_time) . '" type="text/css" media="all">' . "\n";
    }
    
}

// Initialize plugin
new BebanStore();

