<?php
/**
 * Add to Cart Button Text Management Feature
 * 
 * @package BebanStore
 * @since 1.0.6
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Add to Cart Button Text Management Class
 */
class BebanStore_Add_To_Cart_Button_Text extends BebanStore_Feature {
    
    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct(
            'add_to_cart_button_text',
            __('مدیریت متن دکمه افزودن به سبد', 'bebanstore'),
            __('تغییر متن دکمه افزودن به سبد خرید', 'bebanstore')
        );

        // Add hooks if feature is enabled
        if ($this->is_enabled()) {
            $this->init_hooks();
        }
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Add filter for button text
        add_filter('woocommerce_product_add_to_cart_text', array($this, 'change_button_text'), 10, 2);
        add_filter('woocommerce_product_single_add_to_cart_text', array($this, 'change_button_text'), 10, 2);
        add_filter('woocommerce_loop_add_to_cart_link', array($this, 'change_loop_add_to_cart_link'), 10, 3);
    }

    /**
     * Get feature settings
     */
    public function get_settings() {
        $options = get_option('bebanstore_options', array());
        return isset($options['feature_settings']['add_to_cart_button_text']) 
            ? $options['feature_settings']['add_to_cart_button_text'] 
            : array();
    }
    
    /**
     * Check if current user has purchased a product
     *
     * @param int $product_id Product ID
     * @return bool True if user has purchased, false otherwise
     */
    private function has_user_purchased_product($product_id) {
        if (!is_user_logged_in()) {
            return false;
        }
        
        $user_id = get_current_user_id();
        if (!$user_id) {
            return false;
        }
        
        // Check using WooCommerce function first
        if (function_exists('wc_customer_bought_product')) {
            $bought = wc_customer_bought_product('', $user_id, $product_id);
            if ($bought) {
                return true;
            }
        }
        
        // Additional check: Query orders directly for more reliability
        $customer_orders = wc_get_orders(array(
            'customer_id' => $user_id,
            'status' => array('wc-completed', 'wc-processing'),
            'limit' => -1,
            'return' => 'ids',
        ));
        
        if (empty($customer_orders)) {
            return false;
        }
        
        // Check if product exists in any order
        foreach ($customer_orders as $order_id) {
            $order = wc_get_order($order_id);
            if (!$order) {
                continue;
            }
            
            foreach ($order->get_items() as $item) {
                $item_product_id = $item->get_product_id();
                $item_variation_id = $item->get_variation_id();
                
                // Check if this is the product or a variation of it
                if ($item_product_id == $product_id) {
                    return true;
                }
                
                // For variable products, check parent product
                if ($item_variation_id > 0) {
                    $variation = wc_get_product($item_variation_id);
                    if ($variation && $variation->get_parent_id() == $product_id) {
                        return true;
                    }
                }
            }
        }
        
        return false;
    }
    
    /**
     * Check if product is in cart
     *
     * @param int $product_id Product ID
     * @return bool True if product is in cart, false otherwise
     */
    private function is_product_in_cart($product_id) {
        if (!function_exists('WC') || !WC()->cart) {
            return false;
        }
        
        $cart = WC()->cart;
        if (!$cart || $cart->is_empty()) {
            return false;
        }
        
        // Check if product exists in cart
        foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
            $cart_product_id = $cart_item['product_id'];
            $cart_variation_id = isset($cart_item['variation_id']) ? $cart_item['variation_id'] : 0;
            
            // Check direct product match
            if ($cart_product_id == $product_id) {
                return true;
            }
            
            // For variable products, check parent
            if ($cart_variation_id > 0) {
                $variation = wc_get_product($cart_variation_id);
                if ($variation && $variation->get_parent_id() == $product_id) {
                    return true;
                }
            }
        }
        
        return false;
    }

    /**
     * Change the add to cart button text
     *
     * @param string $text Original button text
     * @param WC_Product $product Product object
     * @return string Modified button text
     */
    public function change_button_text($text, $product) {
        if (!$product) {
            return $text;
        }

        $settings = $this->get_settings();
        $product_id = $product->get_id();
        
        // Priority 1: Check if current user has purchased this product
        $has_purchased = $this->has_user_purchased_product($product_id);
        if ($has_purchased) {
            $purchased_text = isset($settings['purchased_button_text']) ? sanitize_text_field($settings['purchased_button_text']) : '';
            if (!empty($purchased_text)) {
                return $purchased_text;
            }
        }
        
        // Priority 2: Check if product is in cart
        $in_cart = $this->is_product_in_cart($product_id);
        if ($in_cart) {
            $in_cart_text = isset($settings['in_cart_button_text']) ? sanitize_text_field($settings['in_cart_button_text']) : '';
            if (!empty($in_cart_text)) {
                return $in_cart_text;
            }
        }
        
        // Priority 3: Use the regular button text
        $custom_text = isset($settings['button_text']) ? sanitize_text_field($settings['button_text']) : '';

        if (!empty($custom_text)) {
            return $custom_text;
        }

        return $text;
    }
    
    /**
     * Change the loop add to cart link text
     *
     * @param string $html The add to cart link HTML
     * @param WC_Product $product Product object
     * @param array $args Arguments for the link
     * @return string Modified HTML
     */
    public function change_loop_add_to_cart_link($html, $product, $args) {
        if (!$product) {
            return $html;
        }

        $settings = $this->get_settings();
        $product_id = $product->get_id();
        
        $custom_text = '';
        
        // Priority 1: Check if current user has purchased this product
        $has_purchased = $this->has_user_purchased_product($product_id);
        if ($has_purchased) {
            $purchased_text = isset($settings['purchased_button_text']) ? sanitize_text_field($settings['purchased_button_text']) : '';
            if (!empty($purchased_text)) {
                $custom_text = $purchased_text;
            }
        }
        
        // Priority 2: Check if product is in cart
        if (empty($custom_text)) {
            $in_cart = $this->is_product_in_cart($product_id);
            if ($in_cart) {
                $in_cart_text = isset($settings['in_cart_button_text']) ? sanitize_text_field($settings['in_cart_button_text']) : '';
                if (!empty($in_cart_text)) {
                    $custom_text = $in_cart_text;
                }
            }
        }
        
        // Priority 3: Use regular button text
        if (empty($custom_text)) {
            $custom_text = isset($settings['button_text']) ? sanitize_text_field($settings['button_text']) : '';
        }
        
        if (!empty($custom_text)) {
            $args['text'] = $custom_text;
            $html = sprintf(
                '<a href="%s" data-quantity="%s" class="%s" %s>%s</a>',
                esc_url($product->add_to_cart_url()),
                esc_attr(isset($args['quantity']) ? $args['quantity'] : 1),
                esc_attr(isset($args['class']) ? $args['class'] : 'button'),
                isset($args['attributes']) ? wc_implode_html_attributes($args['attributes']) : '',
                esc_html($args['text'])
            );
        }

        return $html;
    }

    /**
     * Cleanup feature resources
     */
    public function cleanup() {
        remove_filter('woocommerce_product_add_to_cart_text', array($this, 'change_button_text'), 10);
        remove_filter('woocommerce_product_single_add_to_cart_text', array($this, 'change_button_text'), 10);
        remove_filter('woocommerce_loop_add_to_cart_link', array($this, 'change_loop_add_to_cart_link'), 10);
    }
}

