<?php
/**
 * Ajax Add to Cart Feature
 *
 * Adds Ajax functionality to the "Add to Cart" button on single product pages.
 * This feature enhances the user experience by allowing products to be added to cart
 * without page reload.
 *
 * @package BebanStore
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Ajax Add to Cart Feature Class
 */
class BebanStore_Ajax_Add_To_Cart extends BebanStore_Feature {
    /**
     * Initialize the feature
     */
    public function __construct() {
        parent::__construct(
            'ajax_add_to_cart',
            __('افزودن به سبد خرید با Ajax', 'bebanstore'),
            __('افزودن محصول به سبد خرید بدون بارگذاری مجدد صفحه', 'bebanstore')
        );

        // Add hooks if feature is enabled
        if ($this->is_enabled()) {
            $this->init_hooks();
        }

        add_action('wp_ajax_bebanstore_add_to_cart', array($this, 'ajax_add_to_cart'));
        add_action('wp_ajax_nopriv_bebanstore_add_to_cart', array($this, 'ajax_add_to_cart'));
        add_action('wp_footer', array($this, 'render_cart_popup'));
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Add styles and scripts
        add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    /**
     * Enqueue required assets
     */
    public function enqueue_assets() {
        // Don't load on cart and checkout pages to avoid conflicts with WooCommerce's own Ajax
        if (is_cart() || is_checkout()) {
            return;
        }

        // Get animation type setting from options
        $options = get_option('bebanstore_options', array());
        $feature_options = isset($options['feature_settings']) ? $options['feature_settings'] : array();
        $animation_type = isset($feature_options['ajax_add_to_cart']['animation_type']) 
            ? $feature_options['ajax_add_to_cart']['animation_type'] 
            : 'none';

        // Enqueue CSS
        wp_enqueue_style(
            'bebanstore-ajax-add-to-cart',
            BEBANSTORE_PLUGIN_URL . 'assets/css/ajax-add-to-cart.css',
            array(),
            BEBANSTORE_VERSION
        );

        // Always enqueue WooCommerce scripts first
        wp_enqueue_script('wc-add-to-cart');
        wp_enqueue_script('wc-cart-fragments');

        // Ensure wc_cart_params is available
        if (!wp_script_is('wc-cart-fragments', 'done')) {
            wp_localize_script('wc-cart-fragments', 'wc_cart_params', array(
                'ajax_url' => WC()->ajax_url(),
                'fragment_name' => apply_filters('woocommerce_cart_fragment_name', 'wc_fragments_'),
                'cart_hash_key' => apply_filters('woocommerce_cart_hash_key', 'wc_cart_hash')
            ));
        }

        // Enqueue our script after WooCommerce scripts
        wp_enqueue_script(
            'bebanstore-ajax-add-to-cart',
            BEBANSTORE_PLUGIN_URL . 'assets/js/ajax-add-to-cart.js',
            array('jquery', 'wc-add-to-cart', 'wc-cart-fragments'),
            BEBANSTORE_VERSION,
            true
        );

        // Localize our script
        wp_localize_script(
            'bebanstore-ajax-add-to-cart',
            'bebanstore_ajax_add_to_cart',
            array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('bebanstore-ajax-add-to-cart'),
                'animation_type' => $animation_type,
                'success_message_type' => isset($feature_options['ajax_add_to_cart']['success_message_type']) 
                    ? $feature_options['ajax_add_to_cart']['success_message_type'] 
                    : 'no_message',
                'loader_overlay_color' => isset($feature_options['ajax_add_to_cart']['loader_overlay_color']) 
                    ? $feature_options['ajax_add_to_cart']['loader_overlay_color'] 
                    : '#EF4056',
                'loader_animation_color' => isset($feature_options['ajax_add_to_cart']['loader_animation_color']) 
                    ? $feature_options['ajax_add_to_cart']['loader_animation_color'] 
                    : '#ffffff',
                'i18n' => array(
                    'adding' => __('در حال افزودن...', 'bebanstore'),
                    'added' => __('افزودن به سبد خرید', 'bebanstore')
                )
            )
        );
    }

    /**
     * Cleanup feature resources
     */
    public function cleanup() {
        remove_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
    }

    public function ajax_add_to_cart() {
        if (!isset($_POST['product_id']) || !is_numeric($_POST['product_id'])) {
            wp_send_json_error(['message' => 'شناسه محصول نامعتبر']);
        }

        $product_id = intval($_POST['product_id']);
        $quantity = isset($_POST['quantity']) ? intval($_POST['quantity']) : 1;
        $variation_id = isset($_POST['variation_id']) ? intval($_POST['variation_id']) : 0;
        $variation = [];

        if ($variation_id) {
            foreach ($_POST as $key => $value) {
                if (strpos($key, 'attribute_') === 0) {
                    $variation[$key] = sanitize_text_field($value);
                }
            }
        }

        $product = wc_get_product($product_id);
        if (!$product) {
            wp_send_json_error(['message' => 'محصول یافت نشد']);
        }

        // Check stock availability before adding to cart
        $product_to_check = $variation_id ? wc_get_product($variation_id) : $product;
        $stock_quantity = $product_to_check->get_stock_quantity();
        $max_quantity = $product_to_check->get_max_purchase_quantity();
        
        // Check if product is in stock
        if (!$product_to_check->is_in_stock()) {
            wp_send_json_error(['message' => 'این محصول موجود نیست']);
        }
        
        // Check if we can add the requested quantity
        // Only check if stock quantity is set and greater than 0
        if ($stock_quantity !== null && $stock_quantity > 0 && $quantity > $stock_quantity) {
            wp_send_json_error(['message' => 'حداکثر تعداد مجاز این محصول ' . $stock_quantity . ' عدد است']);
        }
        
        // Check max purchase quantity
        if ($max_quantity && $max_quantity > 0 && $quantity > $max_quantity) {
            wp_send_json_error(['message' => 'حداکثر تعداد مجاز این محصول ' . $max_quantity . ' عدد است']);
        }
        
        // Check if quantity is valid (should be at least 1)
        if ($quantity < 1) {
            wp_send_json_error(['message' => 'تعداد محصول باید حداقل 1 عدد باشد']);
        }
        
        $cart_item_key = WC()->cart->add_to_cart($product_id, $quantity, $variation_id, $variation);
        
        if ($cart_item_key) {
            $image = wp_get_attachment_image_src(get_post_thumbnail_id($product_id), 'thumbnail');
            $image_url = $image ? $image[0] : wc_placeholder_img_src();
            $price = $variation_id ? wc_get_product($variation_id)->get_price_html() : $product->get_price_html();
            
            wp_send_json_success([
                'image' => $image_url,
                'title' => $product->get_name(),
                'price' => $price,
                'message' => 'این کالا به سبد خرید اضافه شد',
                'cart_count' => WC()->cart->get_cart_contents_count()
            ]);
        } else {
            // Get WooCommerce notices to provide more specific error messages
            $notices = wc_get_notices('error');
            $error_message = 'خطا در افزودن به سبد خرید';
            
            if (!empty($notices)) {
                $notice = $notices[0]['notice'];
                
                // Check if it's a stock quantity error and extract the number
                if (preg_match('/(\d+)\s+موجودی\s+در\s+انبار/', $notice, $matches)) {
                    $stock_quantity = $matches[1];
                    // Only show stock limit if stock quantity is greater than 0
                    if ($stock_quantity > 0) {
                        $error_message = 'حداکثر تعداد مجاز این محصول ' . $stock_quantity . ' عدد است';
                    }
                } elseif (preg_match('/(\d+)\s+موجودی/', $notice, $matches)) {
                    $stock_quantity = $matches[1];
                    // Only show stock limit if stock quantity is greater than 0
                    if ($stock_quantity > 0) {
                        $error_message = 'حداکثر تعداد مجاز این محصول ' . $stock_quantity . ' عدد است';
                    }
                } else {
                    $error_message = $notice;
                }
            }
            
            wp_send_json_error(['message' => $error_message]);
        }
    }

    public function render_cart_popup() {
        // Don't render popup on cart and checkout pages
        if (is_cart() || is_checkout()) {
            return;
        }
        ?>
        <div id="bebanstoreCartPopup" class="bebanstore-cart-popup" style="display: none;">
            <div class="bebanstore-cart-popup-content">
                <button class="bebanstore-cart-popup-close">
                    <svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M15 15L1 1" stroke="#3C3D45" stroke-width="1.5" stroke-linecap="round"></path>
                        <path d="M1 15L15 1" stroke="#3C3D45" stroke-width="1.5" stroke-linecap="round"></path>
                    </svg>
                </button>
                <h2 class="bebanstore-cart-popup-title">این کالا به سبد خرید اضافه شد</h2>
                <hr class="bebanstore-cart-popup-divider">
                <div class="bebanstore-cart-popup-product">
                    <img id="bebanstorePopupProductImage" src="" alt="محصول" class="bebanstore-cart-popup-image">
                    <div class="bebanstore-cart-popup-details">
                        <h3 id="bebanstorePopupProductTitle" class="bebanstore-cart-popup-product-title"></h3>
                        <p id="bebanstorePopupProductPrice" class="bebanstore-cart-popup-product-price"></p>
                        <div id="bebanstorePopupProductVariation" class="bebanstore-cart-popup-product-variations"></div>
                    </div>
                </div>
                <a href="<?php echo esc_url(wc_get_cart_url()); ?>" class="bebanstore-cart-popup-button">برو به سبد خرید</a>
            </div>
        </div>
        <?php
    }
}