<?php

/**
 * Cart Confirmation Feature
 * 
 * Handles the cart item removal confirmation modal and AJAX functionality
 */
class BebanStore_Cart_Confirmation extends BebanStore_Feature {
    /**
     * Initialize the feature
     */
    public function __construct() {
        parent::__construct(
            'cart_confirmation',
            __('پاپ آپ حذف محصول', 'bebanstore'),
            __('نمایش پنجره تایید قبل از حذف محصول از سبد خرید', 'bebanstore')
        );

        // Register assets
        $this->css_files = array(
            'cart-confirmation' => 'cart-confirmation.css'
        );
        $this->js_files = array(
            'cart-confirmation' => 'cart-confirmation.js'
        );
    }

    /**
     * Register hooks
     */
    protected function register_hooks() {
        // Add AJAX handlers
        add_action('wp_ajax_bebanstore_remove_cart_item', array($this, 'ajax_remove_cart_item'));
        add_action('wp_ajax_nopriv_bebanstore_remove_cart_item', array($this, 'ajax_remove_cart_item'));

        // Add modal template to footer
        add_action('wp_footer', array($this, 'render_modal_template'));

        // Register and enqueue assets
        add_action('wp_enqueue_scripts', array($this, 'enqueue_feature_assets'));
        add_action('wp_enqueue_scripts', array($this, 'localize_script'), 20);
    }

    /**
     * Enqueue feature assets
     */
    public function enqueue_feature_assets() {
        // Enqueue CSS
        wp_enqueue_style(
            'bebanstore-cart-confirmation',
            BEBANSTORE_PLUGIN_URL . 'assets/css/cart-confirmation.css',
            array(),
            BEBANSTORE_VERSION
        );

        // Enqueue WooCommerce scripts first
        wp_enqueue_script('wc-add-to-cart');
        wp_enqueue_script('wc-cart-fragments');

        // Ensure wc_cart_params is available
        if (!wp_script_is('wc-cart-fragments', 'done')) {
            wp_localize_script('wc-cart-fragments', 'wc_cart_params', array(
                'ajax_url' => WC()->ajax_url(),
                'fragment_name' => apply_filters('woocommerce_cart_fragment_name', 'wc_fragments_'),
                'cart_hash_key' => apply_filters('woocommerce_cart_hash_key', 'wc_cart_hash')
            ));
        }

        // Enqueue JS
        wp_enqueue_script(
            'bebanstore-cart-confirmation',
            BEBANSTORE_PLUGIN_URL . 'assets/js/cart-confirmation.js',
            array('jquery', 'wc-add-to-cart', 'wc-cart-fragments'),
            BEBANSTORE_VERSION,
            true
        );
    }

    /**
     * Get feature settings
     */
    public function get_settings() {
        $options = get_option('bebanstore_options', array());
        return isset($options['feature_settings']['cart_confirmation']) ? $options['feature_settings']['cart_confirmation'] : array();
    }

    /**
     * Localize script with translations and settings
     */
    public function localize_script() {
        $settings = $this->get_settings();
        
        wp_localize_script('bebanstore-cart-confirmation', 'bebanstoreCart', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('bebanstore_cart_nonce'),
            'confirmText' => isset($settings['confirmation_message']) ? $settings['confirmation_message'] : 'آیا مطمئن هستید که می‌خواهید این محصول را از سبد خرید حذف کنید؟',
            'confirmButton' => isset($settings['confirm_button_text']) ? $settings['confirm_button_text'] : 'بله، حذف شود',
            'cancelButton' => isset($settings['cancel_button_text']) ? $settings['cancel_button_text'] : 'انصراف',
            'errorMessage' => __('خطا در حذف محصول. لطفا مجددا تلاش کنید.', 'bebanstore'),
            'shopUrl' => wc_get_page_permalink('shop')
        ));
    }

    /**
     * Render modal template
     */
    public function render_modal_template() {
        $settings = $this->get_settings();
        
        // Get color settings with defaults
        $modal_bg_color = isset($settings['modal_background_color']) ? $settings['modal_background_color'] : '#ffffff';
        $confirm_bg_color = isset($settings['confirm_button_bg_color']) ? $settings['confirm_button_bg_color'] : '#dc3545';
        $confirm_text_color = isset($settings['confirm_button_text_color']) ? $settings['confirm_button_text_color'] : '#ffffff';
        $cancel_bg_color = isset($settings['cancel_button_bg_color']) ? $settings['cancel_button_bg_color'] : '#ffffff';
        $cancel_text_color = isset($settings['cancel_button_text_color']) ? $settings['cancel_button_text_color'] : '#007bff';
        $modal_title_color = isset($settings['modal_title_color']) ? $settings['modal_title_color'] : '#000000';
        $modal_description_color = isset($settings['modal_description_color']) ? $settings['modal_description_color'] : '#666666';
        
        ?>
        <div id="bebanstore-cart-confirmation" class="bebanstore-modal">
            <div class="bebanstore-modal-overlay"></div>
            <div class="bebanstore-modal-content" style="background-color: <?php echo esc_attr($modal_bg_color); ?>;">
                <div class="bebanstore-modal-header">
                    <h3 style="color: <?php echo esc_attr($modal_title_color); ?> !important;"><?php echo esc_html(isset($settings['modal_title']) ? $settings['modal_title'] : 'تایید حذف محصول'); ?></h3>
                </div>
                <div class="bebanstore-modal-body">
                    <p class="bebanstore-confirm-message" style="color: <?php echo esc_attr($modal_description_color); ?> !important;"></p>
                </div>
                <div class="bebanstore-modal-footer">
                    <button type="button" class="bebanstore-modal-confirm" style="background-color: <?php echo esc_attr($confirm_bg_color); ?> !important; color: <?php echo esc_attr($confirm_text_color); ?> !important;">
                        <?php echo esc_html(isset($settings['confirm_button_text']) ? $settings['confirm_button_text'] : 'بله، حذف شود'); ?>
                    </button>
                    <button type="button" class="bebanstore-modal-cancel" style="background-color: <?php echo esc_attr($cancel_bg_color); ?> !important; color: <?php echo esc_attr($cancel_text_color); ?> !important;">
                        <?php echo esc_html(isset($settings['cancel_button_text']) ? $settings['cancel_button_text'] : 'انصراف'); ?>
                    </button>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Handle AJAX remove cart item
     */
    public function ajax_remove_cart_item() {
        check_ajax_referer('bebanstore_cart_nonce', 'nonce');

        $cart_item_key = isset($_POST['cart_item_key']) ? sanitize_text_field($_POST['cart_item_key']) : '';

        if (empty($cart_item_key)) {
            wp_send_json_error(array(
                'message' => __('شناسه محصول نامعتبر است.', 'bebanstore')
            ));
        }

        // Remove cart item
        WC()->cart->remove_cart_item($cart_item_key);

        // Return updated cart fragments
        WC_AJAX::get_refreshed_fragments();
    }

    /**
     * Get feature title
     */
    public function get_title() {
        return $this->feature_name;
    }

    /**
     * Cleanup feature resources
     */
    public function cleanup() {
        remove_action('wp_ajax_bebanstore_remove_cart_item', array($this, 'ajax_remove_cart_item'));
        remove_action('wp_ajax_nopriv_bebanstore_remove_cart_item', array($this, 'ajax_remove_cart_item'));
        remove_action('wp_footer', array($this, 'render_modal_template'));
        remove_action('wp_enqueue_scripts', array($this, 'enqueue_feature_assets'));
        remove_action('wp_enqueue_scripts', array($this, 'localize_script'));
    }
} 