<?php
if (!defined('ABSPATH')) {
    exit;
}

class BebanStore_Checkout_Fields_Manager extends BebanStore_Feature {
    /**
     * Fields to manage
     * 
     * @var array
     */
    private $managed_fields = array(
        'company',
        'country',
        'state',
        'city',
        'address_1',
        'address_2',
        'postcode'
    );

    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct(
            'checkout_fields_manager',
            __('مدیریت فیلدهای صفحه تسویه حساب', 'bebanstore'),
            __('امکان مدیریت و سفارشی‌سازی فیلدهای صفحه تسویه حساب ووکامرس', 'bebanstore')
        );
    }

    /**
     * Register hooks
     */
    protected function register_hooks() {
        // Hook into WooCommerce checkout fields
        add_filter('woocommerce_checkout_fields', array($this, 'apply'), 999);
        
        // Hook into billing fields
        add_filter('woocommerce_billing_fields', array($this, 'apply_billing_fields'), 999);
        
        // Hook into shipping fields
        add_filter('woocommerce_shipping_fields', array($this, 'apply_shipping_fields'), 999);
        
        // Hook into address fields
        add_filter('woocommerce_default_address_fields', array($this, 'apply_address_fields'), 999);
        
        // Hook into checkout validation
        add_action('woocommerce_checkout_process', array($this, 'validate_checkout_fields'));
    }

    /**
     * Get feature settings
     * 
     * @return array
     */
    public function get_settings() {
        $options = get_option('bebanstore_options', array());
        return isset($options['feature_settings']['checkout_fields_manager']) 
            ? $options['feature_settings']['checkout_fields_manager'] 
            : array();
    }

    /**
     * Apply settings to checkout fields
     * 
     * @param array $fields
     * @return array
     */
    public function apply($fields) {
        if (!is_array($fields)) {
            return $fields;
        }

        $settings = $this->get_settings();

        foreach ($this->managed_fields as $field) {
            if (!isset($settings[$field]) || $settings[$field] === 'default') {
                continue; // Skip if setting is 'default' (no change)
            }

            $setting = $settings[$field];

            // Apply to billing fields
            if (isset($fields['billing'][$field])) {
                if ($setting === 'hidden') {
                    unset($fields['billing'][$field]);
                } else {
                    $fields['billing'][$field] = $this->modify_field($fields['billing'][$field], $setting);
                }
            }

            // Apply to shipping fields
            if (isset($fields['shipping'][$field])) {
                if ($setting === 'hidden') {
                    unset($fields['shipping'][$field]);
                } else {
                    $fields['shipping'][$field] = $this->modify_field($fields['shipping'][$field], $setting);
                }
            }
        }

        return $fields;
    }

    /**
     * Apply settings to billing fields
     * 
     * @param array $fields
     * @return array
     */
    public function apply_billing_fields($fields) {
        return $this->apply(array('billing' => $fields))['billing'];
    }

    /**
     * Apply settings to shipping fields
     * 
     * @param array $fields
     * @return array
     */
    public function apply_shipping_fields($fields) {
        return $this->apply(array('shipping' => $fields))['shipping'];
    }

    /**
     * Apply settings to address fields
     * 
     * @param array $fields
     * @return array
     */
    public function apply_address_fields($fields) {
        $settings = $this->get_settings();
        
        foreach ($this->managed_fields as $field) {
            if (isset($settings[$field]) && isset($fields[$field])) {
                if ($settings[$field] === 'default') {
                    continue; // Skip if setting is 'default' (no change)
                }
                
                if ($settings[$field] === 'hidden') {
                    unset($fields[$field]);
                } else {
                    $fields[$field] = $this->modify_field($fields[$field], $settings[$field]);
                }
            }
        }
        
        return $fields;
    }

    /**
     * Modify field based on setting
     * 
     * @param array $field
     * @param string $setting
     * @return array
     */
    private function modify_field($field, $setting) {
        switch ($setting) {
            case 'optional':
                $field['required'] = false;
                $field['label_class'] = array('optional');
                return $field;
            
            case 'required':
                $field['required'] = true;
                $field['label_class'] = array('required');
                return $field;
            
            default:
                return $field;
        }
    }

    /**
     * Validate checkout fields
     */
    public function validate_checkout_fields() {
        $settings = $this->get_settings();
        
        // Get checkout fields using the correct WooCommerce method
        $checkout_fields = WC()->checkout->get_checkout_fields();
        
        foreach ($this->managed_fields as $field) {
            if (!isset($settings[$field]) || $settings[$field] === 'default' || $settings[$field] !== 'required') {
                continue;
            }

            // Check if billing fields exist and are required
            if (isset($checkout_fields['billing']) && isset($checkout_fields['billing'][$field])) {
                if (empty($_POST['billing_' . $field])) {
                    $field_label = $checkout_fields['billing'][$field]['label'] ?? $field;
                    wc_add_notice(
                        sprintf(__('لطفا فیلد %s را در بخش اطلاعات صورتحساب پر کنید.', 'bebanstore'), $field_label),
                        'error'
                    );
                }
            }

            // Check if shipping fields exist and are required
            if (isset($checkout_fields['shipping']) && isset($checkout_fields['shipping'][$field])) {
                // Only validate shipping if shipping is needed
                if (WC()->cart->needs_shipping()) {
                    if (empty($_POST['shipping_' . $field])) {
                        $field_label = $checkout_fields['shipping'][$field]['label'] ?? $field;
                        wc_add_notice(
                            sprintf(__('لطفا فیلد %s را در بخش اطلاعات ارسال پر کنید.', 'bebanstore'), $field_label),
                            'error'
                        );
                    }
                }
            }
        }
    }
} 