<?php
/**
 * Product Discount Countdown Feature Class
 */
class BebanStore_Countdown extends BebanStore_Shortcode_Feature {
    /**
     * Class constructor
     */
    public function __construct() {
        parent::__construct(
            'countdown',
            __('شمارنده معکوس تخفیف', 'bebanstore'),
            __('نمایش شمارنده معکوس برای تخفیف محصولات', 'bebanstore'),
            'bebanstore_countdown',
            __('برای استفاده از این ویژگی شورت‌کد را کپی کنید و در سیستم حلقه المنتور رو باکس محصولات یا در صفحه محصول هر جا که دوست دارید قرار بدید تا شمارنده معکوس تخفیف نمایش داده شود.', 'bebanstore')
        );

        // Add styles and scripts
        if ($this->is_enabled()) {
            add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
        }
    }

    /**
     * Get feature settings
     */
    public function get_settings() {
        $options = get_option('bebanstore_options', array());
        $settings = isset($options['feature_settings']['countdown']) ? $options['feature_settings']['countdown'] : array();
        
        // Merge with defaults
        $defaults = array(
            'countdown_color' => '#000000',
            'number_font_size' => '24px',
            'separator_color' => '#000000',
            'separator_font_size' => '16px',
            'label_color' => '#666666',
            'label_font_size' => '12px',
            'box_padding' => '8px',
            'label_gap' => '8px',
            'card_background_color' => '#ffffff',
            'card_border_color' => '#ededed',
            'card_border_width' => '1px',
            'card_border_radius' => '8px',
            'card_gap' => '4px',
            'variable_countdown_mode' => 'default_variation',
            'display_style' => 'card'
        );
        
        $settings = array_merge($defaults, $settings);
        
        // تعیین نمایش برچسب‌ها بر اساس سبک نمایش
        $settings['show_labels'] = ($settings['display_style'] !== 'classic');
        
        return $settings;
    }

    /**
     * Enqueue required assets
     */
    public function enqueue_assets() {
        wp_enqueue_style(
            'bebanstore-countdown',
            BEBANSTORE_PLUGIN_URL . 'assets/css/countdown.css',
            array(),
            BEBANSTORE_VERSION
        );

        wp_enqueue_script(
            'bebanstore-countdown',
            BEBANSTORE_PLUGIN_URL . 'assets/js/countdown.js',
            array('jquery'),
            BEBANSTORE_VERSION,
            true
        );
    }

    /**
     * Get feature title
     */
    public function get_title() {
        return $this->feature_name;
    }

    /**
     * Get feature description
     */
    public function get_description() {
        return $this->feature_description;
    }

    /**
     * Get shortcode
     */
    public function get_shortcode() {
        return '[bebanstore_countdown]';
    }

    /**
     * Render the shortcode
     */
    public function render_shortcode($atts, $content = null) {
        // Get current product
        global $product;
        if (!$product || !is_a($product, 'WC_Product')) {
            return '';
        }

        // Get sale data
        $sale_data = $this->get_sale_data($product);
        if (!$sale_data) {
            return '';
        }
        
        // Format dates for JavaScript
        $end_date = $sale_data['end_date'];
        $end_time = $sale_data['end_time'];
        
        // Convert to ISO 8601 format for JavaScript
        $end_date_iso = date('c', $end_date);
        $end_time_iso = date('c', $end_time);

        // Get settings
        $settings = $this->get_settings();
        
        // Generate unique ID for this instance
        $unique_id = 'bebanstore-countdown-' . uniqid();

        ob_start();
        ?>
        <div class="bebanstore-countdown bebanstore-countdown-<?php echo esc_attr($settings['display_style']); ?>" 
             id="<?php echo esc_attr($unique_id); ?>" 
             data-product-id="<?php echo esc_attr($product->get_id()); ?>"
             data-end-date="<?php echo esc_attr($end_date_iso); ?>"
             data-end-time="<?php echo esc_attr($end_time_iso); ?>"
             data-countdown-color="<?php echo esc_attr($settings['countdown_color']); ?>"
             data-separator-color="<?php echo esc_attr($settings['separator_color']); ?>"
             data-label-color="<?php echo esc_attr($settings['label_color']); ?>"
             data-show-labels="<?php echo esc_attr($settings['show_labels'] ? 'true' : 'false'); ?>"
             data-number-font-size="<?php echo esc_attr($settings['number_font_size']); ?>"
             data-separator-font-size="<?php echo esc_attr($settings['separator_font_size']); ?>"
             data-label-font-size="<?php echo esc_attr($settings['label_font_size']); ?>"
             data-box-padding="<?php echo esc_attr($settings['box_padding']); ?>"
             data-label-gap="<?php echo esc_attr($settings['label_gap']); ?>"
             data-card-background-color="<?php echo esc_attr($settings['card_background_color']); ?>"
             data-card-border-color="<?php echo esc_attr($settings['card_border_color']); ?>"
             data-card-border-width="<?php echo esc_attr($settings['card_border_width']); ?>"
             data-card-border-radius="<?php echo esc_attr($settings['card_border_radius']); ?>"
             data-card-gap="<?php echo esc_attr($settings['card_gap']); ?>"
             data-display-style="<?php echo esc_attr($settings['display_style']); ?>">
            <div class="bebanstore-countdown-timer">
                <?php if ($settings['display_style'] === 'inline'): ?>
                    <!-- Inline style: numbers and labels in one line -->
                    <span class="bebanstore-countdown-days">00</span>
                    <?php if ($settings['show_labels']): ?>
                    <span class="bebanstore-countdown-label bebanstore-countdown-day-label">روز</span>
                    <?php endif; ?>
                    <span class="bebanstore-countdown-separator">:</span>
                    <span class="bebanstore-countdown-hours">00</span>
                    <?php if ($settings['show_labels']): ?>
                    <span class="bebanstore-countdown-label bebanstore-countdown-hour-label">ساعت</span>
                    <?php endif; ?>
                    <span class="bebanstore-countdown-separator">:</span>
                    <span class="bebanstore-countdown-minutes">00</span>
                    <?php if ($settings['show_labels']): ?>
                    <span class="bebanstore-countdown-label bebanstore-countdown-minute-label">دقیقه</span>
                    <?php endif; ?>
                    <span class="bebanstore-countdown-separator">:</span>
                    <span class="bebanstore-countdown-seconds">00</span>
                    <?php if ($settings['show_labels']): ?>
                    <span class="bebanstore-countdown-label bebanstore-countdown-second-label">ثانیه</span>
                    <?php endif; ?>
                <?php else: ?>
                    <!-- Classic and Card styles: separate items -->
                    <div class="bebanstore-countdown-item">
                        <span class="bebanstore-countdown-days">00</span>
                        <?php if ($settings['show_labels']): ?>
                        <span class="bebanstore-countdown-label">روز</span>
                        <?php endif; ?>
                    </div>
                    <div class="bebanstore-countdown-item">
                        <span class="bebanstore-countdown-hours">00</span>
                        <?php if ($settings['show_labels']): ?>
                        <span class="bebanstore-countdown-label">ساعت</span>
                        <?php endif; ?>
                    </div>
                    <div class="bebanstore-countdown-item">
                        <span class="bebanstore-countdown-minutes">00</span>
                        <?php if ($settings['show_labels']): ?>
                        <span class="bebanstore-countdown-label">دقیقه</span>
                        <?php endif; ?>
                    </div>
                    <div class="bebanstore-countdown-item">
                        <span class="bebanstore-countdown-seconds">00</span>
                        <?php if ($settings['show_labels']): ?>
                        <span class="bebanstore-countdown-label">ثانیه</span>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Get sale data for a product
     */
    private function get_sale_data($product) {
        if (!$product->is_on_sale()) {
            return false;
        }

        // Get settings
        $settings = $this->get_settings();
        $variable_mode = $settings['variable_countdown_mode'];

        // Handle variable products
        if ($product->is_type('variable')) {
            return $this->get_variable_product_sale_data($product, $variable_mode);
        }

        // Handle simple products
        $sale_data = array(
            'end_date' => $product->get_date_on_sale_to(),
            'end_time' => $product->get_date_on_sale_to('time')
        );

        if (!$sale_data['end_date']) {
            return false;
        }

        // Convert to timestamp
        $sale_data['end_date'] = strtotime($sale_data['end_date']);
        $sale_data['end_time'] = strtotime($sale_data['end_time']);

        return $sale_data;
    }

    /**
     * Get sale data for variable products based on selected mode
     */
    private function get_variable_product_sale_data($product, $mode) {
        $variations = $product->get_available_variations();
        $sale_times = array();
        $default_variation_id = null;

        // Get default variation ID using WooCommerce method
        $default_attributes = $product->get_default_attributes();
        if (!empty($default_attributes)) {
            foreach ($variations as $variation_data) {
                $variation = wc_get_product($variation_data['variation_id']);
                if ($variation && $this->is_default_variation($variation, $default_attributes)) {
                    $default_variation_id = $variation_data['variation_id'];
                    break;
                }
            }
        }

        // Collect all sale end times from variations
        foreach ($variations as $variation_data) {
            $variation = wc_get_product($variation_data['variation_id']);
            if ($variation && $variation->is_on_sale()) {
                $end_date = $variation->get_date_on_sale_to();
                if ($end_date) {
                    $sale_times[] = array(
                        'variation_id' => $variation_data['variation_id'],
                        'end_date' => strtotime($end_date),
                        'is_default' => ($variation_data['variation_id'] == $default_variation_id)
                    );
                }
            }
        }

        if (empty($sale_times)) {
            return false;
        }

        // Select time based on mode
        $selected_time = null;

        switch ($mode) {
            case 'default_variation':
                // Try to find default variation first
                foreach ($sale_times as $time_data) {
                    if ($time_data['is_default']) {
                        $selected_time = $time_data;
                        break;
                    }
                }
                // If no default variation found, fallback to shortest time
                if (!$selected_time) {
                    $selected_time = $this->get_shortest_time($sale_times);
                }
                break;

            case 'shortest_time':
                $selected_time = $this->get_shortest_time($sale_times);
                break;

            case 'longest_time':
                $selected_time = $this->get_longest_time($sale_times);
                break;

            default:
                $selected_time = $this->get_shortest_time($sale_times);
                break;
        }

        if (!$selected_time) {
            return false;
        }

        return array(
            'end_date' => $selected_time['end_date'],
            'end_time' => $selected_time['end_date']
        );
    }

    /**
     * Check if variation matches default attributes
     */
    private function is_default_variation($variation, $default_attributes) {
        $variation_attributes = $variation->get_attributes();
        
        foreach ($default_attributes as $attribute_name => $default_value) {
            $clean_name = str_replace('attribute_', '', $attribute_name);
            if (!isset($variation_attributes[$clean_name]) || $variation_attributes[$clean_name] !== $default_value) {
                return false;
            }
        }
        
        return true;
    }

    /**
     * Get shortest time from sale times array
     */
    private function get_shortest_time($sale_times) {
        $shortest = null;
        foreach ($sale_times as $time_data) {
            if ($shortest === null || $time_data['end_date'] < $shortest['end_date']) {
                $shortest = $time_data;
            }
        }
        return $shortest;
    }

    /**
     * Get longest time from sale times array
     */
    private function get_longest_time($sale_times) {
        $longest = null;
        foreach ($sale_times as $time_data) {
            if ($longest === null || $time_data['end_date'] > $longest['end_date']) {
                $longest = $time_data;
            }
        }
        return $longest;
    }

    /**
     * Cleanup feature resources
     */
    public function cleanup() {
        remove_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
    }
} 