<?php
/**
 * Discount Badges Management Class
 *
 * @package BebanStore
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Discount Badges Feature
 */
class BebanStore_Discount_Badges extends BebanStore_Shortcode_Feature {
    /**
     * Feature ID
     */
    protected $id = 'discount_badges';

    /**
     * Feature title
     */
    protected $title;

    /**
     * Feature description
     */
    protected $description;

    /**
     * Initialize the feature
     */
    public function __construct() {
        $this->title = __('نشان‌های تخفیف', 'bebanstore');
        $this->description = __('نمایش نشان‌های تخفیف برای محصولات', 'bebanstore');

        parent::__construct(
            $this->id,
            $this->title,
            $this->description,
            'beban_discount_badges',
            __('برای استفاده از این ویژگی شورتکد را کپی کنید و در سیستم حلقه المنتور رو باکس محصولات یا در صفحه محصول هر جا که دوست دارید قرار بدید تا نشان تخفیف نمایش داده شود.', 'bebanstore')
        );

        // Add hooks if feature is enabled
        if ($this->is_enabled()) {
            $this->init_hooks();
        }
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Add styles
        add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    /**
     * Render shortcode
     * 
     * @param array $atts Shortcode attributes
     * @param string|null $content Shortcode content
     * @return string
     */
    public function render_shortcode($atts, $content = null) {
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            return '';
        }

        // Get current product
        global $product;

        // If $product is not a WC_Product object, try to get it
        if (!$product instanceof WC_Product) {
            // If $product is a string or numeric ID, try to get the product
            if (is_string($product) || is_numeric($product)) {
                $product = wc_get_product($product);
            } else {
                // If we're in a product loop, try to get the current product
                $product = wc_get_product(get_the_ID());
            }
        }

        // If we still don't have a valid product, return empty
        if (!$product instanceof WC_Product) {
            return '';
        }
        
        // Check if product has discount
        if (!$product->is_on_sale()) {
            return '';
        }
        
        // Get discount type
        $has_sale_end = $product->get_date_on_sale_to();
        $badge_type = $has_sale_end ? 'special' : 'regular';

        // Get settings
        $options = get_option('bebanstore_options', array());
        $settings = isset($options['feature_settings']['discount_badges']) ? $options['feature_settings']['discount_badges'] : array();

        // Get badge image from settings or use default
        $badge_url = '';
        if ($badge_type === 'special') {
            $badge_url = isset($settings['category_discount_image']) && !empty($settings['category_discount_image']) 
                ? $settings['category_discount_image'] 
                : BEBANSTORE_PLUGIN_URL . 'assets/img/SpecialSell.svg';
        } else {
            $badge_url = isset($settings['product_discount_image']) && !empty($settings['product_discount_image']) 
                ? $settings['product_discount_image'] 
                : BEBANSTORE_PLUGIN_URL . 'assets/img/sell.svg';
        }

        // Return badge HTML
        return sprintf(
            '<div class="bebanstore-discount-badge %s"><img src="%s" alt="%s"></div>',
            esc_attr($badge_type),
            esc_url($badge_url),
            esc_attr($badge_type === 'special' ? 'تخفیف ویژه' : 'تخفیف')
        );
    }

    /**
     * Enqueue required assets
     */
    public function enqueue_assets() {
        wp_enqueue_style(
            'bebanstore-discount-badges',
            BEBANSTORE_PLUGIN_URL . 'assets/css/discount-badges.css',
            array(),
            BEBANSTORE_VERSION
        );
    }

    /**
     * Render show shortcode button
     */
    public function render_show_shortcode_button() {
        ?>
        <button type="button" 
                class="bebanstore-show-shortcode button" 
                data-feature-id="<?php echo esc_attr($this->id); ?>"
                data-feature-title="<?php echo esc_attr($this->title); ?>"
                data-shortcode="[beban_discount_badges]"
                data-description="<?php echo esc_attr($this->usage_instructions); ?>">
            <?php _e('نمایش شورتکد', 'bebanstore'); ?>
        </button>
        <?php
    }

    /**
     * Cleanup feature resources
     */
    public function cleanup() {
        remove_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
    }
} 