<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Express Shipping Feature
 */
class BebanStore_Express_Shipping extends BebanStore_Shortcode_Feature {
    /**
     * Initialize the feature
     */
    public function __construct() {
        parent::__construct(
            'express_shipping',
            __('ارسال فوری', 'bebanstore'),
            __('نمایش نشان ارسال فوری برای محصولات', 'bebanstore'),
            'beban_express_shipping',
            __('برای استفاده از این ویژگی شورتکد را کپی کنید و در سیستم حلقه المنتور رو باکس محصولات یا در صفحه محصول هر جا که دوست دارید قرار بدید تا نشان ارسال فوری نمایش داده شود.', 'bebanstore')
        );

        // Add hooks if feature is enabled
        if ($this->is_enabled()) {
            $this->init_hooks();
        }

        add_action('woocommerce_before_shop_loop_item_title', array($this, 'display_express_shipping_badge'));
        add_action('woocommerce_single_product_summary', array($this, 'display_express_shipping_badge'));
        
        // Add hook for cart display - check inside the method
        add_action('woocommerce_cart_item_name', array($this, 'display_express_shipping_badge_in_cart'), 10, 3);
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Add express shipping checkbox to product admin
        add_action('woocommerce_product_options_general_product_data', [$this, 'add_express_shipping_checkbox']);
        
        // Save express shipping checkbox
        add_action('woocommerce_admin_process_product_object', [$this, 'save_express_shipping_checkbox']);
        
        // Add styles
        add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    /**
     * Enqueue required assets
     */
    public function enqueue_assets() {
        wp_enqueue_style(
            'bebanstore-express-shipping',
            BEBANSTORE_PLUGIN_URL . 'assets/css/express-shipping.css',
            array(),
            BEBANSTORE_VERSION
        );
        
        wp_enqueue_script(
            'bebanstore-express-shipping',
            BEBANSTORE_PLUGIN_URL . 'assets/js/express-shipping.js',
            array('jquery'),
            BEBANSTORE_VERSION,
            true
        );
    }

    /**
     * Add express shipping checkbox to product admin
     */
    public function add_express_shipping_checkbox() {
        echo '<div class="options_group">';
        woocommerce_wp_checkbox([
            'id' => '_beban_express_shipping',
            'label' => 'ارسال فوری',
            'description' => 'فعال کردن ارسال فوری برای این محصول',
            'wrapper_class' => 'beban-express-checkbox',
        ]);
        echo '</div>';
    }

    /**
     * Save express shipping checkbox
     */
    public function save_express_shipping_checkbox($product) {
        $product->update_meta_data('_beban_express_shipping', isset($_POST['_beban_express_shipping']) ? 'yes' : 'no');
    }

    /**
     * Render the shortcode
     */
    public function render_shortcode($atts, $content = null) {
        global $product;
        if (!$product) {
            $product = wc_get_product(get_the_ID());
        }
        if ($product && $product->get_meta('_beban_express_shipping') === 'yes') {
            return $this->get_express_shipping_badge();
        }
        return '';
    }

    /**
     * Get feature settings
     */
    public function get_settings() {
        $options = get_option('bebanstore_options', array());
        return isset($options['feature_settings']['express_shipping']) ? $options['feature_settings']['express_shipping'] : array();
    }

    /**
     * Get express shipping badge HTML
     */
    private function get_express_shipping_badge() {
        $settings = $this->get_settings();
        $badge_text = $settings['express_shipping_badge_text'] ?? 'ارسال فوری';
        $badge_color = $settings['express_shipping_badge_color'] ?? '#4CAF50';
        $badge_text_color = $settings['express_shipping_badge_text_color'] ?? '#ffffff';
        $show_tooltip = isset($settings['express_shipping_show_tooltip']) ? $settings['express_shipping_show_tooltip'] : true;
        $tooltip_text = isset($settings['express_shipping_tooltip_text']) ? $settings['express_shipping_tooltip_text'] : 'ارسال سریع به تهران';
        
        // Create inline styles
        $badge_style = sprintf(
            'background: %s; color: %s !important;',
            esc_attr($badge_color),
            esc_attr($badge_text_color)
        );
        
        $tooltip_attr = '';
        if ($show_tooltip) {
            $tooltip_attr = sprintf(
                ' data-tooltip="%s"',
                esc_attr($tooltip_text)
            );
        }
        
        return sprintf(
            '<span class="beban-express-badge" style="%s"%s>%s</span>',
            esc_attr($badge_style),
            $tooltip_attr,
            esc_html($badge_text)
        );
    }

    /**
     * Display express shipping badge in cart
     */
    public function display_express_shipping_badge_in_cart($name, $cart_item, $cart_item_key) {
        $product = $cart_item['data'];
        
        // Only show in main cart, not mini cart
        if ($this->is_mini_cart()) {
            return $name;
        }
        
        // Check if we should show badge in cart
        if (!$this->should_show_in_cart()) {
            return $name;
        }
        
        // Check if this is a variation product
        $product_to_check = $product;
        if ($product->is_type('variation')) {
            // For variations, check the variation itself first, then fall back to parent
            if ($this->is_express_shipping($product)) {
                $product_to_check = $product;
            } else {
                $product_to_check = wc_get_product($product->get_parent_id());
            }
        }
        
        if ($this->is_express_shipping($product_to_check)) {
            $settings = $this->get_settings();
            $badge_text = isset($settings['express_shipping_badge_text']) ? $settings['express_shipping_badge_text'] : 'ارسال سریع';
            $badge_color = isset($settings['express_shipping_badge_color']) ? $settings['express_shipping_badge_color'] : '#4CAF50';
            $text_color = isset($settings['express_shipping_badge_text_color']) ? $settings['express_shipping_badge_text_color'] : '#ffffff';
            $show_tooltip = isset($settings['express_shipping_show_tooltip']) ? $settings['express_shipping_show_tooltip'] : true;
            $tooltip_text = isset($settings['express_shipping_tooltip_text']) ? $settings['express_shipping_tooltip_text'] : 'ارسال سریع به تهران';
            
            $tooltip_attr = '';
            if ($show_tooltip) {
                $tooltip_attr = sprintf(
                    ' data-tooltip="%s"',
                    esc_attr($tooltip_text)
                );
            }
            
            $badge = sprintf(
                '<span class="beban-express-badge" style="background-color: %s; color: %s;"%s>%s</span>',
                esc_attr($badge_color),
                esc_attr($text_color),
                $tooltip_attr,
                esc_html($badge_text)
            );
            
            return $name . ' ' . $badge;
        }
        return $name;
    }

    /**
     * Check if should show in cart
     */
    private function should_show_in_cart() {
        $settings = $this->get_settings();
        return isset($settings['express_shipping_show_in_cart']) && $settings['express_shipping_show_in_cart'] === true;
    }



    /**
     * Check if we're in mini cart context
     */
    private function is_mini_cart() {
        // Check if we're in a mini cart context by looking at the current action
        if (did_action('woocommerce_before_mini_cart') || did_action('woocommerce_widget_shopping_cart_before_buttons')) {
            return true;
        }
        
        // Check if we're in Elementor mini cart
        if (did_action('elementor/widget/render_content')) {
            return true;
        }
        
        // Check if we're in a widget context
        if (is_active_widget(false, false, 'woocommerce_widget_cart')) {
            return true;
        }
        
        return false;
    }

    /**
     * Check if product has express shipping enabled
     */
    private function is_express_shipping($product) {
        return $product && $product->get_meta('_beban_express_shipping') === 'yes';
    }

    /**
     * Cleanup feature resources
     */
    public function cleanup() {
        remove_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
        remove_action('woocommerce_before_shop_loop_item_title', array($this, 'display_express_shipping_badge'));
        remove_action('woocommerce_single_product_summary', array($this, 'display_express_shipping_badge'));
        remove_action('woocommerce_cart_item_name', array($this, 'display_express_shipping_badge_in_cart'));
        remove_action('woocommerce_product_options_general_product_data', array($this, 'add_express_shipping_checkbox'));
        remove_action('woocommerce_admin_process_product_object', array($this, 'save_express_shipping_checkbox'));
    }

    /**
     * Display express shipping badge
     */
    public function display_express_shipping_badge() {
        global $product;
        if (!$product) {
            $product = wc_get_product(get_the_ID());
        }
        if ($product && $this->is_express_shipping($product)) {
            echo $this->get_express_shipping_badge();
        }
    }
} 