<?php

/**
 * Abstract base class for all BebanStore features
 */
abstract class BebanStore_Feature {
    /**
     * Feature identifier
     */
    protected $feature_id;

    /**
     * Feature name
     */
    protected $feature_name;

    /**
     * Feature description
     */
    protected $feature_description;

    /**
     * CSS assets
     */
    protected $css_files = array();

    /**
     * JS assets
     */
    protected $js_files = array();

    /**
     * Plugin options
     */
    protected $options;

    /**
     * Initialize the feature
     */
    public function __construct($feature_id, $feature_name, $feature_description) {
        $this->feature_id = $feature_id;
        $this->feature_name = $feature_name;
        $this->feature_description = $feature_description;
        $this->options = get_option('bebanstore_options', array());

        // Only initialize if feature is enabled
        if ($this->is_enabled()) {
            $this->init();
            $this->register_assets();
            $this->register_hooks();
        }
    }

    /**
     * Check if the feature is enabled
     */
    public function is_enabled() {
        return isset($this->options[$this->feature_id]) && $this->options[$this->feature_id];
    }

    /**
     * Initialize the feature
     * Override this method in child classes to add initialization logic
     */
    protected function init() {
        // Base initialization
    }

    /**
     * Register feature assets
     */
    protected function register_assets() {
        if ($this->validate_assets()) {
            add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
        }
    }

    /**
     * Validate feature assets
     * 
     * @return bool Whether all assets are valid
     */
    protected function validate_assets() {
        $valid = true;

        // Validate CSS files
        foreach ($this->css_files as $handle => $file) {
            $file_path = BEBANSTORE_PLUGIN_DIR . "assets/css/{$file}";
            if (!file_exists($file_path)) {
                $valid = false;
            }
        }

        // Validate JS files
        foreach ($this->js_files as $handle => $file) {
            $file_path = BEBANSTORE_PLUGIN_DIR . "assets/js/{$file}";
            if (!file_exists($file_path)) {
                $valid = false;
            }
        }

        return $valid;
    }

    /**
     * Enqueue feature assets
     */
    public function enqueue_assets() {
        // Enqueue CSS files
        foreach ($this->css_files as $handle => $file) {
            wp_enqueue_style(
                "bebanstore-{$handle}",
                BEBANSTORE_PLUGIN_URL . "assets/css/{$file}",
                array(),
                BEBANSTORE_VERSION
            );
        }

        // Enqueue JS files
        foreach ($this->js_files as $handle => $file) {
            wp_enqueue_script(
                "bebanstore-{$handle}",
                BEBANSTORE_PLUGIN_URL . "assets/js/{$file}",
                array('jquery'),
                BEBANSTORE_VERSION,
                true
            );

            // Add nonce for security
            $js_var_name = str_replace('-', '_', $handle);
            wp_localize_script(
                "bebanstore-{$handle}",
                'bebanstore_' . $js_var_name,
                array(
                    'nonce' => wp_create_nonce('bebanstore_' . $js_var_name),
                    'ajaxurl' => admin_url('admin-ajax.php')
                )
            );
        }
    }

    /**
     * Register hooks
     * Override this method in child classes to add hook registration
     */
    protected function register_hooks() {
        // Base hook registration
    }

    /**
     * Get feature settings
     * Override this method in child classes to add feature-specific settings
     */
    public function get_settings() {
        return array();
    }

    /**
     * Cleanup feature resources
     * Override this method in child classes to add cleanup logic
     */
    public function cleanup() {
        // Base cleanup
        remove_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
    }

    /**
     * Get feature title
     */
    public function get_title() {
        return $this->feature_name;
    }

    /**
     * Get feature ID
     */
    public function get_feature_id() {
        return $this->feature_id;
    }

    /**
     * Get feature name
     */
    public function get_feature_name() {
        return $this->feature_name;
    }

    /**
     * Get feature description
     */
    public function get_feature_description() {
        return $this->feature_description;
    }
} 