<?php
if (!defined('ABSPATH')) {
    exit;
}

class BebanStore_Order_View extends BebanStore_Feature {
    public function __construct() {
        parent::__construct(
            'order_view',
            __('طراحی صفحه سفارش', 'bebanstore'),
            __('بهبود ظاهر صفحه جزئیات سفارش ووکامرس با استایل‌های مختلف', 'bebanstore')
        );
    }

    /**
     * Register hooks
     */
    protected function register_hooks() {
        // Enqueue styles
        add_action('wp_enqueue_scripts', array($this, 'enqueue_styles'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
        
        // Remove default WooCommerce elements
        add_action('woocommerce_order_details_after_order_table', array($this, 'remove_order_again_button'), 1);
        

        
        // Add custom elements
        add_action('wp_head', array($this, 'add_custom_scripts'));
        add_action('woocommerce_order_details_before_order_table', array($this, 'add_back_button'), 5);
        add_action('woocommerce_order_details_before_order_table', array($this, 'add_order_summary'), 10);
        add_action('woocommerce_order_details_after_order_table', array($this, 'add_order_products'), 20);
        add_filter('body_class', array($this, 'add_body_class'));
    }

    public function enqueue_assets() {
        // Check if we should load assets
        if (!$this->should_load_assets()) {
            return;
        }

        wp_enqueue_style(
            'bebanstore-order-view',
            plugin_dir_url(dirname(__FILE__)) . 'assets/css/order-view.css',
            array(),
            BEBANSTORE_VERSION
        );
    }

    public function enqueue_styles() {
        // Check if we should load assets
        if (!$this->should_load_assets()) {
            return;
        }
    }

    public function remove_order_again_button($order) {
        remove_action('woocommerce_order_details_after_order_table', 'woocommerce_order_again_button', 10);
    }



    public function add_custom_scripts() {
        // Check if we should load assets
        if (!$this->should_load_assets()) {
            return;
        }
        // Custom scripts can be added here if needed
    }

    public function add_back_button() {
        // Only show back button on view-order page, not on order-received
        if (!is_wc_endpoint_url('view-order')) {
            return;
        }
        ?>
        <a href="<?php echo esc_url(wc_get_account_endpoint_url('orders')); ?>" class="beban-back-button">
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M14.4301 5.92993L20.5001 11.9999L14.4301 18.0699" stroke="#333333" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
                <path d="M3.5 12H20.33" stroke="#333333" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/>
            </svg>
            همه سفارشات
        </a>
        <?php
    }

    /**
     * Check if assets should be loaded based on settings and current page
     */
    private function should_load_assets() {
        // Always load on view-order page
        if (is_wc_endpoint_url('view-order')) {
            return true;
        }
        
        // Check if we should load on order-received page
        if (is_wc_endpoint_url('order-received')) {
            $options = get_option('bebanstore_options', array());
            $feature_settings = isset($options['feature_settings']['order_view']) ? $options['feature_settings']['order_view'] : array();
            $show_in_thank_you = isset($feature_settings['show_in_thank_you']) ? $feature_settings['show_in_thank_you'] : false;
            
            return $show_in_thank_you;
        }
        
        return false;
    }

    public function add_body_class($classes) {
        // Check if we should add body class
        if (!$this->should_load_assets()) {
            return $classes;
        }
        
            // Get feature settings
            $options = get_option('bebanstore_options', array());
            $feature_settings = isset($options['feature_settings']['order_view']) ? $options['feature_settings']['order_view'] : array();
            $selected_style = isset($feature_settings['style']) ? $feature_settings['style'] : 'style1';

            // Add the appropriate class based on selected style
            $classes[] = 'beban-order-view-' . $selected_style;

        return $classes;
    }

    /**
     * Calculate order savings
     * 
     * @param WC_Order $order Order object
     * @return float Total savings
     */
    private function calculate_order_savings($order) {
        $total_savings = 0;
        
        // Calculate product discounts
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            if ($product) {
                $regular_price = $product->get_regular_price();
                $sale_price = $product->get_sale_price();
                
                if ($regular_price && $sale_price && $regular_price > $sale_price) {
                    $item_savings = ($regular_price - $sale_price) * $item->get_quantity();
                    $total_savings += $item_savings;
                }
            }
        }
        
        // Add coupon discounts
        $coupon_discount = $order->get_discount_total();
        $total_savings += $coupon_discount;
        
        return $total_savings;
    }

    public function add_order_summary($order) {
        // Check if we should load assets
        if (!$this->should_load_assets()) {
            return;
        }
        
        // Get feature settings
        $options = get_option('bebanstore_options', array());
        $feature_settings = isset($options['feature_settings']['order_view']) ? $options['feature_settings']['order_view'] : array();
        $selected_style = isset($feature_settings['style']) ? $feature_settings['style'] : 'style1';

        // Order Info
        $order_id = $order->get_id();
        $order_date = wc_format_datetime($order->get_date_created(), 'Y-m-d');
        $order_status = wc_get_order_status_name($order->get_status());
        
        // Financial Info
        $subtotal = wc_price($order->get_subtotal());
        $discount = $order->get_discount_total();
        
        // Check if "Your Savings" feature is enabled and show_in_order_details is enabled
        $your_savings_options = isset($options['feature_settings']['your_savings']) 
            ? $options['feature_settings']['your_savings'] 
            : array();
        
        $show_in_order_details = isset($your_savings_options['show_in_order_details']) 
            ? $your_savings_options['show_in_order_details'] 
            : false;
        
        $savings = '';
        if ($show_in_order_details) {
            $savings = $this->calculate_order_savings($order) > 0 ? wc_price($this->calculate_order_savings($order)) : '';
        }
        
        $payment_method = $order->get_payment_method_title();
        $shipping_method = $order->get_shipping_method() ?: 'رایگان';
        
        // Recipient Info
        $recipient_name = $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();
        $recipient_phone = $order->get_billing_phone();
        $recipient_address = $order->get_formatted_billing_address();

        if ($selected_style === 'style1') {
            ?>
                <div class="beban-order-summary">
                    <!-- Order Info Section -->
                    <div class="beban-summary-section">
                        <div class="beban-summary-boxs">
                            <div class="beban-summary-box">
                                <span class="beban-summary-label">شماره سفارش:</span>
                                <span class="beban-summary-value"><?php echo esc_html($order_id); ?></span>
                            </div>
                            <div class="beban-summary-box">
                                <span class="beban-summary-label">تاریخ ثبت:</span>
                                <span class="beban-summary-value"><?php echo esc_html($order_date); ?></span>
                            </div>
                            <div class="beban-summary-box">
                                <span class="beban-summary-label">وضعیت:</span>
                                <span class="beban-summary-value"><?php echo esc_html($order_status); ?></span>
                            </div>
                        </div>
                    </div>

                    <!-- Financial and Methods Section -->
                    <div class="beban-summary-section">
                        <div class="beban-summary-boxs">
                            <div class="beban-summary-box">
                                <span class="beban-summary-label">مبلغ سفارش:</span>
                                <span class="beban-summary-value"><?php echo $subtotal; ?></span>
                            </div>
                            <?php if ($savings) : ?>
                                <div class="beban-summary-box">
                                    <span class="beban-summary-label">سود شما از این خرید:</span>
                                    <span class="beban-summary-value"><?php echo $savings; ?></span>
                                </div>
                            <?php endif; ?>
                            <div class="beban-summary-box">
                                <span class="beban-summary-label">روش پرداخت:</span>
                                <span class="beban-summary-value"><?php echo esc_html($payment_method); ?></span>
                            </div>
                            <div class="beban-summary-box">
                                <span class="beban-summary-label">روش حمل و نقل:</span>
                                <span class="beban-summary-value"><?php echo esc_html($shipping_method); ?></span>
                            </div>
                        </div>
                    </div>

                    <!-- Recipient Info Section -->
                    <div class="beban-summary-section">
                        <div class="beban-summary-boxs">
                            <div class="beban-summary-box">
                                <span class="beban-summary-label">نام گیرنده:</span>
                                <span class="beban-summary-value"><?php echo esc_html($recipient_name); ?></span>
                            </div>
                            <div class="beban-summary-box">
                                <span class="beban-summary-label">شماره تلفن:</span>
                                <span class="beban-summary-value"><?php echo esc_html($recipient_phone); ?></span>
                            </div>
                            <div class="beban-summary-box beban-info-address">
                                <span class="beban-summary-label">آدرس:</span>
                                <span class="beban-summary-value"><?php echo wp_kses_post($recipient_address); ?></span>
                            </div>
                        </div>
                    </div>
                </div>

            <?php
        } else {
            ?>
                
            <?php
        }
    }

    public function add_order_products($order) {
        // Check if we should load assets
        if (!$this->should_load_assets()) {
            return;
        }
        
        // Get feature settings
        $options = get_option('bebanstore_options', array());
        $feature_settings = isset($options['feature_settings']['order_view']) ? $options['feature_settings']['order_view'] : array();
        $selected_style = isset($feature_settings['style']) ? $feature_settings['style'] : 'style1';

        if ($selected_style === 'style1') {
            ?>
                <div class="beban-order-products">
                    <h2 class="beban-order-title" style="margin: 0 0 0px 0; font-size: 24px; font-weight: bold; color: #333;">مرسولات</h2>
                    <?php foreach ($order->get_items() as $item_id => $item) : 
                        $product = $item->get_product();
                        $product_name = $item->get_name();
                        $quantity = $item->get_quantity();
                        $total = wc_price($item->get_total());
                        $product_url = $product ? get_permalink($product->get_id()) : '#';
                        $product_image = $product ? $product->get_image(array(60, 60)) : wc_placeholder_img(array(60, 60));
                        $is_available = $product && $product->is_in_stock() && $product->is_purchasable();
                        $add_to_cart_url = $is_available ? esc_url(wc_get_cart_url() . '?add-to-cart=' . $item->get_product_id()) : '#';
                    ?>
                        <div class="beban-product-card">
                            <div class="beban-product-image">
                                <?php echo $product_image; ?>
                            </div>
                            <div class="beban-product-details">
                                <div class="beban-product-name">
                                    <a href="<?php echo esc_url($product_url); ?>"><?php echo esc_html($product_name); ?></a>
                                </div>
                                <div class="beban-product-total"><?php echo $total; ?></div>
                                <span class="beban-product-quantity"><?php echo esc_html($quantity); ?> عدد</span>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php
        } else {
            ?>

            <?php
        }
    }
} 