<?php

if (!defined('ABSPATH')) {
    exit;
}

class BebanStore_Product_Discount extends BebanStore_Shortcode_Feature {
    public function __construct() {
        parent::__construct(
            'product_discount',
            __('درصد تخفیف محصول', 'bebanstore'),
            __('نمایش درصد تخفیف محصول به صورت شورت‌کد', 'bebanstore'),
            'beban_product_discount_percentage',
            __('برای استفاده از این ویژگی شورت‌کد را کپی کنید و در صفحه محصول هر جا که دوست دارید قرار بدید تا درصد تخفیف محصول نمایش داده شود.', 'bebanstore'),
            true
        );

        // Add hooks if feature is enabled
        if ($this->is_enabled()) {
            $this->init_hooks();
        }
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Add styles
        add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
        
        // Add hooks for displaying discount next to price
        $settings = $this->get_settings();
        
        if (isset($settings['show_discount_next_to_price_simple']) && $settings['show_discount_next_to_price_simple']) {
            add_filter('woocommerce_get_price_html', [$this, 'add_discount_to_simple_price'], 10, 2);
        }
        
        if (isset($settings['show_discount_next_to_price_variable']) && $settings['show_discount_next_to_price_variable']) {
            add_filter('woocommerce_variable_price_html', [$this, 'add_discount_to_variable_price'], 10, 2);
            add_filter('woocommerce_variable_sale_price_html', [$this, 'add_discount_to_variable_price'], 10, 2);
        }
    }

    /**
     * Enqueue required assets
     */
    public function enqueue_assets() {
        wp_enqueue_style(
            'bebanstore-product-discount',
            plugin_dir_url(dirname(__FILE__)) . 'assets/css/product-discount.css',
            array(),
            BEBANSTORE_VERSION
        );
    }

    /**
     * Get feature settings
     */
    public function get_settings() {
        $options = get_option('bebanstore_options', array());
        return isset($options['feature_settings']['product_discount']) ? $options['feature_settings']['product_discount'] : array();
    }

    public function get_shortcode_callback() {
        return array($this, 'render_shortcode');
    }

    /**
     * Render shortcode
     */
    public function render_shortcode($atts, $content = null) {
        if (!class_exists('WooCommerce')) {
            return '';
        }

        global $product;
        if (!$product) {
            return '';
        }

        $settings = $this->get_settings();

        if ($product->is_type('variable')) {
            // Get variation data
            $variations = $product->get_available_variations();
            $variation_data = array();
            $discount_percentages = array();

            foreach ($variations as $variation) {
                $variation_obj = wc_get_product($variation['variation_id']);
                $original_price = $variation_obj->get_regular_price();
                $sale_price = $variation_obj->get_sale_price();

                if ($original_price > 0 && $sale_price > 0) {
                    $discount_percentage = (($original_price - $sale_price) / $original_price) * 100;
                    $discount_percentages[] = ceil(round($discount_percentage, 2));
                    $variation_data[$variation['variation_id']] = array(
                        'discount' => ceil(round($discount_percentage, 2)),
                        'attributes' => $variation['attributes']
                    );
                }
            }

            // Check if we're on a product page (single product)
            if (is_product()) {
                // On product page - use JavaScript for dynamic discount
                wp_enqueue_script('bebanstore-product-discount', plugin_dir_url(dirname(__FILE__)) . 'assets/js/product-discount.js', array('jquery'), BEBANSTORE_VERSION, true);
                
                // Send settings to script
                wp_localize_script('bebanstore-product-discount', 'bebanstoreDiscountSettings', array(
                    'background_color' => $settings['background_color'],
                    'text_color' => $settings['text_color']
                ));

                // Send variation data to script
                wp_localize_script('bebanstore-product-discount', 'bebanstoreVariationData', $variation_data);

                // Display element for script
                return '<span class="beban-product-discount" style="display: none;"></span>';
            } else {
                // On product boxes/archives - show maximum discount
                if (!empty($discount_percentages)) {
                    $max_discount = max($discount_percentages);
                    $text_color = esc_attr($settings['text_color']);
                    $percent_svg = '<svg xmlns:xlink="http://www.w3.org/1999/xlink" xmlns="http://www.w3.org/2000/svg" width="12" height="13" viewBox="0 0 12 13" fill="none"><path d="M3.51562 9.43164L10.0156 2.93164" stroke=\'' . $text_color . '\' stroke-width="2px" stroke-linecap="round" stroke-linejoin="round" fill="none"/><path d="M9.46875 8.89062H9.48132" stroke=\'' . $text_color . '\' stroke-width="3px" stroke-linecap="round" stroke-linejoin="round" fill="none"/><path d="M4.05134 3.47331H4.06107" stroke=\'' . $text_color . '\' stroke-width="3px" stroke-linecap="round" stroke-linejoin="round" fill="none"/></svg>';
                    
                    return sprintf(
                        '<span class="beban-product-discount" style="--discount-bg-color: %s; color: %s;">
                            <span class="beban-discount-value-number">%s</span>
                            <span class="beban-discount-value-icon">%s</span>
                        </span>',
                        esc_attr($settings['background_color']),
                        $text_color,
                        esc_html($max_discount),
                        $percent_svg
                    );
                }
            }
        } else {
            // Handle simple products and other product types
            $original_price = $product->get_regular_price();
            $sale_price = $product->get_sale_price();
            $show_out_of_stock = isset($settings['show_out_of_stock_discount']) ? $settings['show_out_of_stock_discount'] : false;

            // Check if product has discount
            if ($original_price > 0 && $sale_price > 0 && $sale_price < $original_price) {
                // Check stock status if setting is disabled
                if (!$show_out_of_stock && !$product->is_in_stock()) {
                    return '';
                }

                $discount_percentage = (($original_price - $sale_price) / $original_price) * 100;
                $formatted_discount = ceil(round($discount_percentage, 2));
                $text_color = esc_attr($settings['text_color']);
                $percent_svg = '<svg xmlns:xlink="http://www.w3.org/1999/xlink" xmlns="http://www.w3.org/2000/svg" width="12" height="13" viewBox="0 0 12 13" fill="none"><path d="M3.51562 9.43164L10.0156 2.93164" stroke=\'' . $text_color . '\' stroke-width="2px" stroke-linecap="round" stroke-linejoin="round" fill="none"/><path d="M9.46875 8.89062H9.48132" stroke=\'' . $text_color . '\' stroke-width="3px" stroke-linecap="round" stroke-linejoin="round" fill="none"/><path d="M4.05134 3.47331H4.06107" stroke=\'' . $text_color . '\' stroke-width="3px" stroke-linecap="round" stroke-linejoin="round" fill="none"/></svg>';
                
                return sprintf(
                    '<span class="beban-product-discount" style="--discount-bg-color: %s; color: %s;">
                        <span class="beban-discount-value-number">%s</span>
                        <span class="beban-discount-value-icon">%s</span>
                    </span>',
                    esc_attr($settings['background_color']),
                    $text_color,
                    esc_html($formatted_discount),
                    $percent_svg
                );
            }
        }

        return '';
    }

    /**
     * Add discount to simple product price
     */
    public function add_discount_to_simple_price($price, $product) {
        // فقط برای محصولات عادی و فقط در صفحات محصول
        if (!$product || $product->is_type('variable') || !is_product()) {
            return $price;
        }

        $settings = $this->get_settings();
        $original_price = $product->get_regular_price();
        $sale_price = $product->get_sale_price();
        $show_out_of_stock = isset($settings['show_out_of_stock_discount']) ? $settings['show_out_of_stock_discount'] : false;

        // Check if product has discount
        if ($original_price > 0 && $sale_price > 0 && $sale_price < $original_price) {
            // Check stock status if setting is disabled
            if (!$show_out_of_stock && !$product->is_in_stock()) {
                return $price;
            }

            $discount_percentage = (($original_price - $sale_price) / $original_price) * 100;
            $formatted_discount = ceil(round($discount_percentage, 2));
            $discount_html = $this->get_discount_html($formatted_discount, $settings);
            
            return $price . ' ' . $discount_html;
        }

        return $price;
    }

    /**
     * Add discount to variable product price
     */
    public function add_discount_to_variable_price($price, $product) {
        // فقط برای محصولات متغیر و فقط در صفحات محصول
        if (!$product || !$product->is_type('variable') || !is_product()) {
            return $price;
        }

        $settings = $this->get_settings();
        $variations = $product->get_available_variations();
        $discount_percentages = array();

        foreach ($variations as $variation) {
            $variation_obj = wc_get_product($variation['variation_id']);
            $original_price = $variation_obj->get_regular_price();
            $sale_price = $variation_obj->get_sale_price();

            if ($original_price > 0 && $sale_price > 0) {
                $discount_percentage = (($original_price - $sale_price) / $original_price) * 100;
                $discount_percentages[] = ceil(round($discount_percentage, 2));
            }
        }

        if (!empty($discount_percentages)) {
            $max_discount = max($discount_percentages);
            $discount_html = $this->get_discount_html($max_discount, $settings);
            
            return $price . ' ' . $discount_html;
        }

        return $price;
    }


    /**
     * Get discount HTML
     */
    private function get_discount_html($discount_percentage, $settings) {
        $text_color = esc_attr($settings['text_color']);
        $percent_svg = '<svg xmlns:xlink="http://www.w3.org/1999/xlink" xmlns="http://www.w3.org/2000/svg" width="12" height="13" viewBox="0 0 12 13" fill="none"><path d="M3.51562 9.43164L10.0156 2.93164" stroke=\'' . $text_color . '\' stroke-width="2px" stroke-linecap="round" stroke-linejoin="round" fill="none"/><path d="M9.46875 8.89062H9.48132" stroke=\'' . $text_color . '\' stroke-width="3px" stroke-linecap="round" stroke-linejoin="round" fill="none"/><path d="M4.05134 3.47331H4.06107" stroke=\'' . $text_color . '\' stroke-width="3px" stroke-linecap="round" stroke-linejoin="round" fill="none"/></svg>';
        
        return sprintf(
            '<span class="beban-product-discount" style="--discount-bg-color: %s; color: %s;">
                <span class="beban-discount-value-number">%s</span>
                <span class="beban-discount-value-icon">%s</span>
            </span>',
            esc_attr($settings['background_color']),
            $text_color,
            esc_html($discount_percentage),
            $percent_svg
        );
    }

    /**
     * Cleanup feature resources
     */
    public function cleanup() {
        remove_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
        remove_filter('woocommerce_get_price_html', array($this, 'add_discount_to_simple_price'));
        remove_filter('woocommerce_variable_price_html', array($this, 'add_discount_to_variable_price'));
        remove_filter('woocommerce_variable_sale_price_html', array($this, 'add_discount_to_variable_price'));
    }
} 