<?php

/**
 * Custom Quantity Control Feature
 */
class BebanStore_Quantity_Control extends BebanStore_Feature {
    /**
     * Initialize the feature
     */
    public function __construct() {
        parent::__construct(
            'custom_quantity_control',
            __('دکمه‌های افزایش/کاهش محصول', 'bebanstore'),
            __('نمایش دکمه‌های + و - به جای فیلد عددی برای تغییر تعداد محصول', 'bebanstore')
        );

        // Register assets
        $this->css_files = array(
            'quantity-control' => 'quantity-control.css'
        );
        $this->js_files = array(
            'quantity-control' => 'quantity-control.js'
        );
    }

    /**
     * Register hooks
     */
    protected function register_hooks() {
        // Replace quantity input with custom control
        remove_action('woocommerce_before_quantity_input_field', array($this, 'add_quantity_minus_button'));
        remove_action('woocommerce_after_quantity_input_field', array($this, 'add_quantity_plus_button'));
        
        // Add our custom wrapper for product page
        add_action('woocommerce_before_add_to_cart_quantity', array($this, 'add_custom_quantity_wrapper_start'));
        add_action('woocommerce_after_add_to_cart_quantity', array($this, 'add_custom_quantity_wrapper_end'));

        // Add our custom wrapper for cart page
        add_filter('woocommerce_cart_item_quantity', array($this, 'customize_cart_item_quantity'), 10, 3);

        // Add quantity controls
        add_action('woocommerce_before_quantity_input_field', array($this, 'add_quantity_minus_button'), 10);
        add_action('woocommerce_after_quantity_input_field', array($this, 'add_quantity_plus_button'), 10);

        // Customize quantity input
        add_filter('woocommerce_quantity_input_args', array($this, 'customize_quantity_input'), 10, 2);

        // Hide default remove button if enabled
        add_action('wp_head', array($this, 'hide_default_remove_button'));

        // Add AJAX handlers for cart item removal
        add_action('wp_ajax_bebanstore_remove_cart_item', array($this, 'ajax_remove_cart_item'));
        add_action('wp_ajax_nopriv_bebanstore_remove_cart_item', array($this, 'ajax_remove_cart_item'));

        // Add modal template to footer
        add_action('wp_footer', array($this, 'render_modal_template'));

        // Register and enqueue assets
        add_action('wp_enqueue_scripts', array($this, 'enqueue_feature_assets'));
    }

    /**
     * Customize cart item quantity HTML
     */
    public function customize_cart_item_quantity($product_quantity, $cart_item_key, $cart_item) {
        $_product = apply_filters('woocommerce_cart_item_product', $cart_item['data'], $cart_item, $cart_item_key);
        
        if (!$_product || !$_product->exists()) {
            return $product_quantity;
        }

        $max_value = $_product->get_max_purchase_quantity();
        $min_value = 1; // Minimum quantity should be 1
        $product_name = $_product->get_name();

        ob_start();
        ?>
        <div class="bebanstore-quantity-section">
            <div class="quantity">
                <?php
                do_action('woocommerce_before_quantity_input_field');
                ?>
                <label class="screen-reader-text" for="quantity_<?php echo esc_attr($cart_item_key); ?>"><?php echo esc_html($product_name); ?></label>
                <input 
                    type="text" 
                    id="quantity_<?php echo esc_attr($cart_item_key); ?>" 
                    class="input-text qty text bebanstore-qty-input" 
                    name="cart[<?php echo esc_attr($cart_item_key); ?>][qty]" 
                    value="<?php echo esc_attr($cart_item['quantity']); ?>"
                    aria-label="<?php esc_attr_e('تعداد محصول', 'bebanstore'); ?>"
                    size="4" 
                    min="<?php echo esc_attr($min_value); ?>" 
                    max="<?php echo esc_attr($max_value); ?>"
                    readonly="readonly"
                />
                <?php
                do_action('woocommerce_after_quantity_input_field');
                ?>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Add custom wrapper start for quantity in add to cart section
     */
    public function add_custom_quantity_wrapper_start() {
        echo '<div class="bebanstore-quantity-section">';
    }

    /**
     * Add custom wrapper end for quantity in add to cart section
     */
    public function add_custom_quantity_wrapper_end() {
        echo '</div>';
    }

    /**
     * Add minus button before quantity input
     */
    public function add_quantity_minus_button() {
        $minus_icon = '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
            <path d="M6 12H18" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
        </svg>';
        
        $trash_icon = '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M21 5.98C17.67 5.65 14.32 5.48 10.98 5.48C9 5.48 7.02 5.58 5.04 5.78L3 5.98M8.5 4.97L8.72 3.66C8.88 2.71 9 2 10.69 2H13.31C15 2 15.13 2.75 15.28 3.67L15.5 4.97M18.85 9.14L18.2 19.21C18.09 20.78 18 22 15.21 22H8.79C6 22 5.91 20.78 5.8 19.21L5.15 9.14M10.33 16.5H13.66M9.5 12.5H14.5" stroke="currentColor" stroke-width="1.25" stroke-linecap="round" stroke-linejoin="round"/>
</svg>';
        
        echo '<button type="button" class="bebanstore-qty-button minus" data-minus-icon="' . esc_attr($minus_icon) . '" data-trash-icon="' . esc_attr($trash_icon) . '" aria-label="' . esc_attr__('کاهش تعداد', 'bebanstore') . '">' . $minus_icon . '</button>';
    }

    /**
     * Add plus button after quantity input
     */
    public function add_quantity_plus_button() {
        $plus_icon = '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
            <path d="M6 12H18" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
            <path d="M12 18V6" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
        </svg>';
        
        echo '<button type="button" class="bebanstore-qty-button plus" aria-label="' . esc_attr__('افزایش تعداد', 'bebanstore') . '">' . $plus_icon . '</button>';
    }

    /**
     * Customize quantity input
     */
    public function customize_quantity_input($args, $product) {
        $args['readonly'] = true;
        $args['classes'] = isset($args['classes']) ? array_merge($args['classes'], ['bebanstore-qty-input']) : ['bebanstore-qty-input'];
        return $args;
    }

    /**
     * Hide default remove button if enabled
     */
    public function hide_default_remove_button() {
        $settings = $this->get_settings();
        if (isset($settings['remove_default_remove_button']) && $settings['remove_default_remove_button']) {
            ?>
            <style>
                .cart_item .remove,
                .woocommerce-cart-form .product-remove a {
                    display: none !important;
                }
            </style>
            <?php
        }
    }

    /**
     * AJAX handler for removing cart item
     */
    public function ajax_remove_cart_item() {
        check_ajax_referer('bebanstore_quantity_nonce', 'nonce');

        if (!isset($_POST['cart_item_key'])) {
            wp_send_json_error('Cart item key not provided.');
        }

        $cart_item_key = sanitize_text_field($_POST['cart_item_key']);

        // Remove the cart item
        $removed = WC()->cart->remove_cart_item($cart_item_key);

        if ($removed) {
            // Calculate cart totals
            WC()->cart->calculate_totals();
            
            // Get cart fragments (WooCommerce standard)
            $fragments = apply_filters('woocommerce_add_to_cart_fragments', array());
            
            // Manually add cart totals fragments
            ob_start();
            woocommerce_cart_totals();
            $cart_totals = ob_get_clean();
            
            if (!empty($cart_totals)) {
                $fragments['.cart_totals'] = $cart_totals;
            }
            
            // Add cart subtotal
            ob_start();
            ?>
            <tr class="cart-subtotal">
                <th><?php esc_html_e('Subtotal', 'woocommerce'); ?></th>
                <td data-title="<?php esc_attr_e('Subtotal', 'woocommerce'); ?>"><?php wc_cart_totals_subtotal_html(); ?></td>
            </tr>
            <?php
            $subtotal = ob_get_clean();
            
            if (!empty($subtotal)) {
                $fragments['.cart_totals tr.cart-subtotal'] = $subtotal;
                $fragments['.cart-subtotal'] = $subtotal;
            }
            
            // Add order total fragment
            ob_start();
            ?>
            <tr class="order-total">
                <th><?php esc_html_e('Total', 'woocommerce'); ?></th>
                <td data-title="<?php esc_attr_e('Total', 'woocommerce'); ?>"><?php wc_cart_totals_order_total_html(); ?></td>
            </tr>
            <?php
            $order_total = ob_get_clean();
            
            if (!empty($order_total)) {
                $fragments['.cart_totals tr.order-total'] = $order_total;
                $fragments['.order-total'] = $order_total;
            }
            
            // Trigger Elementor menu cart fragments filter (for Elementor compatibility)
            $fragments = apply_filters('elementor/widgets/menu-cart/fragments', $fragments);
            
            // Get cart hash
            $cart_hash = WC()->cart->get_cart_hash();
            
            // Get updated cart contents count
            $cart_contents_count = WC()->cart->get_cart_contents_count();
            
            wp_send_json_success(array(
                'fragments' => $fragments,
                'cart_hash' => $cart_hash,
                'cart_contents_count' => $cart_contents_count,
                'message' => __('محصول از سبد خرید حذف شد.', 'bebanstore')
            ));
        } else {
            wp_send_json_error(array(
                'message' => __('خطا در حذف محصول. لطفا مجددا تلاش کنید.', 'bebanstore')
            ));
        }
    }

    /**
     * Render modal template for cart item removal
     */
    public function render_modal_template() {
        $settings = $this->get_settings();
        
        // Get color settings with defaults
        $modal_bg_color = isset($settings['modal_background_color']) ? $settings['modal_background_color'] : '#ffffff';
        $confirm_bg_color = isset($settings['confirm_button_bg_color']) ? $settings['confirm_button_bg_color'] : '#dc3545';
        $confirm_text_color = isset($settings['confirm_button_text_color']) ? $settings['confirm_button_text_color'] : '#ffffff';
        $cancel_bg_color = isset($settings['cancel_button_bg_color']) ? $settings['cancel_button_bg_color'] : '#ffffff';
        $cancel_text_color = isset($settings['cancel_button_text_color']) ? $settings['cancel_button_text_color'] : '#007bff';
        $modal_title_color = isset($settings['modal_title_color']) ? $settings['modal_title_color'] : '#000000';
        $modal_description_color = isset($settings['modal_description_color']) ? $settings['modal_description_color'] : '#666666';
        
        ?>
        <div id="bebanstore-remove-cart-item-modal" class="bebanstore-modal" style="display:none;">
            <div class="bebanstore-modal-content" style="background-color: <?php echo esc_attr($modal_bg_color); ?>;">
                <div class="bebanstore-modal-header">
                    <h3 style="color: <?php echo esc_attr($modal_title_color); ?> !important;"><?php esc_html_e('تایید حذف محصول', 'bebanstore'); ?></h3>
                    <button type="button" class="bebanstore-modal-close">&times;</button>
                </div>
                <div class="bebanstore-modal-body">
                    <p class="bebanstore-confirm-message" style="color: <?php echo esc_attr($modal_description_color); ?> !important;"><?php esc_html_e('آیا مطمئن هستید که می‌خواهید این محصول را از سبد خرید حذف کنید؟', 'bebanstore'); ?></p>
                </div>
                <div class="bebanstore-modal-footer">
                    <button type="button" class="bebanstore-modal-cancel button" style="background-color: <?php echo esc_attr($cancel_bg_color); ?> !important; color: <?php echo esc_attr($cancel_text_color); ?> !important;"><?php esc_html_e('انصراف', 'bebanstore'); ?></button>
                    <button type="button" class="bebanstore-modal-confirm button button-primary" style="background-color: <?php echo esc_attr($confirm_bg_color); ?> !important; color: <?php echo esc_attr($confirm_text_color); ?> !important;"><?php esc_html_e('بله، حذف شود', 'bebanstore'); ?></button>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Get feature settings
     */
    public function get_settings() {
        $options = get_option('bebanstore_options', array());
        return isset($options['feature_settings']['custom_quantity_control']) ? $options['feature_settings']['custom_quantity_control'] : array();
    }

    /**
     * Enqueue feature assets
     */
    public function enqueue_feature_assets() {
        // Check if we're on a relevant page
        if (!is_product() && !is_cart() && !is_checkout() && !is_shop() && !is_product_category()) {
            return;
        }

        // Enqueue CSS
        wp_enqueue_style(
            'bebanstore-quantity-control',
            BEBANSTORE_PLUGIN_URL . 'assets/css/quantity-control.css',
            array(),
            BEBANSTORE_VERSION
        );

        // Enqueue JS
        wp_enqueue_script(
            'bebanstore-quantity-control',
            BEBANSTORE_PLUGIN_URL . 'assets/js/quantity-control.js',
            array('jquery'),
            BEBANSTORE_VERSION,
            true
        );

        // Get settings
        $settings = $this->get_settings();
        
        // Get color settings with defaults
        $modal_bg_color = isset($settings['modal_background_color']) ? $settings['modal_background_color'] : '#ffffff';
        $confirm_bg_color = isset($settings['confirm_button_bg_color']) ? $settings['confirm_button_bg_color'] : '#dc3545';
        $confirm_text_color = isset($settings['confirm_button_text_color']) ? $settings['confirm_button_text_color'] : '#ffffff';
        $cancel_bg_color = isset($settings['cancel_button_bg_color']) ? $settings['cancel_button_bg_color'] : '#ffffff';
        $cancel_text_color = isset($settings['cancel_button_text_color']) ? $settings['cancel_button_text_color'] : '#007bff';
        $modal_title_color = isset($settings['modal_title_color']) ? $settings['modal_title_color'] : '#000000';
        $modal_description_color = isset($settings['modal_description_color']) ? $settings['modal_description_color'] : '#666666';
        
        // Add additional data for JS
        wp_localize_script('bebanstore-quantity-control', 'bebanstoreQuantity', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('bebanstore_quantity_nonce'),
            'i18n' => array(
                'maxQuantity' => __('حداکثر تعداد مجاز', 'bebanstore'),
                'minQuantity' => __('حداقل تعداد مجاز', 'bebanstore'),
                'confirmRemove' => __('آیا مطمئن هستید؟', 'bebanstore'),
                'confirmRemoveMessage' => __('آیا مطمئن هستید که می‌خواهید این محصول را از سبد خرید حذف کنید؟ این عملیات غیرقابل برگشت است.', 'bebanstore'),
                'cancel' => __('انصراف', 'bebanstore'),
                'remove' => __('حذف', 'bebanstore')
            ),
            'isProduct' => is_product(),
            'isCart' => is_cart(),
            'isCheckout' => is_checkout(),
            'enableTrashButton' => isset($settings['enable_trash_button']) ? $settings['enable_trash_button'] : false,
            'removeDefaultRemoveButton' => isset($settings['remove_default_remove_button']) ? $settings['remove_default_remove_button'] : false,
            'enableRemoveConfirmation' => isset($settings['enable_remove_confirmation']) ? $settings['enable_remove_confirmation'] : true,
            'modalColors' => array(
                'modalBgColor' => $modal_bg_color,
                'confirmBgColor' => $confirm_bg_color,
                'confirmTextColor' => $confirm_text_color,
                'cancelBgColor' => $cancel_bg_color,
                'cancelTextColor' => $cancel_text_color,
                'modalTitleColor' => $modal_title_color,
                'modalDescriptionColor' => $modal_description_color
            )
        ));
    }
} 