<?php

/**
 * Sales Progress Feature
 */
class BebanStore_Sales_Progress extends BebanStore_Shortcode_Feature {
    /**
     * Initialize the feature
     */
    public function __construct() {
        parent::__construct(
            'sales_progress',
            __('نمایش پیشرفت فروش', 'bebanstore'),
            __('نمایش نوار پیشرفت فروش محصول به همراه درصد', 'bebanstore'),
            'bebanstore_sales_progress',
            __('برای استفاده از این ویژگی شورتکد را کپی کنید و در سیستم حلقه المنتور رو باکس محصولات یا در صفحه محصول هر جا که دوست دارید قرار بدید تا درصد و پیشرفت فروش نمایش داده شود.', 'bebanstore')
        );

        // Add styles and scripts
        if ($this->is_enabled()) {
            add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
        }
    }

    /**
     * Enqueue required assets
     */
    public function enqueue_assets() {
        wp_enqueue_style(
            'bebanstore-sales-progress',
            BEBANSTORE_PLUGIN_URL . 'assets/css/sales-progress.css',
            array(),
            BEBANSTORE_VERSION
        );

        wp_enqueue_script(
            'bebanstore-sales-progress',
            BEBANSTORE_PLUGIN_URL . 'assets/js/sales-progress.js',
            array('jquery'),
            BEBANSTORE_VERSION,
            true
        );
    }

    /**
     * Render the shortcode
     */
    public function render_shortcode($atts, $content = null) {
        // Get current product
        global $product;
        if (!$product || !is_a($product, 'WC_Product')) {
            return '';
        }

        // Get stock information with caching
        $total_stock = $product->get_stock_quantity();
        $stock_sold = $this->get_total_sales($product);
        
        // Calculate progress percentage
        $progress = 0;
        if ($total_stock > 0) {
            $progress = min(100, round(($stock_sold / ($total_stock + $stock_sold)) * 100));
        }

        // Get feature settings
        $options = get_option('bebanstore_options', array());
        $feature_settings = isset($options['feature_settings']['sales_progress']) ? $options['feature_settings']['sales_progress'] : array();
        
        // Get settings with defaults
        $sold_text = isset($feature_settings['sold_text']) ? $feature_settings['sold_text'] : 'فروش رفته';
        $progress_color = isset($feature_settings['progress_color']) ? $feature_settings['progress_color'] : '#615fff';
        $progress_style = isset($feature_settings['progress_style']) ? $feature_settings['progress_style'] : 'linear';
        $min_sales = isset($feature_settings['min_sales']) ? intval($feature_settings['min_sales']) : 0;

        // Check if we should show progress based on min_sales
        if ($stock_sold < $min_sales) {
            return '';
        }

        // Generate unique ID for this instance
        $unique_id = 'bebanstore-progress-' . uniqid();

        ob_start();
        ?>
        <div class="bebanstore-sales-progress bebanstore-progress-<?php echo esc_attr($progress_style); ?>" 
             id="<?php echo esc_attr($unique_id); ?>"
             data-product-id="<?php echo esc_attr($product->get_id()); ?>">
            <div class="bebanstore-progress-bar">
                <div class="bebanstore-progress-fill" 
                     style="width: <?php echo esc_attr($progress); ?>%; background-color: <?php echo esc_attr($progress_color); ?>"></div>
            </div>
            <div class="bebanstore-progress-text">
                <?php printf(
                    /* translators: %s: Progress percentage */
                    esc_html__('%s%% %s', 'bebanstore'),
                    esc_html($progress),
                    esc_html($sold_text)
                ); ?>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Get total sales for a product
     */
    private function get_total_sales($product) {
        return $product->get_total_sales();
    }

    /**
     * Validate color value
     */
    private function validate_color($color) {
        if (preg_match('/^#[a-f0-9]{6}$/i', $color)) {
            return $color;
        }
        return '#615fff';
    }
} 