<?php

/**
 * Base class for shortcode features
 */
abstract class BebanStore_Shortcode_Feature extends BebanStore_Feature {
    /**
     * Feature name
     */
    protected $feature_name;

    /**
     * Feature description
     */
    protected $feature_description;

    /**
     * Shortcode tag
     */
    protected $shortcode_tag;

    /**
     * Usage instructions
     */
    protected $usage_instructions;

    /**
     * Shortcode attributes
     */
    protected $shortcode_attributes = array();

    /**
     * Initialize the feature
     */
    public function __construct($feature_id, $feature_name, $feature_description, $shortcode_tag, $usage_instructions) {
        parent::__construct($feature_id, $feature_name, $feature_description);
        
        $this->shortcode_tag = $shortcode_tag;
        $this->usage_instructions = $usage_instructions;

        if ($this->is_enabled()) {
            add_shortcode($this->shortcode_tag, array($this, 'render_shortcode'));
        }

        // Add scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
    }

    /**
     * Get feature title
     */
    public function get_title() {
        return $this->feature_name;
    }

    /**
     * Get feature description
     */
    public function get_description() {
        return $this->feature_description;
    }

    /**
     * Get shortcode
     */
    public function get_shortcode() {
        return '[' . $this->shortcode_tag . ']';
    }

    /**
     * Get usage instructions
     * 
     * @return string Usage instructions for the shortcode
     */
    public function get_usage_instructions() {
        return $this->usage_instructions;
    }

    /**
     * Get shortcode attributes
     * 
     * @return array Array of shortcode attributes with their descriptions
     */
    public function get_shortcode_attributes() {
        return $this->shortcode_attributes;
    }

    /**
     * Validate shortcode attributes
     * 
     * @param array $atts Shortcode attributes
     * @return array Validated attributes
     */
    protected function validate_shortcode_attributes($atts) {
        $validated = array();
        
        foreach ($this->shortcode_attributes as $attr => $config) {
            if (isset($atts[$attr])) {
                $value = $atts[$attr];
                
                // Validate based on type
                if (isset($config['type'])) {
                    switch ($config['type']) {
                        case 'number':
                            $value = intval($value);
                            break;
                        case 'boolean':
                            $value = filter_var($value, FILTER_VALIDATE_BOOLEAN);
                            break;
                        case 'string':
                            $value = sanitize_text_field($value);
                            break;
                    }
                }
                
                $validated[$attr] = $value;
            } elseif (isset($config['default'])) {
                $validated[$attr] = $config['default'];
            }
        }
        
        return $validated;
    }

    /**
     * Render show shortcode button
     */
    public function render_show_shortcode_button() {
        $nonce = wp_create_nonce('bebanstore_shortcode_' . $this->feature_id);
        ?>
        <button type="button" class="button bebanstore-show-shortcode" 
                data-feature-id="<?php echo esc_attr($this->feature_id); ?>"
                data-feature-title="<?php echo esc_attr($this->feature_name); ?>"
                data-shortcode="<?php echo esc_attr($this->get_shortcode()); ?>"
                data-description="<?php echo esc_attr($this->usage_instructions); ?>"
                data-nonce="<?php echo esc_attr($nonce); ?>">
            <?php esc_html_e('نمایش شورت‌کد', 'bebanstore'); ?>
        </button>
        <?php
    }

    /**
     * Abstract method for shortcode rendering
     */
    abstract public function render_shortcode($atts, $content = null);

    /**
     * Enqueue scripts and styles
     */
    public function enqueue_scripts($hook) {
        // Only load on plugin settings page
        if ($hook != 'toplevel_page_bebanstore-settings') {
            return;
        }

        // Enqueue admin styles
        wp_enqueue_style('bebanstore-admin', BEBANSTORE_PLUGIN_URL . 'assets/css/admin.css', array(), BEBANSTORE_VERSION);
        
        // Enqueue admin scripts
        wp_enqueue_script('bebanstore-admin', BEBANSTORE_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), BEBANSTORE_VERSION, true);
        
        // Localize script
        wp_localize_script('bebanstore-admin', 'bebanstoreShortcode', array(
            'copied' => esc_html__('کپی شد!', 'bebanstore'),
            'nonce' => wp_create_nonce('bebanstore_shortcode_action')
        ));
    }

    /**
     * Cleanup feature resources
     */
    public function cleanup() {
        parent::cleanup();
        remove_shortcode($this->shortcode_tag);
        remove_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
    }
} 