<?php
if (!defined('ABSPATH')) {
    exit;
}

class BebanStore_Wishlist extends BebanStore_Shortcode_Feature {
    public function __construct() {
        parent::__construct(
            'wishlist',
            __('سیستم علاقه‌مندی‌ها', 'bebanstore'),
            __('افزودن محصولات به لیست علاقه‌مندی‌ها و مدیریت آنها', 'bebanstore'),
            'beban_wishlist_icon',
            __('[beban_wishlist_icon]: برای نمایش دکمه قلب روی محصولات - این شورت‌کد را در صفحه محصول یا حلقه محصولات قرار دهید.

[beban_wishlist_products]: برای نمایش لیست محصولات علاقه‌مندی - این شورت‌کد را در صفحه علاقه‌مندی‌ها یا هر جای دیگر قرار دهید.', 'bebanstore')
        );

        // Register hooks
        $this->register_hooks();
    }

    protected function register_hooks() {
        // Add AJAX handlers (always available)
        add_action('wp_ajax_beban_handle_wishlist', array($this, 'handle_wishlist_ajax'));
        add_action('wp_ajax_nopriv_beban_handle_wishlist', array($this, 'handle_wishlist_ajax'));
        add_action('wp_ajax_beban_check_stock_status', array($this, 'check_stock_status_ajax'));
        add_action('wp_ajax_nopriv_beban_check_stock_status', array($this, 'check_stock_status_ajax'));
        add_action('wp_ajax_bebanstore_add_to_cart_from_wishlist', array($this, 'add_to_cart_from_wishlist_ajax'));
        add_action('wp_ajax_nopriv_bebanstore_add_to_cart_from_wishlist', array($this, 'add_to_cart_from_wishlist_ajax'));
        
        // Add shortcodes (always available)
        add_shortcode('beban_wishlist_icon', array($this, 'display_wishlist_icon'));
        add_shortcode('beban_wishlist_products', array($this, 'display_wishlist_products'));
        
        // Add assets only if feature is enabled
        if ($this->is_enabled()) {
            add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
            
            // Add account tab if enabled
            $options = get_option('bebanstore_options', array());
            $settings = isset($options['feature_settings']['wishlist']) ? $options['feature_settings']['wishlist'] : array();
            $add_account_tab = isset($settings['add_account_tab']) ? $settings['add_account_tab'] : false;
            
            if ($add_account_tab) {
                add_filter('woocommerce_account_menu_items', array($this, 'add_wishlist_account_tab'));
                add_action('woocommerce_account_wishlist_products_endpoint', array($this, 'wishlist_account_tab_content'));
                add_action('init', array($this, 'add_wishlist_endpoint'));
                // Ensure WooCommerce recognizes endpoint for active tab state
                add_filter('woocommerce_get_query_vars', array($this, 'register_wishlist_query_var'));
                
                // Flush rewrite rules on plugin activation
                add_action('init', array($this, 'maybe_flush_rewrite_rules'), 20);
            }
        }
        
        // Add hook to flush rewrite rules when settings are updated
        add_action('update_option_bebanstore_options', array($this, 'check_wishlist_settings_update'), 10, 3);
    }

    /**
     * Display wishlist icon shortcode
     */
    public function display_wishlist_icon($atts) {
        $atts = shortcode_atts(array(
            'product_id' => 0,
        ), $atts);
        
        // Get product ID
        $product_id = $atts['product_id'];
        if (!$product_id) {
            global $product;
            if ($product) {
                $product_id = $product->get_id();
            }
        }
        
        if (!$product_id) {
            return '';
        }
        
        $user_id = get_current_user_id();
        $wishlist_items = get_user_meta($user_id, 'beban_wishlist_items', true);
        
        if (!is_array($wishlist_items)) {
            $wishlist_items = array();
        }
        
        $is_in_wishlist = in_array($product_id, $wishlist_items);
        $wishlist_class = $is_in_wishlist ? 'beban-wishlisted' : '';
        
        // Get product object for stock status
        $product_obj = wc_get_product($product_id);
        $stock_status = $product_obj ? $product_obj->get_stock_status() : 'instock';
        $is_in_stock = $product_obj ? $product_obj->is_in_stock() : true;
        
        // Get settings for colors
        $settings = $this->get_wishlist_settings();
        
        // رنگ‌های حالت عادی (قلب خالی)
        $normal_fill = isset($settings['wishlist_btn_normal_fill']) ? $settings['wishlist_btn_normal_fill'] : 'none';
        $normal_stroke = isset($settings['wishlist_btn_normal_stroke']) ? $settings['wishlist_btn_normal_stroke'] : '#A9ABAD';
        
        // رنگ‌های حالت فعال (قلب پر)
        $active_fill = isset($settings['wishlist_btn_active_fill']) ? $settings['wishlist_btn_active_fill'] : '#292D32';
        $active_stroke = isset($settings['wishlist_btn_active_stroke']) ? $settings['wishlist_btn_active_stroke'] : '#292D32';
        
        ob_start();
        ?>
        <div class="beban-wishlist-btn <?php echo esc_attr($wishlist_class); ?>" 
             data-product-id="<?php echo esc_attr($product_id); ?>"
             data-stock-status="<?php echo esc_attr($stock_status); ?>"
             data-in-stock="<?php echo $is_in_stock ? 'true' : 'false'; ?>">
            <?php if ($is_in_wishlist) : ?>
                <!-- Filled Heart Icon -->
                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none">
                    <path d="M12.62 20.8101C12.28 20.9301 11.72 20.9301 11.38 20.8101C8.48 19.8201 2 15.6901 2 8.6901C2 5.6001 4.49 3.1001 7.56 3.1001C9.38 3.1001 10.99 3.9801 12 5.3401C13.01 3.9801 14.63 3.1001 16.44 3.1001C19.51 3.1001 22 5.6001 22 8.6901C22 15.6901 15.52 19.8201 12.62 20.8101Z" fill="<?php echo esc_attr($active_fill); ?>" stroke="<?php echo esc_attr($active_stroke); ?>" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                </svg>
            <?php else : ?>
                <!-- Empty Heart Icon -->
                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none">
                    <path d="M12.62 20.8101C12.28 20.9301 11.72 20.9301 11.38 20.8101C8.48 19.8201 2 15.6901 2 8.6901C2 5.6001 4.49 3.1001 7.56 3.1001C9.38 3.1001 10.99 3.9801 12 5.3401C13.01 3.9801 14.63 3.1001 16.44 3.1001C19.51 3.1001 22 5.6001 22 8.6901C22 15.6901 15.52 19.8201 12.62 20.8101Z" fill="<?php echo esc_attr($normal_fill); ?>" stroke="<?php echo esc_attr($normal_stroke); ?>" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                </svg>
            <?php endif; ?>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Display wishlist products shortcode
     */
    public function display_wishlist_products($atts = array()) {
        // Get settings
        $settings = $this->get_wishlist_settings();
        
        // Parse shortcode attributes
        $atts = shortcode_atts(array(
            'page' => 1,
            'per_page' => isset($settings['products_per_page']) ? intval($settings['products_per_page']) : 12
        ), $atts);
        
        // Check if login is required
        $require_login = isset($settings['require_login']) ? $settings['require_login'] : true;
        
        if ($require_login && !is_user_logged_in()) {
            $login_message = isset($settings['login_required_message']) ? $settings['login_required_message'] : __('لطفاً ابتدا وارد حساب کاربری خود شوید.', 'bebanstore');
            return '<p>' . $login_message . '</p>';
        }

        $user_id = get_current_user_id();
        $wishlist_items = get_user_meta($user_id, 'beban_wishlist_items', true);
        
        if (!is_array($wishlist_items) || empty($wishlist_items)) {
            $empty_message = isset($settings['empty_wishlist_message']) ? $settings['empty_wishlist_message'] : __('لیست علاقه‌مندی‌های شما خالی است.', 'bebanstore');
            return '
                <div style="text-align:center; margin-top:90px;">
                    <div style="margin-bottom:20px;">
                        <svg xmlns="http://www.w3.org/2000/svg" version="1.1" xmlns:xlink="http://www.w3.org/1999/xlink" width="80" height="80" x="0" y="0" viewBox="0 0 512 512" style="enable-background:new 0 0 512 512" xml:space="preserve"><defs><linearGradient id="a" x1="107.83" x2="395" y1="312.8" y2="312.8" gradientUnits="userSpaceOnUse"><stop offset="0" stop-color="#a9afc4"></stop><stop offset="1" stop-color="#d4d8e5"></stop></linearGradient><linearGradient id="b" x1="321.03" x2="395" y1="262.61" y2="262.61" gradientUnits="userSpaceOnUse"><stop offset="0" stop-color="#cacfdd"></stop><stop offset="1" stop-color="#eef0f1"></stop></linearGradient></defs><g><linearGradient id="a" x1="107.83" x2="395" y1="312.8" y2="312.8" gradientUnits="userSpaceOnUse"><stop offset="0" stop-color="#a9afc4"></stop><stop offset="1" stop-color="#d4d8e5"></stop></linearGradient><linearGradient id="b" x1="321.03" x2="395" y1="262.61" y2="262.61" gradientUnits="userSpaceOnUse"><stop offset="0" stop-color="#cacfdd"></stop><stop offset="1" stop-color="#eef0f1"></stop></linearGradient><g data-name="Folder Empty"><path fill="#5E6175" d="M127.55 181.16a23.49 23.49 0 0 1-19-27.21 17.5 17.5 0 0 0-14.15-20.37c-12.72-2.6-21.22-14.83-19-27.28l5.91 1c-1.64 9.27 4.74 18.4 14.23 20.34a23.48 23.48 0 0 1 18.94 27.2 17.51 17.51 0 0 0 14.15 20.36zM183.81 178.23a33.76 33.76 0 0 1-1.68-47.74c10-10.28 10.36-28.34-1.37-39.28-13.32-13-14-34.46-1.67-47.76l4.39 4.08c-10.18 10.95-9.54 28.6 1.43 39.34 14.12 13.17 13.82 35.18 1.55 47.77a27.77 27.77 0 0 0 1.44 39.2zM244.48 207.84a39.36 39.36 0 0 1 9.46-54.84 33.41 33.41 0 0 0 8-46.52l4.91-3.45a39.41 39.41 0 0 1-9.47 54.86 33.35 33.35 0 0 0-8 46.52z" opacity="1" data-original="#5e6175"></path><path fill="url(&quot;#a&quot;)" d="m394.75 254.09-22.19 148.47a31.81 31.81 0 0 1-31.46 27.12h-1.59a18.79 18.79 0 0 0 18.39-22.58l-27.37-132.5a22.62 22.62 0 0 0-22.14-18H148.66l-13.35-22A22.61 22.61 0 0 0 116 223.73h-7.54a22.6 22.6 0 0 1 22-27.81h38.23a22.63 22.63 0 0 1 19.12 10.54l13.68 21.69h170.9a22.61 22.61 0 0 1 22.36 25.94z" opacity="1" data-original="url(#a)"></path><path fill="url(&quot;#b&quot;)" d="m394.75 254.09-6.43 43A86.65 86.65 0 0 1 321 228.15h51.36a22.61 22.61 0 0 1 22.39 25.94z" opacity="1" data-original="url(#b)"></path><path fill="#DEE1EC" d="M339.51 429.68H101.35a22.6 22.6 0 0 1-22.13-18L46 250.91a22.61 22.61 0 0 1 22.14-27.18H116a22.61 22.61 0 0 1 19.32 10.86l13.35 22h159.72a22.62 22.62 0 0 1 22.14 18L357.9 407.1a18.79 18.79 0 0 1-18.39 22.58z" opacity="1" data-original="#dee1ec"></path><g fill="#5E6175"><path d="M339.51 432.68H101.35a25.7 25.7 0 0 1-25.07-20.43l-33.2-160.73a25.61 25.61 0 0 1 25.08-30.79H116A25.74 25.74 0 0 1 137.87 233l12.48 20.52h158A25.73 25.73 0 0 1 333.47 274l27.37 132.5a21.78 21.78 0 0 1-21.33 26.18zM68.16 226.73A19.6 19.6 0 0 0 49 250.3L82.15 411a19.69 19.69 0 0 0 19.2 15.64h238.16a15.78 15.78 0 0 0 15.45-19l-27.37-132.5a19.69 19.69 0 0 0-19.21-15.65H147l-14.22-23.41a19.73 19.73 0 0 0-16.78-9.35z" fill="#5E6175" opacity="1" data-original="#5e6175"></path><path d="M341.1 432.68h-1.59v-6a15.78 15.78 0 0 0 15.45-19l-27.37-132.5a19.69 19.69 0 0 0-19.21-15.65H147l-14.22-23.41a19.73 19.73 0 0 0-16.78-9.39h-9.92l-.54-2.31a25.61 25.61 0 0 1 24.92-31.5h38.23a25.45 25.45 0 0 1 21.65 11.94l12.81 20.29h169.24a25.61 25.61 0 0 1 25.32 29.39c0 .1.5-3.3-22.18 148.47a35 35 0 0 1-34.43 29.67zM150.35 253.56h158A25.73 25.73 0 0 1 333.47 274l27.37 132.5a21.77 21.77 0 0 1-1.64 13.71 28.79 28.79 0 0 0 10.39-18.09l22.19-148.47a19.61 19.61 0 0 0-19.39-22.5H199.83l-14.57-23.09a19.51 19.51 0 0 0-16.58-9.14h-38.23A19.61 19.61 0 0 0 111 220.73h5A25.74 25.74 0 0 1 137.87 233zM137.563 327.64l32.854-32.854 11.128 11.128-32.854 32.854z" fill="#5E6175" opacity="1" data-original="#5e6175"></path><path d="m137.563 305.907 11.128-11.128 32.854 32.854-11.128 11.128zM214.041 327.641l32.855-32.854 11.128 11.128-32.854 32.855z" fill="#5E6175" opacity="1" data-original="#5e6175"></path><path d="m214.041 305.908 11.129-11.128 32.854 32.854-11.128 11.129zM253.11 377.28c-.87-.78-21.75-19.21-54.33-19.21s-53.47 18.43-54.34 19.21l-2.57 2.31 11.19 12.47 2.57-2.25a69.77 69.77 0 0 1 86.29 0l2.57 2.28 11.19-12.5z" fill="#5E6175" opacity="1" data-original="#5e6175"></path></g><circle cx="406.04" cy="212.47" r="69.96" fill="#F5F5F5" opacity="1" data-original="#f5f5f5"></circle><path fill="#5E6175" d="M406 285.43a73 73 0 1 1 73-73 73 73 0 0 1-73 73zm0-139.91a67 67 0 1 0 67 66.95 67 67 0 0 0-67-66.95zM316.66 149.05a9.62 9.62 0 1 1 9.62-9.62 9.62 9.62 0 0 1-9.62 9.62zm0-13.23a3.62 3.62 0 1 0 3.62 3.61 3.61 3.61 0 0 0-3.62-3.61zM361.75 125.5A13.21 13.21 0 1 1 375 112.29a13.22 13.22 0 0 1-13.25 13.21zm0-20.41a7.21 7.21 0 1 0 7.21 7.2 7.21 7.21 0 0 0-7.21-7.2zM50.23 448.461l13.743-13.744 4.242 4.242-13.744 13.744zM79.18 447.07h6v19.44h-6zM36 419.03h19.44v6H36z" opacity="1" data-original="#5e6175"></path><path fill="#E1830E" d="M413.77 229.15h-15.45A33 33 0 0 1 412 202.41a10.27 10.27 0 1 0-16.26-8.35h-15.41A25.71 25.71 0 1 1 421.05 215a17.52 17.52 0 0 0-7.28 14.15zM398.15 241.09h15.51v15.51h-15.51z" opacity="1" data-original="#e1830e"></path><path fill="#A9AFC4" d="M340.15 386.22h-6v-6h6zm-16 0h-6v-6h6zm-16 0h-6v-6h6zm-16 0h-6v-6h6zM360.94 380.22h6v6h-6zM406.49 386.22h-6v-6h6zm-16 0h-6v-6h6zM422.21 386.21h-6v-6h6zm0-16h-6v-6h6zm0-16h-6v-6h6zm0-16h-6v-6h6zm0-16h-6v-6h6zm0-16h-6v-6h6z" opacity="1" data-original="#a9afc4"></path></g></g></svg>
                    </div>
                    <p style="color:#999;">' . $empty_message . '</p>
                </div>
            ';
        }

        // Get current page from URL parameter or shortcode attribute
        $current_page = isset($_GET['wishlist_page']) ? max(1, intval($_GET['wishlist_page'])) : max(1, intval($atts['page']));
        $per_page = max(1, intval($atts['per_page']));
        
        // Calculate pagination
        $total_items = count($wishlist_items);
        $total_pages = ceil($total_items / $per_page);
        $offset = ($current_page - 1) * $per_page;
        
        // Get paginated product IDs
        $paginated_items = array_slice($wishlist_items, $offset, $per_page);
        
        // Optimized query with pagination
        $products = wc_get_products(array(
            'include' => $paginated_items,
            'limit' => $per_page,
            'status' => 'publish',
            'orderby' => 'post__in', // Maintain order from wishlist
            'return' => 'objects'
        ));

        if (empty($products)) {
            return '<p>' . __('محصولی در لیست علاقه‌مندی‌ها یافت نشد.', 'bebanstore') . '</p>';
        }

        ob_start();
        ?>
        <div class="beban-wishlist-products">
                        <?php
            // Show pagination at top if enabled and position is top or both
            $pagination_position = isset($settings['pagination_position']) ? $settings['pagination_position'] : 'bottom';
            $enable_pagination = isset($settings['enable_pagination']) ? $settings['enable_pagination'] : true;
            
            if ($enable_pagination && $total_pages > 1 && in_array($pagination_position, array('top', 'both'))) {
                echo $this->render_pagination($current_page, $total_pages, $total_items, $per_page);
            }
            
            // Check if bulk actions are enabled
            $enable_bulk_actions = isset($settings['enable_bulk_actions']) ? $settings['enable_bulk_actions'] : true;
            
            // Get display style setting
            $display_style = isset($settings['wishlist_display_style']) ? $settings['wishlist_display_style'] : 'grid';
            
            // Get button style settings
            $add_to_cart_bg = isset($settings['add_to_cart_normal_bg']) ? $settings['add_to_cart_normal_bg'] : '#ffffff';
            $add_to_cart_text = isset($settings['add_to_cart_normal_text']) ? $settings['add_to_cart_normal_text'] : '#ef394e';
            $add_to_cart_icon = isset($settings['add_to_cart_normal_icon']) ? $settings['add_to_cart_normal_icon'] : '#ef394e';
            $add_to_cart_border = isset($settings['add_to_cart_normal_border']) ? $settings['add_to_cart_normal_border'] : '#ef394e';
            
            $remove_btn_bg = isset($settings['remove_btn_normal_bg']) ? $settings['remove_btn_normal_bg'] : '#ffffff';
            $remove_btn_icon = isset($settings['remove_btn_normal_icon']) ? $settings['remove_btn_normal_icon'] : '#81858B';
            $remove_btn_border = isset($settings['remove_btn_normal_border']) ? $settings['remove_btn_normal_border'] : '#E0E0E2';
            
            // Get product content style settings
            $title_font_size = isset($settings['product_title_font_size']) ? intval($settings['product_title_font_size']) : 14;
            $title_color = isset($settings['product_title_color']) ? $settings['product_title_color'] : '#333333';
            $regular_price_font_size = isset($settings['product_regular_price_font_size']) ? intval($settings['product_regular_price_font_size']) : 13;
            $regular_price_color = isset($settings['product_regular_price_color']) ? $settings['product_regular_price_color'] : '#999999';
            $sale_price_font_size = isset($settings['product_sale_price_font_size']) ? intval($settings['product_sale_price_font_size']) : 16;
            $sale_price_color = isset($settings['product_sale_price_color']) ? $settings['product_sale_price_color'] : '#ef394e';
            ?>
            
            <?php if ($enable_bulk_actions) : ?>
            <!-- Bulk Actions -->
            <div class="beban-wishlist-bulk-actions">
                <div class="beban-wishlist-bulk-controls">
                    <div class="beban-wishlist-checkbox-wrapper">
                        <label class="beban-wishlist-checkbox">
                            <input class="beban-wishlist-checkbox__trigger visuallyhidden" type="checkbox" id="beban-wishlist-select-all" class="beban-wishlist-checkbox" />
                            <span class="beban-wishlist-checkbox__symbol">
                                <svg aria-hidden="true" class="beban-wishlist-checkbox__icon" width="28px" height="28px" viewBox="0 0 28 28" version="1" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M4 14l8 7L24 7"></path>
                                </svg>
                            </span>
                            <p class="beban-wishlist-checkbox__textwrapper"><?php _e('انتخاب همه', 'bebanstore'); ?></p>
                        </label>
                    </div>
                    <button type="button" class="beban-wishlist-bulk-remove-btn" disabled>
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M8 2v2h8V2H8zM4 7V5h16v2H4zm13 1h2v11a3 3 0 01-3 3H8a3 3 0 01-3-3V8h2v11a1 1 0 001 1h8a1 1 0 001-1V8zm-6 0H9v10h2V8zm2 0h2v10h-2V8z" fill="currentColor"/>
                        </svg>
                        <?php _e('حذف انتخاب شده', 'bebanstore'); ?>
                        <span class="beban-wishlist-selected-count">(0)</span>
                    </button>
                </div>
            </div>
            <?php endif; ?>
            
            <div class="beban-wishlist-<?php echo esc_attr($display_style); ?>">
                <?php foreach ($products as $product) : 
                    // Check if product is in cart
                    $product_id = $product->get_id();
                    $is_in_cart = false;
                    $cart_quantity = 0;
                    
                    if (WC()->cart) {
                        foreach (WC()->cart->get_cart() as $cart_item) {
                            if ($cart_item['product_id'] == $product_id) {
                                $is_in_cart = true;
                                $cart_quantity = $cart_item['quantity'];
                                break;
                            }
                        }
                    }
                ?>
                    <div class="beban-wishlist-item beban-wishlist-item-<?php echo esc_attr($display_style); ?> <?php echo $is_in_cart ? 'in-cart' : ''; ?>" data-product-id="<?php echo esc_attr($product->get_id()); ?>">
                        <?php if ($enable_bulk_actions) : ?>
                        <div class="beban-wishlist-item-checkbox">
                            <div class="beban-wishlist-checkbox-wrapper">
                                <label class="beban-wishlist-checkbox">
                                    <input class="beban-wishlist-checkbox__trigger visuallyhidden beban-wishlist-product-checkbox" 
                                           type="checkbox" 
                                           value="<?php echo esc_attr($product->get_id()); ?>"
                                           data-product-id="<?php echo esc_attr($product->get_id()); ?>" />
                                    <span class="beban-wishlist-checkbox__symbol">
                                        <svg aria-hidden="true" class="beban-wishlist-checkbox__icon" width="28px" height="28px" viewBox="0 0 28 28" version="1" xmlns="http://www.w3.org/2000/svg">
                                            <path d="M4 14l8 7L24 7"></path>
                                        </svg>
                                    </span>
                                </label>
                            </div>
                        </div>
                        <?php endif; ?>
                        <div class="beban-wishlist-image">
                            <a href="<?php echo esc_url($product->get_permalink()); ?>">
                                <?php echo $product->get_image('medium'); ?>
                            </a>
                            <?php if ($is_in_cart) : ?>
                                <div class="beban-wishlist-cart-notice">
                                    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M12 22C17.5 22 22 17.5 22 12C22 6.5 17.5 2 12 2C6.5 2 2 6.5 2 12C2 17.5 6.5 22 12 22Z" stroke="#ef394e" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                                        <path d="M7.75 12L10.58 14.83L16.25 9.17004" stroke="#ef394e" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
                                    </svg>
                                    <span><?php _e('این محصول در سبد خرید شما است', 'bebanstore'); ?></span>
                        </div>
                            <?php endif; ?>
                        </div>
                        <div class="beban-wishlist-content">
                            <div class="beban-wishlist-details">
                                <h4><a href="<?php echo esc_url($product->get_permalink()); ?>" style="color: <?php echo esc_attr($title_color); ?>; font-size: <?php echo esc_attr($title_font_size); ?>px;"><?php echo esc_html($product->get_name()); ?></a></h4>
                                <div class="beban-wishlist-price">
                                    <span class="beban-price-wrapper" data-regular-price-size="<?php echo esc_attr($regular_price_font_size); ?>" data-regular-price-color="<?php echo esc_attr($regular_price_color); ?>" data-sale-price-size="<?php echo esc_attr($sale_price_font_size); ?>" data-sale-price-color="<?php echo esc_attr($sale_price_color); ?>">
                                        <?php echo $product->get_price_html(); ?>
                                    </span>
                                </div>
                            </div>
                            <div class="beban-wishlist-buttons">
                                <?php if ($product->is_type('variable')) : ?>
                                    <!-- Variable Product - Show "Select Options" button -->
                                    <a href="<?php echo esc_url($product->get_permalink()); ?>" 
                                       class="beban-wishlist-select-options-btn"
                                       style="background: <?php echo esc_attr($add_to_cart_bg); ?> !important; color: <?php echo esc_attr($add_to_cart_text); ?> !important; border-color: <?php echo esc_attr($add_to_cart_border); ?> !important;">
                                        <span style="display: inline-flex; align-items: center; gap: 6px;">
                                            <svg xmlns="http://www.w3.org/2000/svg" style="width: 24px; height: 24px; fill: <?php echo esc_attr($add_to_cart_icon); ?>; transform: scale(0.9);" width="24" height="24">
                                                <defs>
                                                    <symbol id="cartOff" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
                                                        <path fill-rule="evenodd" d="M20 4h2V2h-3a1 1 0 00-1 1v1H3a1 1 0 00-.995 1.1l1 10A1 1 0 004 16h15a1 1 0 001-1V4zm-2 17a2 2 0 110-4 2 2 0 010 4zM5 21a2 2 0 110-4 2 2 0 010 4zm13-7V6H4.105l.8 8H18z" clip-rule="evenodd"></path>
                                                    </symbol>
                                                </defs>
                                                <g>
                                                    <path fill-rule="evenodd" d="M20 4h2V2h-3a1 1 0 00-1 1v1H3a1 1 0 00-.995 1.1l1 10A1 1 0 004 16h15a1 1 0 001-1V4zm-2 17a2 2 0 110-4 2 2 0 010 4zM5 21a2 2 0 110-4 2 2 0 010 4zm13-7V6H4.105l.8 8H18z" clip-rule="evenodd"></path>
                                                </g>
                                            </svg>
                                            <?php _e('انتخاب گزینه', 'bebanstore'); ?>
                                        </span>
                                    </a>
                                <?php else : ?>
                                    <!-- Simple Product - Show "Add to Cart" button -->
                                    <button class="beban-wishlist-add-to-cart-btn" 
                                            data-product-id="<?php echo esc_attr($product->get_id()); ?>"
                                            style="background: <?php echo esc_attr($add_to_cart_bg); ?> !important; color: <?php echo esc_attr($add_to_cart_text); ?> !important; border-color: <?php echo esc_attr($add_to_cart_border); ?> !important;">
                                        <span style="display: inline-flex; align-items: center; gap: 6px;">
                                            <svg xmlns="http://www.w3.org/2000/svg" style="width: 24px; height: 24px; fill: <?php echo esc_attr($add_to_cart_icon); ?>; transform: scale(0.9);" width="24" height="24">
                                                <defs>
                                                    <symbol id="cartOff" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
                                                        <path fill-rule="evenodd" d="M20 4h2V2h-3a1 1 0 00-1 1v1H3a1 1 0 00-.995 1.1l1 10A1 1 0 004 16h15a1 1 0 001-1V4zm-2 17a2 2 0 110-4 2 2 0 010 4zM5 21a2 2 0 110-4 2 2 0 010 4zm13-7V6H4.105l.8 8H18z" clip-rule="evenodd"></path>
                                                    </symbol>
                                                </defs>
                                                <g>
                                                    <path fill-rule="evenodd" d="M20 4h2V2h-3a1 1 0 00-1 1v1H3a1 1 0 00-.995 1.1l1 10A1 1 0 004 16h15a1 1 0 001-1V4zm-2 17a2 2 0 110-4 2 2 0 010 4zM5 21a2 2 0 110-4 2 2 0 010 4zm13-7V6H4.105l.8 8H18z" clip-rule="evenodd"></path>
                                                </g>
                                            </svg>
                                            <?php _e('افزودن به سبد', 'bebanstore'); ?>
                                        </span>
                                    </button>
                                <?php endif; ?>
                                <button class="beban-wishlist-remove-btn" 
                                        data-product-id="<?php echo esc_attr($product->get_id()); ?>"
                                        style="background: <?php echo esc_attr($remove_btn_bg); ?> !important; border-color: <?php echo esc_attr($remove_btn_border); ?> !important;">
                                    <svg xmlns="http://www.w3.org/2000/svg" style="fill: <?php echo esc_attr($remove_btn_icon); ?>; transform: scale(0.9);" width="24" height="24">
                                            <defs>
                                                <symbol id="delete" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
                                                    <path fill-rule="evenodd" d="M8 2v2h8V2H8zM4 7V5h16v2H4zm13 1h2v11a3 3 0 01-3 3H8a3 3 0 01-3-3V8h2v11a1 1 0 001 1h8a1 1 0 001-1V8zm-6 0H9v10h2V8zm2 0h2v10h-2V8z" clip-rule="evenodd"></path>
                                                </symbol>
                                            </defs>
                                            <g>
                                                <path fill-rule="evenodd" d="M8 2v2h8V2H8zM4 7V5h16v2H4zm13 1h2v11a3 3 0 01-3 3H8a3 3 0 01-3-3V8h2v11a1 1 0 001 1h8a1 1 0 001-1V8zm-6 0H9v10h2V8zm2 0h2v10h-2V8z" clip-rule="evenodd"></path>
                                            </g>
                                        </svg>
                                </button>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
            
            <?php 
            // Show pagination at bottom if enabled and position is bottom or both
            if ($enable_pagination && $total_pages > 1 && in_array($pagination_position, array('bottom', 'both'))) {
                echo $this->render_pagination($current_page, $total_pages, $total_items, $per_page);
            }
            ?>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Get wishlist settings
     * 
     * @return array Wishlist settings
     */
    private function get_wishlist_settings() {
        $options = get_option('bebanstore_options', array());
        return isset($options['feature_settings']['wishlist']) ? $options['feature_settings']['wishlist'] : array();
    }

    /**
     * Render pagination for wishlist products
     * 
     * @param int $current_page Current page number
     * @param int $total_pages Total number of pages
     * @param int $total_items Total number of items
     * @param int $per_page Items per page
     * @return string Pagination HTML
     */
    private function render_pagination($current_page, $total_pages, $total_items, $per_page) {
        if ($total_pages <= 1) {
            return '';
        }

        $start_item = (($current_page - 1) * $per_page) + 1;
        $end_item = min($current_page * $per_page, $total_items);
        
        ob_start();
        ?>
        <div class="beban-wishlist-pagination">
            <div class="beban-wishlist-pagination-links">
                <?php
                // Calculate page range to show
                $range = 2;
                $start_page = max(1, $current_page - $range);
                $end_page = min($total_pages, $current_page + $range);
                
                // Show first page if not in range
                if ($start_page > 1) {
                    echo '<a href="' . esc_url(add_query_arg('wishlist_page', 1)) . '" class="beban-wishlist-page-link">1</a>';
                    if ($start_page > 2) {
                        echo '<span class="beban-wishlist-page-dots">...</span>';
                    }
                }
                
                // Show page numbers
                for ($i = $start_page; $i <= $end_page; $i++) {
                    $class = ($i == $current_page) ? 'beban-wishlist-page-link beban-wishlist-page-current' : 'beban-wishlist-page-link';
                    echo '<a href="' . esc_url(add_query_arg('wishlist_page', $i)) . '" class="' . $class . '">' . $i . '</a>';
                }
                
                // Show last page if not in range
                if ($end_page < $total_pages) {
                    if ($end_page < $total_pages - 1) {
                        echo '<span class="beban-wishlist-page-dots">...</span>';
                    }
                    echo '<a href="' . esc_url(add_query_arg('wishlist_page', $total_pages)) . '" class="beban-wishlist-page-link">' . $total_pages . '</a>';
                }
                ?>
            </div>
            
            <div class="beban-wishlist-pagination-info">
                <?php printf(
                    __('نمایش %1$d تا %2$d از %3$d محصول', 'bebanstore'),
                    $start_item,
                    $end_item,
                    $total_items
                ); ?>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }



    /**
     * Validate product for wishlist operations
     * 
     * @param int $product_id Product ID to validate
     * @return array Validation result with 'valid' boolean and 'message' string
     */
    private function validate_product($product_id) {
        try {
            // Check if product exists
            $product = wc_get_product($product_id);
            if (!$product) {
                return array(
                    'valid' => false,
                    'message' => __('این محصول موجود نیست.', 'bebanstore')
                );
            }

            // Check if product is published
            if ($product->get_status() !== 'publish') {
                return array(
                    'valid' => false,
                    'message' => __('این محصول در دسترس نیست.', 'bebanstore')
                );
            }

            // Check stock status
            if (!$product->is_in_stock()) {
                return array(
                    'valid' => false,
                    'message' => __('این محصول در حال حاضر ناموجود است.', 'bebanstore')
                );
            }

            return array(
                'valid' => true,
                'message' => ''
            );
            
        } catch (Exception $e) {
            return array(
                'valid' => false,
                'message' => __('خطا در بررسی محصول. لطفاً دوباره امتحان کنید.', 'bebanstore')
            );
        }
    }

    /**
     * Handle wishlist AJAX requests
     */
    public function handle_wishlist_ajax() {
        try {
            check_ajax_referer('beban_wishlist_nonce', 'nonce');

            // Get settings
            $settings = $this->get_wishlist_settings();
            
            // Check if login is required
            $require_login = isset($settings['require_login']) ? $settings['require_login'] : true;
            
            if ($require_login && !is_user_logged_in()) {
                $login_message = isset($settings['login_required_message']) ? $settings['login_required_message'] : __('لطفاً ابتدا وارد حساب کاربری خود شوید.', 'bebanstore');
                wp_send_json_error(array('message' => $login_message));
            }

            $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
            $action_type = isset($_POST['action_type']) ? sanitize_text_field($_POST['action_type']) : '';

            if (!$product_id || !in_array($action_type, array('add', 'remove'), true)) {
                wp_send_json_error(array('message' => __('درخواست نامعتبر است.', 'bebanstore')));
            }

            // Test with non-existent product ID
            if ($product_id == 999999) {
                wp_send_json_error(array('message' => __('این محصول موجود نیست.', 'bebanstore')));
            }

            // Validate product (only for add operations)
            if ($action_type === 'add') {
                $validation = $this->validate_product($product_id);
                if (!$validation['valid']) {
                    wp_send_json_error(array('message' => $validation['message']));
                }
            }

            $user_id = get_current_user_id();
            $wishlist_items = get_user_meta($user_id, 'beban_wishlist_items', true);
            
            if (!is_array($wishlist_items)) {
                $wishlist_items = array();
            }

            if ($action_type === 'add') {
                // Check maximum wishlist items limit
                $max_items = isset($settings['max_wishlist_items']) ? intval($settings['max_wishlist_items']) : 50;
                
                if ($max_items > 0 && count($wishlist_items) >= $max_items && !in_array($product_id, $wishlist_items)) {
                    $limit_message = isset($settings['max_wishlist_limit_message']) ? $settings['max_wishlist_limit_message'] : __('شما به حداکثر تعداد مجاز محصولات در لیست علاقه‌مندی‌ها رسیده‌اید. لطفاً ابتدا برخی محصولات را حذف کنید.', 'bebanstore');
                    wp_send_json_error(array('message' => $limit_message));
                }
                
                if (!in_array($product_id, $wishlist_items)) {
                    $wishlist_items[] = $product_id;
                    $update_result = update_user_meta($user_id, 'beban_wishlist_items', $wishlist_items);
                    
                    if ($update_result === false) {
                        wp_send_json_error(array('message' => __('خطا در ذخیره‌سازی اطلاعات. لطفاً دوباره امتحان کنید.', 'bebanstore')));
                    }
                    
                    $message = isset($settings['add_to_wishlist_message']) ? $settings['add_to_wishlist_message'] : __('محصول به لیست علاقه‌مندی‌ها اضافه شد.', 'bebanstore');
                    $status = 'added';
                } else {
                    $message = __('این محصول قبلاً در لیست علاقه‌مندی‌ها موجود است.', 'bebanstore');
                    $status = 'exists';
                }
            } elseif ($action_type === 'remove') {
                if (in_array($product_id, $wishlist_items)) {
                    $wishlist_items = array_diff($wishlist_items, array($product_id));
                    $update_result = update_user_meta($user_id, 'beban_wishlist_items', $wishlist_items);
                    
                    if ($update_result === false) {
                        wp_send_json_error(array('message' => __('خطا در حذف محصول. لطفاً دوباره امتحان کنید.', 'bebanstore')));
                    }
                    
                    $message = isset($settings['remove_from_wishlist_message']) ? $settings['remove_from_wishlist_message'] : __('محصول از لیست علاقه‌مندی‌ها حذف شد.', 'bebanstore');
                    $status = 'removed';
                } else {
                    $message = isset($settings['already_removed_message']) ? $settings['already_removed_message'] : __('این محصول قبلاً از لیست علاقه‌مندی‌ها حذف شده است.', 'bebanstore');
                    $status = 'already_removed';
                }
            }

            wp_send_json_success(array(
                'message' => $message,
                'status' => $status,
                'wishlist_count' => count($wishlist_items)
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array('message' => __('خطای غیرمنتظره رخ داد. لطفاً دوباره امتحان کنید.', 'bebanstore')));
        }
    }

    /**
     * Check stock status via AJAX
     */
    public function check_stock_status_ajax() {
        try {
            check_ajax_referer('beban_wishlist_nonce', 'nonce');
            
            $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
            
            if (!$product_id) {
                wp_send_json_error(array('message' => __('شناسه محصول نامعتبر است.', 'bebanstore')));
            }
            
            $product = wc_get_product($product_id);
            if (!$product) {
                wp_send_json_error(array('message' => __('این محصول موجود نیست.', 'bebanstore')));
            }
            
            $stock_status = $product->get_stock_status();
            $is_in_stock = $product->is_in_stock();
            
            wp_send_json_success(array(
                'stock_status' => $stock_status,
                'is_in_stock' => $is_in_stock,
                'message' => $is_in_stock ? __('محصول موجود است.', 'bebanstore') : __('این محصول در حال حاضر ناموجود است.', 'bebanstore')
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array('message' => __('خطا در بررسی وضعیت موجودی محصول.', 'bebanstore')));
        }
    }

    /**
     * AJAX handler for adding product to cart from wishlist
     */
    public function add_to_cart_from_wishlist_ajax() {
        try {
            check_ajax_referer('beban_wishlist_nonce', 'nonce');
            
            $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
            $quantity = isset($_POST['quantity']) ? intval($_POST['quantity']) : 1;
            
            if (!$product_id) {
                wp_send_json_error(array('message' => __('شناسه محصول نامعتبر است.', 'bebanstore')));
            }
            
            if ($quantity <= 0) {
                $quantity = 1;
            }
            
            // Validate product
            $product = wc_get_product($product_id);
            if (!$product) {
                wp_send_json_error(array('message' => __('این محصول موجود نیست.', 'bebanstore')));
            }
            
            // Check if product is in stock
            if (!$product->is_in_stock()) {
                wp_send_json_error(array('message' => __('این محصول در حال حاضر ناموجود است.', 'bebanstore')));
            }
            
            // Add to cart
            $cart_item_key = WC()->cart->add_to_cart($product_id, $quantity);
            
            if ($cart_item_key) {
                // Remove from wishlist after adding to cart
                $user_id = get_current_user_id();
                if ($user_id) {
                    $wishlist_items = get_user_meta($user_id, 'beban_wishlist_items', true);
                    
                    if (is_array($wishlist_items)) {
                        $wishlist_items = array_diff($wishlist_items, array($product_id));
                        $update_result = update_user_meta($user_id, 'beban_wishlist_items', $wishlist_items);
                        
                        if ($update_result === false) {
                            // Log error but don't fail the entire operation
                            error_log('BebanStore Wishlist: Failed to remove product from wishlist after adding to cart. Product ID: ' . $product_id . ', User ID: ' . $user_id);
                        }
                    }
                }
                
                // Get updated cart count
                $cart_count = WC()->cart->get_cart_contents_count();
                
                wp_send_json_success(array(
                    'message' => __('محصول به سبد خرید اضافه شد و از لیست علاقه‌مندی‌ها حذف شد.', 'bebanstore'),
                    'cart_count' => $cart_count,
                    'removed_from_wishlist' => true
                ));
            } else {
                wp_send_json_error(array('message' => __('خطا در افزودن محصول به سبد خرید.', 'bebanstore')));
            }
            
        } catch (Exception $e) {
            wp_send_json_error(array('message' => __('خطا در افزودن محصول به سبد خرید. لطفاً دوباره امتحان کنید.', 'bebanstore')));
        }
    }

    /**
     * Enqueue assets
     */
    public function enqueue_assets() {
        wp_enqueue_style(
            'bebanstore-wishlist',
            BEBANSTORE_PLUGIN_URL . 'assets/css/wishlist.css',
            array(),
            BEBANSTORE_VERSION
        );

        wp_enqueue_script(
            'bebanstore-wishlist',
            BEBANSTORE_PLUGIN_URL . 'assets/js/wishlist.js',
            array('jquery'),
            BEBANSTORE_VERSION,
            true
        );

        // Get settings for messages
        $settings = $this->get_wishlist_settings();
        
        wp_localize_script('bebanstore-wishlist', 'bebanWishlistAjax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('beban_wishlist_nonce'),
            'user_id' => get_current_user_id() ? get_current_user_id() : 0,
            'login_message' => isset($settings['login_required_message']) ? $settings['login_required_message'] : __('لطفاً ابتدا وارد حساب کاربری خود شوید.', 'bebanstore'),
            'add_message' => isset($settings['add_to_wishlist_message']) ? $settings['add_to_wishlist_message'] : __('محصول به لیست علاقه‌مندی‌ها اضافه شد.', 'bebanstore'),
            'remove_message' => isset($settings['remove_from_wishlist_message']) ? $settings['remove_from_wishlist_message'] : __('محصول از لیست علاقه‌مندی‌ها حذف شد.', 'bebanstore'),
            'already_removed_message' => isset($settings['already_removed_message']) ? $settings['already_removed_message'] : __('این محصول قبلاً از لیست علاقه‌مندی‌ها حذف شده است.', 'bebanstore'),
            'empty_message' => isset($settings['empty_wishlist_message']) ? $settings['empty_wishlist_message'] : __('لیست علاقه‌مندی‌های شما خالی است.', 'bebanstore'),
            'animation_type' => isset($settings['animation_type']) ? $settings['animation_type'] : 'dots',
            // استایل دکمه افزودن به سبد - حالت عادی
            'add_to_cart_normal_bg' => isset($settings['add_to_cart_normal_bg']) ? $settings['add_to_cart_normal_bg'] : '#ffffff',
            'add_to_cart_normal_text' => isset($settings['add_to_cart_normal_text']) ? $settings['add_to_cart_normal_text'] : '#ef394e',
            'add_to_cart_normal_icon' => isset($settings['add_to_cart_normal_icon']) ? $settings['add_to_cart_normal_icon'] : '#ef394e',
            'add_to_cart_normal_border' => isset($settings['add_to_cart_normal_border']) ? $settings['add_to_cart_normal_border'] : '#ef394e',
            // استایل دکمه افزودن به سبد - حالت لودینگ
            'add_to_cart_loader_overlay_color' => isset($settings['add_to_cart_loader_overlay_color']) ? $settings['add_to_cart_loader_overlay_color'] : '#ffffff',
            'add_to_cart_loader_animation_color' => isset($settings['add_to_cart_loader_animation_color']) ? $settings['add_to_cart_loader_animation_color'] : '#EF394F',
            'add_to_cart_loader_border_color' => isset($settings['add_to_cart_loader_border_color']) ? $settings['add_to_cart_loader_border_color'] : '#ef394e',
            // استایل دکمه حذف - حالت عادی
            'remove_btn_normal_bg' => isset($settings['remove_btn_normal_bg']) ? $settings['remove_btn_normal_bg'] : '#ffffff',
            'remove_btn_normal_icon' => isset($settings['remove_btn_normal_icon']) ? $settings['remove_btn_normal_icon'] : '#81858B',
            'remove_btn_normal_border' => isset($settings['remove_btn_normal_border']) ? $settings['remove_btn_normal_border'] : '#E0E0E2',
            // استایل دکمه حذف - حالت لودینگ
            'remove_loader_overlay_color' => isset($settings['remove_loader_overlay_color']) ? $settings['remove_loader_overlay_color'] : '#ffffff',
            'remove_loader_animation_color' => isset($settings['remove_loader_animation_color']) ? $settings['remove_loader_animation_color'] : '#81858B',
            'remove_loader_border_color' => isset($settings['remove_loader_border_color']) ? $settings['remove_loader_border_color'] : '#E0E0E2',
            // استایل دکمه قلب علاقه‌مندی - حالت عادی (قلب خالی)
            'wishlist_btn_normal_fill' => isset($settings['wishlist_btn_normal_fill']) ? $settings['wishlist_btn_normal_fill'] : 'none',
            'wishlist_btn_normal_stroke' => isset($settings['wishlist_btn_normal_stroke']) ? $settings['wishlist_btn_normal_stroke'] : '#A9ABAD',
            // استایل دکمه قلب علاقه‌مندی - حالت لودینگ
            'wishlist_btn_loader_animation_color' => isset($settings['wishlist_btn_loader_animation_color']) ? $settings['wishlist_btn_loader_animation_color'] : '#292D32',
            // استایل دکمه قلب علاقه‌مندی - حالت فعال (قلب پر)
            'wishlist_btn_active_fill' => isset($settings['wishlist_btn_active_fill']) ? $settings['wishlist_btn_active_fill'] : '#292D32',
            'wishlist_btn_active_stroke' => isset($settings['wishlist_btn_active_stroke']) ? $settings['wishlist_btn_active_stroke'] : '#292D32',
            'i18n' => array(
                'adding' => __('در حال افزودن...', 'bebanstore'),
                'removing' => __('در حال حذف...', 'bebanstore')
            )
        ));
    }



    /**
     * Get feature title
     */
    public function get_title() {
        return $this->feature_name;
    }

    /**
     * Get feature description
     */
    public function get_description() {
        return $this->feature_description;
    }

    /**
     * Get shortcode
     */
    public function get_shortcode() {
        return '[beban_wishlist_icon]';
    }

    /**
     * Render show shortcode button
     */
    public function render_show_shortcode_button() {
        ?>
        <button type="button" 
                class="bebanstore-show-shortcode button" 
                data-feature-id="<?php echo esc_attr($this->feature_id); ?>"
                data-feature-title="<?php echo esc_attr($this->feature_name); ?>"
                data-shortcode="[beban_wishlist_icon]"
                data-description="<?php echo esc_attr($this->usage_instructions); ?>">
            <?php _e('نمایش شورت‌کدها', 'bebanstore'); ?>
        </button>
        <?php
    }

    /**
     * Render shortcode modal content
     */
    public function render_shortcode($atts, $content = NULL) {
        ?>
        <div class="bebanstore-shortcode-modal">
            <h3><?php _e('شورت‌کدهای سیستم علاقه‌مندی‌ها', 'bebanstore'); ?></h3>
            
            <div class="bebanstore-shortcode-section">
                <h4><?php _e('دکمه علاقه‌مندی', 'bebanstore'); ?></h4>
                <p><?php _e('برای نمایش دکمه قلب روی محصولات از این شورت‌کد استفاده کنید:', 'bebanstore'); ?></p>
                <div class="bebanstore-shortcode-example">
                    <code>[beban_wishlist_icon]</code>
                    <button class="bebanstore-copy-btn" data-clipboard-text="[beban_wishlist_icon]">
                        <?php _e('کپی', 'bebanstore'); ?>
                    </button>
                </div>
                <p class="bebanstore-shortcode-note">
                    <strong><?php _e('نکته:', 'bebanstore'); ?></strong> 
                    <?php _e('این شورت‌کد را در صفحه محصول یا حلقه محصولات قرار دهید.', 'bebanstore'); ?>
                </p>
            </div>

            <div class="bebanstore-shortcode-section">
                <h4><?php _e('لیست علاقه‌مندی‌ها', 'bebanstore'); ?></h4>
                <p><?php _e('برای نمایش لیست محصولات علاقه‌مندی از این شورت‌کد استفاده کنید:', 'bebanstore'); ?></p>
                <div class="bebanstore-shortcode-example">
                    <code>[beban_wishlist_products]</code>
                    <button class="bebanstore-copy-btn" data-clipboard-text="[beban_wishlist_products]">
                        <?php _e('کپی', 'bebanstore'); ?>
                    </button>
                </div>
                <p class="bebanstore-shortcode-note">
                    <strong><?php _e('نکته:', 'bebanstore'); ?></strong> 
                    <?php _e('این شورت‌کد را در صفحه علاقه‌مندی‌ها یا هر جای دیگر قرار دهید.', 'bebanstore'); ?>
                </p>
            </div>

            <div class="bebanstore-shortcode-section">
                <h4><?php _e('مثال استفاده', 'bebanstore'); ?></h4>
                <div class="bebanstore-shortcode-example">
                    <pre><code>&lt;div class="product-card"&gt;
    &lt;img src="product-image.jpg" alt="Product"&gt;
    &lt;h3&gt;نام محصول&lt;/h3&gt;
    &lt;p&gt;قیمت: 100,000 تومان&lt;/p&gt;
    [beban_wishlist_icon]
&lt;/div&gt;</code></pre>
                </div>
            </div>

            <div class="bebanstore-shortcode-section">
                <h4><?php _e('ویژگی‌ها', 'bebanstore'); ?></h4>
                <ul>
                    <li><?php _e('افزودن/حذف محصولات با AJAX', 'bebanstore'); ?></li>
                    <li><?php _e('انیمیشن‌های زیبا', 'bebanstore'); ?></li>
                    <li><?php _e('اعلان‌های Toastr', 'bebanstore'); ?></li>
                    <li><?php _e('Responsive design', 'bebanstore'); ?></li>
                    <li><?php _e('امنیت بالا با nonce', 'bebanstore'); ?></li>
                </ul>
            </div>
        </div>
        <?php
    }

    /**
     * Add wishlist endpoint
     */
    public function add_wishlist_endpoint() {
        add_rewrite_endpoint('wishlist_products', EP_ROOT | EP_PAGES);
    }

    /**
     * Register query var for wishlist endpoint so WooCommerce marks tab as active
     * @param array $vars
     * @return array
     */
    public function register_wishlist_query_var($vars) {
        $vars['wishlist_products'] = 'wishlist_products';
        return $vars;
    }

    /**
     * Maybe flush rewrite rules
     */
    public function maybe_flush_rewrite_rules() {
        // Only flush if the option is set
        $options = get_option('bebanstore_options', array());
        $settings = isset($options['feature_settings']['wishlist']) ? $options['feature_settings']['wishlist'] : array();
        $add_account_tab = isset($settings['add_account_tab']) ? $settings['add_account_tab'] : false;
        
        if ($add_account_tab && get_option('bebanstore_wishlist_flush_needed', false)) {
            flush_rewrite_rules();
            delete_option('bebanstore_wishlist_flush_needed');
        }
    }

    /**
     * Check if wishlist settings were updated
     */
    public function check_wishlist_settings_update($old_value, $value, $option) {
        // Check if add_account_tab setting changed
        $old_settings = isset($old_value['feature_settings']['wishlist']) ? $old_value['feature_settings']['wishlist'] : array();
        $new_settings = isset($value['feature_settings']['wishlist']) ? $value['feature_settings']['wishlist'] : array();
        
        $old_add_tab = isset($old_settings['add_account_tab']) ? $old_settings['add_account_tab'] : false;
        $new_add_tab = isset($new_settings['add_account_tab']) ? $new_settings['add_account_tab'] : false;
        
        if ($old_add_tab !== $new_add_tab) {
            update_option('bebanstore_wishlist_flush_needed', true);
        }
    }

    /**
     * Add wishlist tab to account menu
     */
    public function add_wishlist_account_tab($items) {
        // Get settings
        $options = get_option('bebanstore_options', array());
        $settings = isset($options['feature_settings']['wishlist']) ? $options['feature_settings']['wishlist'] : array();
        $tab_title = isset($settings['account_tab_title']) ? $settings['account_tab_title'] : __('علاقه‌مندی‌ها', 'bebanstore');
        
        // Insert wishlist tab after orders
        $new_items = array();
        foreach ($items as $key => $value) {
            $new_items[$key] = $value;
            if ($key === 'orders') {
                $new_items['wishlist_products'] = $tab_title;
            }
        }
        
        return $new_items;
    }

    /**
     * Wishlist account tab content
     */
    public function wishlist_account_tab_content() {
        // Enqueue assets for this page
        $this->enqueue_assets();
        
        // Display wishlist products
        echo '<div class="beban-wishlist-account-page">';
        echo '<div class="beban-wishlist-account-header">
            <h2>' . __('لیست علاقه‌مندی‌های شما', 'bebanstore') . '</h2>
            <p>' . __('در این بخش می‌توانید محصولات مورد علاقه خود را مشاهده و مدیریت کنید.', 'bebanstore') . '</p>
        </div>';
        echo do_shortcode('[beban_wishlist_products]');
        echo '</div>';
    }
} 