<?php

/*
Plugin Name: Deactivate Plugins per Page
Description: Deactivate plugins on individual pages to speed up your site.
Author: Nikolay Nikolov
Author URI: https://nikolaydev.com/
Plugin URI: https://codecanyon.net/item/deactivate-plugins-per-page-improve-wordpress-performance/23801359
Update URI: https://codecanyon.net/item/deactivate-plugins-per-page-improve-wordpress-performance/23801359
Text Domain: dppp
Domain Path: /abzarwp/languages
Version: 1.17.0
*/

define( "DPPP_DEACTIVATE_PLUGINS_PER_PAGE_VERSION", "1.17.0" );

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// On plugin activation it creates a database table and copies a must use plugin file in the mu-plugins folder
register_activation_hook( __FILE__, 'dppp_on_activate_plugin' );

/*
 * On plugin deactivation, if it is not a multisite, it removes the must use plugin file, since must use plugins cannot be deactivated in any other way.
 * In a multisite we have to keep it, in case the plugin is active on another site from the network.
 */
register_deactivation_hook( __FILE__, 'dppp_on_deactivate_plugin' );

// On plugin deletion we remove the must use plugin file.
register_uninstall_hook( __FILE__, 'dppp_on_delete_plugin' );

// Adds the CSS and JS files on the front-end, needed to show the debug box
add_action( 'wp_enqueue_scripts', 'dppp_register_debug_assets' );

// Adds the CSS and JS files on the back-end, needed to show the debug box
add_action( 'admin_enqueue_scripts', 'dppp_register_debug_assets' );

// Adds the HTML code in the footer for the debug box on the front-end
add_action( 'wp_footer', 'dppp_add_debug_box_to_footer', 999999 );

// Adds the HTML code in the footer for the debug box on the back-end
add_action( 'admin_footer', 'dppp_add_debug_box_to_footer', 999999 );

// Adds the HTML code in the footer for the debug box on the customizer back-end page
add_action( 'customize_controls_print_footer_scripts', 'dppp_add_debug_box_to_footer', 999999 );

// Adds the Settings action link on the plugins admin page
add_filter( 'plugin_action_links', 'dppp_add_settings_plugin_action_link', 10, 2 );

// Adds the admin menu element for the plugin settings page
add_action( 'admin_menu', 'dppp_add_admin_options_menu' );

// Handles the ajax request for deleting a rule
add_action( 'wp_ajax_dppp_delete_rule_action', 'dppp_delete_rule_callback' );

// Handles the ajax request for pausing a rule
add_action( 'wp_ajax_dppp_pause_rule_action', 'dppp_pause_rule_callback' );

// Handles the ajax request for starting a rule
add_action( 'wp_ajax_dppp_start_rule_action', 'dppp_start_rule_callback' );

// Handles the ajax request for refreshing the plugin rules layer displayed for a given plugin
add_action( 'wp_ajax_dppp_update_rules_content_action', 'dppp_update_rules_content_callback' );
                                                             
// Handles the ajax request for displaying the box with the form for adding a new plugin rule
add_action( 'wp_ajax_dppp_add_new_rule_action', 'dppp_add_new_rule_callback' );

// Handles the ajax request for displaying the box with the form for adding a new plugin group
add_action( 'wp_ajax_dppp_add_new_plugin_group_action', 'dppp_add_new_plugin_group_callback' );

// Handles the ajax request for displaying the box with the explanation of the rule
add_action( 'wp_ajax_dppp_explain_rule_action', 'dppp_explain_rule_callback' );

// Handles the ajax request for saving the plugin settings
add_action( 'wp_ajax_dppp_save_settings_action', 'dppp_save_settings_callback' );

// Handles the ajax request for saving the state of the debug box (to be fully shown or minimized)
add_action( 'wp_ajax_dppp_save_debug_box_action', 'dppp_save_debug_box_callback' );

// Handles the ajax request for adding a new rule to the database
add_action( 'wp_ajax_dppp_save_rule_data_action', 'dppp_save_rule_data_callback' );

// Handles the ajax request for adding a plugin group to the database
add_action( 'wp_ajax_dppp_save_plugin_group_data_action', 'dppp_save_plugin_group_data_callback' );

// Handles the ajax request for deleting a plugin group and all its rules
add_action( 'wp_ajax_dppp_delete_plugin_group_action', 'dppp_delete_plugin_group_callback' );

// Handles the ajax request for updating the plugin groups section with fresh data
add_action( 'wp_ajax_dppp_update_plugin_groups_action', 'dppp_update_plugin_groups_callback' );

// Handles the ajax request for global actions (delete all rules, pause all rules, start all rules)
add_action( 'wp_ajax_dppp_global_actions_action', 'dppp_global_actions_callback' );

// Handles the ajax request for plugin or group actions (delete rules, pause rules, start rules, deactivate on front-end, deactivate on back-end)
add_action( 'wp_ajax_dppp_plugin_or_group_actions_action', 'dppp_plugin_or_group_actions_callback' );

// Handles the ajax request for displaying the box for editing an existing rule
add_action( 'wp_ajax_dppp_edit_rule_action', 'dppp_edit_rule_callback' );

// Handles the ajax request for saving the rule to the database after editing it
add_action( 'wp_ajax_dppp_save_edit_rule_data_action', 'dppp_save_edit_rule_data_callback' );

// Handles the ajax request for editing a plugin group
add_action( 'wp_ajax_dppp_edit_plugin_group_action', 'dppp_edit_plugin_group_callback' );

// Handles the ajax request for saving the group data after editing a plugin group
add_action( 'wp_ajax_dppp_save_edit_plugin_group_data_action', 'dppp_save_edit_plugin_group_data_callback' );

// Handles the ajax request for showing and hiding a section
add_action( 'wp_ajax_dppp_show_hide_section_action', 'dppp_show_hide_section_callback' );

// Handles the ajax request for performing quick plugin actions from the debug box
add_action( 'wp_ajax_dppp_debug_quick_action_action', 'dppp_debug_quick_action_callback' );

// An ajax request that gets the plugin icon address and stores it in the database
add_action( 'wp_ajax_dppp_get_plugin_icon_action', 'dppp_get_plugin_icon_callback' );

// An ajax request that makes this device local (creates a cookie)
add_action( 'wp_ajax_dppp_make_device_local_action', 'dppp_make_device_local_callback' );

// An ajax request that removes this local device (removes a cookie)
add_action( 'wp_ajax_dppp_remove_local_device_action', 'dppp_remove_local_device_action_callback' );

// An ajax request that pastes the copied rules adding them to the chosen plugin or group
add_action( 'wp_ajax_dppp_paste_rules_action', 'dppp_paste_rules_callback' );

// When the permalink structure is updated, we update the URI values for the page type rules we have
add_action( 'admin_init', 'dppp_permalinks_updated' );

// When a page is updated, we update the URI values of the rules
add_action( 'post_updated', 'dppp_update_page_rules', 10, 3 );

// When a page is trashed, we update the URI values of the rules
add_action( 'trashed_post', 'dppp_refresh_rules_after_trashed' );

// When the option that says which page to be a front page is created, we update the URI values of the page type rules
add_action( 'add_option_page_on_front', 'dppp_add_option_front_page_refresh', 10, 2 );

// When the option that says which page to be a blog page is created, we update the URI values of the page type rules
add_action( 'add_option_page_for_posts', 'dppp_add_option_front_page_refresh', 10, 2 );

// When the option that says which page to be a front page is updated, we update the URI values of the page type rules
add_action( 'update_option_page_on_front', 'dppp_update_option_front_page_refresh', 10, 2 );

// When the option that says which page to be a blog page is updated, we update the URI values of the page type rules
add_action( 'update_option_page_for_posts', 'dppp_update_option_front_page_refresh', 10, 2 );

// Updates the database and mu-plugin file after a new version of the plugin is uploaded (and the site is visited)
add_action( 'init', 'dppp_update_database_and_mu_plugin' );

// Loads the plugin's translated strings from the languages folder inside the plugin folder
add_action( 'init', 'dppp_load_plugin_textdomain' );

// Deletes the admin menu cache on plugin activation, since this adds menus usually
add_action( 'activated_plugin', 'dppp_delete_cache_on_plugin_activate', 10, 2 );

// Deletes the admin menu cache on plugin deactivation, since this removes menus usually
add_action( 'deactivated_plugin', 'dppp_delete_cache_on_plugin_deactivate', 10, 2 );

// Deletes the admin menu cache on theme switch, since this creates or removes menus usually
add_action( 'after_switch_theme', 'dppp_delete_cache_on_theme_switch', 10, 2 );

// Loads the styles and scripts needed for the admin menu cache feature
add_action( 'admin_enqueue_scripts', 'dppp_register_admin_menus_cache_assets' );

// Here we decide if we should output the admin menu cache later or a message or do nothing
add_action( 'admin_init', 'dppp_prepare_cached_admin_menus' );

// Creates or recreates the admin menu cache
add_action( 'adminmenu', 'dppp_create_admin_menu_cache' );

// Fix database table detection by WP-Optimize
add_filter( 'wpo_get_tables_data', 'dppp_fix_wp_optimize_table_detection' );

// Adds the hook to change edit post links and add post type parameter
add_action( 'init', 'dppp_add_hook_edit_post_link' );

// Loads the plugin's translated strings from the languages folder inside the plugin folder
function dppp_load_plugin_textdomain() {
    load_plugin_textdomain( 'dppp', FALSE, basename( dirname( __FILE__ ) ) . '/languages' );
}

// Adds the hook to change edit post links and add post type parameter
function dppp_add_hook_edit_post_link() {
    if ( dppp_get_option( 'dppp-edit-post-parameter' ) === 'on' ) {
        add_filter( 'admin_url', 'dppp_admin_url_post_type_parameter', 9999999 );
    }
}

/**
 * Adds a custom parameter with post type to edit post links on the back-end so we can target these admin pages with rules
 * @param string $url
 * @return string
 */
function dppp_admin_url_post_type_parameter( $url ) {
    if ( strpos( $url, '/post.php?post=' ) !== false ) {
        $parts = parse_url( $url );
        if ( is_array( $parts ) && isset( $parts['query'] ) ) {
            $query_string = $parts['query'];
            parse_str( $query_string, $parameters );
            if ( is_array( $parameters ) && isset( $parameters['post'] ) && function_exists( 'get_post_type' ) ) {
                $post_type = get_post_type( $parameters['post'] );
                return add_query_arg( 'dppp_post_type', $post_type, $url );
            }
        }
    }
    return $url;
}

// Makes changes to the database and the mu plugin after a new version of the plugin has been uploaded (if needed)
function dppp_update_database_and_mu_plugin() {

    // The most recent version of the plugin that has ever been activated in this installation
    $plugin_version_in_database = get_option( 'dppp-plugin-version-option' );

    // If we do not have data in the database about the version, we will check the file, since sometimes database data is incorrect or corrupted
    if ( $plugin_version_in_database === false ) {
        $upload_directory = wp_upload_dir();
        $file_path = $upload_directory['basedir'] . '/dppp-plugin-data/db-version.php';
        if ( file_exists( $file_path ) && is_readable( $file_path ) ) {
            $file_data = file_get_contents( $file_path );
            if ( false !== $file_data ) {
                $plugin_version_in_database = str_replace( "<?php // ", "", $file_data );
            }
        }
    }

    // If there is no data for the version in the database and file, it is 1.0.0
    if ( $plugin_version_in_database === false ) {
        $plugin_version_in_database = '1.0.0';
    }

    // If the version in the database does not match the version in this current file, we need to update
    if ( $plugin_version_in_database !== DPPP_DEACTIVATE_PLUGINS_PER_PAGE_VERSION ) {

        // These are all the versions that require database update
        $version_updates = Array( '1.1.0', '1.2.0', '1.3.0', '1.13.0', '1.13.1', '1.13.2', '1.15.0' );

        /**
         * For each version we compare the one in the database and the current one from the array, and if the one in the database is lower
         * we run the update function for the current one from the array. This way if the database version is for example 1.1.0 we will update first to 1.2.0
         * and then to 1.3.0 and so on until the current most recent version is reached.
         * Now to add a new update all I need to do is add the version in the array and add the changes to the dppp_update_to_version function. If a new
         * version is released that does not need a database update, I must NOT add it to the array. Then only the version will be updated in the database
         * and no other changes will be made.
         */
        foreach ( $version_updates as $version_update ) {
            if ( version_compare( $plugin_version_in_database, $version_update, '<' ) ) {
                dppp_update_to_version( $version_update );
            }
        }

        // Changes the database version to the current one
        update_option( 'dppp-plugin-version-option', DPPP_DEACTIVATE_PLUGINS_PER_PAGE_VERSION );

        // We replace the mu plugin with the new file from this version
        dppp_update_mu_plugin();

        // We store the database version in a file too, since sometimes the database data could be incorrect due to various problems
        $upload_directory = wp_upload_dir();
        $directory_path = $upload_directory['basedir'] . '/dppp-plugin-data/';
        if ( ! file_exists( $directory_path ) ) {
            mkdir( $directory_path, 0755, true );
        }
        if ( is_writable( $directory_path ) ) {
            $file_path = $directory_path . 'db-version.php';
            if ( ! file_exists( $file_path ) || filesize( $file_path ) < 1000000 ) {
                file_put_contents( $file_path, "<?php // " . dppp_sanitize_version( DPPP_DEACTIVATE_PLUGINS_PER_PAGE_VERSION ), LOCK_EX );
            }
        }
    }
}

/**
 * Makes the required changes to update the database data to a specific plugin version.
 * @param string $version
 */
function dppp_update_to_version( $version ) {

    if ( '1.1.0' === $version ) {

        // Create the new database
        dppp_create_database_table();

        // This will run only if you ever used version 1.0.0. If the old database from version 1.0.0 was created, we need to move the rules to the new one
        if ( get_option( 'dppp-table-created' ) === 'yes' ) {
            global $wpdb;
            if ( $wpdb->get_var( "SHOW TABLES LIKE '{$wpdb->prefix}dppp_plugin_rules'" ) === "{$wpdb->prefix}dppp_plugin_rules" ) {
                $results_multi_array = $wpdb->get_results( "SELECT status, plugin_file, deactivation_type, uri_type, page_id, "
                    . "uri_condition, uri_value, devices, unix_time_added FROM {$wpdb->prefix}dppp_plugin_rules WHERE 1", ARRAY_A );
                if ( ! empty( $results_multi_array ) ) {
                    foreach ( $results_multi_array as $results ) {
                        $wpdb->insert(
                        	"{$wpdb->prefix}dppp_plugin_deactivation_rules",
                        	array(
                        		'status' => $results['status'],
                        		'plugin_file' => $results['plugin_file'],
                        		'deactivation_type' => $results['deactivation_type'],
                        		'uri_type' => $results['uri_type'],
                        		'page_ids' => $results['page_id'],
                        		'uri_condition' => $results['uri_condition'],
                        		'uri_value' => $results['uri_value'],
                        		'devices' => $results['devices'],
                        		'unix_time_added' => $results['unix_time_added'],
                        	),
                        	array(
                        		'%s',
                        		'%s',
                        		'%s',
                        		'%s',
                        		'%s',
                        		'%s',
                        		'%s',
                        		'%s',
                        		'%d',
                        	)
                        );
                    }
                }
            }
        }
    } elseif ( '1.2.0' === $version ) {
        global $wpdb;
        $wpdb->query( "ALTER TABLE {$wpdb->prefix}dppp_plugin_deactivation_rules MODIFY plugin_file longtext DEFAULT ''" );
    } elseif ( '1.3.0' === $version ) {
        global $wpdb;
        $wpdb->query( "ALTER TABLE {$wpdb->prefix}dppp_plugin_deactivation_rules MODIFY uri_condition longtext DEFAULT ''" );
    } elseif ( '1.13.0' === $version ) {
        global $wpdb;

        // If it does not exist, we create the column for notes
        $results = $wpdb->query( "SHOW COLUMNS FROM {$wpdb->prefix}dppp_plugin_deactivation_rules LIKE 'note'" );
        if ( ! $results ) {
            $wpdb->query( "ALTER TABLE {$wpdb->prefix}dppp_plugin_deactivation_rules ADD note longtext AFTER devices" );
        }
    } elseif ( '1.13.1' === $version ) {
        global $wpdb;

        // If it does not exist, we create the column for notes
        $results = $wpdb->query( "SHOW COLUMNS FROM {$wpdb->prefix}dppp_plugin_deactivation_rules LIKE 'note'" );
        if ( ! $results ) {
            $wpdb->query( "ALTER TABLE {$wpdb->prefix}dppp_plugin_deactivation_rules ADD note longtext AFTER devices" );
        }
    } elseif ( '1.13.2' === $version ) {

        // We will add a URI parameter to ignore so the Nitro Pack plugin bot generates correct cache following our rules
        $chosen_parameters = dppp_get_option( 'dppp-chosen-uri-parameters' );
        if ( ! empty( $chosen_parameters ) && is_string( $chosen_parameters ) && strpos( $chosen_parameters, 'ignorenitro' ) === false
            && substr( dppp_strip_whitespace( $chosen_parameters ), -1 ) !== ',' ) {
            update_option( 'dppp-chosen-uri-parameters', $chosen_parameters . ', ignorenitro' );
        }
    } elseif ( '1.15.0' === $version ) {
        global $wpdb;

        // If it does not exist, we create the column for custom function condition
        $results = $wpdb->query( "SHOW COLUMNS FROM {$wpdb->prefix}dppp_plugin_deactivation_rules LIKE 'custom_function'" );
        if ( ! $results ) {
            $wpdb->query( "ALTER TABLE {$wpdb->prefix}dppp_plugin_deactivation_rules ADD custom_function longtext AFTER devices" );
        }
    }
}

// On plugin activation it creates a database table and copies a must use plugin file in the mu-plugins folder
function dppp_on_activate_plugin() {
    dppp_create_database_table();
    dppp_check_and_add_mu_plugin();
    add_option( 'dppp-plugin-version-option', DPPP_DEACTIVATE_PLUGINS_PER_PAGE_VERSION );

    /*
     * We store the database version in a file too, since sometimes the database data could be incorrect due to various problems.
     * But here on activation we do it only if it does not exists.
     */
    $upload_directory = wp_upload_dir();
    $directory_path = $upload_directory['basedir'] . '/dppp-plugin-data/';
    if ( ! file_exists( $directory_path ) ) {
        mkdir( $directory_path, 0755, true );
    }
    if ( is_writable( $directory_path ) ) {
        $file_path = $directory_path . 'db-version.php';
        if ( ! file_exists( $file_path ) ) {
            file_put_contents( $file_path, "<?php // " . dppp_sanitize_version( DPPP_DEACTIVATE_PLUGINS_PER_PAGE_VERSION ), LOCK_EX );
        }
    }
}

/*
 * On plugin deactivation, if it is not a multisite, it removes the must use plugin file, since it cannot be deactivated in any other way.
 * In a multisite we have to keep it, in case the plugin is active on another site from the network.
 */
function dppp_on_deactivate_plugin() {
    if ( ! is_multisite() ) {
        dppp_remove_mu_plugin();
    }
}

// On plugin deletion we remove the must use plugin file. If the settings allow it, we also remove the rules and settings.
function dppp_on_delete_plugin() {

    // We remove the must use plugin file
    dppp_remove_mu_plugin();

    if ( ! is_multisite() && dppp_get_option( 'dppp-completely-remove' ) === 'on' ) {
        global $wpdb;

        // We remove the database tables (one of them was used only in version 1.0.0 of the plugin)
        $wpdb->query( "DROP TABLE IF EXISTS {$wpdb->prefix}dppp_plugin_deactivation_rules" );
        $wpdb->query( "DROP TABLE IF EXISTS {$wpdb->prefix}dppp_plugin_rules" );

        // We remove some of our options which name starts with a specific string
        $results = $wpdb->get_results( "SELECT option_name FROM {$wpdb->prefix}options WHERE option_name LIKE '%dppp-debug-box-state%' "
            . "OR option_name LIKE '%dppp-plugin-icon-%' OR option_name LIKE '%dppp-cache-%'", ARRAY_A );
        if ( ! empty( $results ) ) {
            foreach ( $results as $result ) {
                if ( strpos( $result['option_name'], 'dppp-debug-box-state' ) !== false || strpos( $result['option_name'], 'dppp-plugin-icon-' ) !== false
                    || strpos( $result['option_name'], 'dppp-cache-' ) !== false ) {
                    delete_option( $result['option_name'] );
                }
            }
        }

        // We remove some options
        delete_option( 'dppp-database-table-created' );
        delete_option( 'dppp-completely-remove' );
        delete_option( 'dppp-inactive-plugins' );
        delete_option( 'dppp-table-created' );
        delete_option( 'dppp-plugin-version-option' );
        delete_option( 'dppp-debug-mode' );
        delete_option( 'dppp-plugin-groups' );
        delete_option( 'dppp-hide-group-plugins' );
        delete_option( 'dppp-groups-first' );
        delete_option( 'dppp-trailing-slash' );
        delete_option( 'dppp-affect-back-end' );
        delete_option( 'dppp-uri-parameters' );
        delete_option( 'dppp-uri-parameters-custom' );
        delete_option( 'dppp-chosen-uri-parameters' );
        delete_option( 'dppp-section-show-hide-settings' );
        delete_option( 'dppp-section-show-hide-active-plugins' );
        delete_option( 'dppp-section-show-hide-inactive-plugins' );
        delete_option( 'dppp-section-show-hide-plugin-groups' );
        delete_option( 'dppp-plugin-icons' );
        delete_option( 'dppp-local-mode' );
        delete_option( 'dppp-rules-first' );
        delete_option( 'dppp-hide-plugins-in-group' );
        delete_option( 'dppp-priority' );
        delete_option( 'dppp-affect-visual-editors' );
        delete_option( 'dppp-edit-post-parameter' );
        delete_option( 'dppp-admin-menu-cache-time' );

        // We remove the folder and file that stores the version of the plugin
        $upload_directory = wp_upload_dir();
        $directory_path = $upload_directory['basedir'] . '/dppp-plugin-data/';
        $file_path = $directory_path . 'db-version.php';
        if ( file_exists( $file_path ) && is_writable( $file_path ) ) {
            @unlink( $file_path );
        }
        if ( file_exists( $directory_path ) && is_writable( $directory_path ) ) {
            @rmdir( $directory_path );
        }
    }
}

// Adds the CSS and JS files on the front-end, needed to show the debug box
function dppp_register_debug_assets() {
    if ( current_user_can( 'manage_options' ) && dppp_get_option( 'dppp-debug-mode' ) !== 'off' ) {
        if ( ( is_admin() && dppp_skip_this_admin_page() ) || ( is_multisite() && is_network_admin() && dppp_skip_this_network_admin_page() )
            || ( is_admin() && dppp_get_option( 'dppp-debug-mode' ) === 'front' ) || ( ! is_admin() && dppp_get_option( 'dppp-debug-mode' ) === 'back' ) ) {
            return;
        }
        if ( isset( $_GET['elementor-preview'] ) || isset( $_GET['et_fb'] ) || isset( $_GET['vc_editable'] ) ) {
            return;
        }
        wp_register_script( 'dppp-debug-script', dppp_plugin_dir_url() . 'scripts/debug.js', array( 'jquery' ),
            DPPP_DEACTIVATE_PLUGINS_PER_PAGE_VERSION, false );
        $localize = array(
            'ajaxNonce' => wp_create_nonce( 'dppp_debug_ajax_nonce' ),
            'ajaxurl' => admin_url( 'admin-ajax.php' ),
            'errorResponse' => esc_js( __( 'Error: We got an unexpected response.', 'dppp' ) ),
            'errorAccess' => esc_js( __( 'Error: You do not have sufficient permissions to make this request.', 'dppp' ) ),
            'errorNonce' => esc_js( __( 'Error: Invalid security nonce.', 'dppp' ) ),
            'errorExists' => esc_js( __( 'Error: The rule already exists.', 'dppp' ) ),
            'errorInsert' => esc_js( __( 'Error: Could not insert rule data.', 'dppp' ) ),
            'errorNoGroups' => esc_js( __( 'Error: There are no plugin groups.', 'dppp' ) ),
            'errorAlreadyInGroup' => esc_js( __( 'Error: The plugin is already in this group.', 'dppp' ) ),
            'errorExistsGroup' => esc_js( __( 'Error: Adding the plugin to this group will result in a group that is the same as another existing group.',
                'dppp' ) ),
            'errorGroupNotExist' => esc_js( __( 'Error: The plugin group does not exist.', 'dppp' ) ),
            'errorAtLeastTwoPlugins' => esc_js( __( 'Error: The group must have at least two plugins.', 'dppp' ) ),
            'errorEditGroup' => esc_js( __( 'Error: Could not edit the group.', 'dppp' ) ),
            'done' => esc_js( __( 'Done', 'dppp' ) ),
            'error' => esc_js( __( 'Error', 'dppp' ) ),
            'errorServerFail' => esc_js( __( 'Error: Unexpected server error. If you have WordPress debugging and logging enabled, '
                . 'you should be able to see more details about the error in the /wp-content/debug.log file.', 'dppp' ) ),
        );
        wp_localize_script( 'dppp-debug-script', 'dpppLocalizedDebug', $localize );
        wp_enqueue_script( 'dppp-debug-script', '', array(), false, true );
        wp_register_style( 'dppp-debug-style', dppp_plugin_dir_url() . 'styles/debug.css',
            false, DPPP_DEACTIVATE_PLUGINS_PER_PAGE_VERSION );
        wp_enqueue_style( 'dppp-debug-style' );
    }
}

// Adds the HTML code in the footer for the debug box on the front-end
function dppp_add_debug_box_to_footer() {
    if ( current_user_can( 'manage_options' ) && dppp_get_option( 'dppp-debug-mode' ) !== 'off' ) {

        if ( ( is_admin() && dppp_skip_this_admin_page() ) || ( is_multisite() && is_network_admin() && dppp_skip_this_network_admin_page() )
            || ( is_admin() && dppp_get_option( 'dppp-debug-mode' ) === 'front' ) || ( ! is_admin() && dppp_get_option( 'dppp-debug-mode' ) === 'back' )
            || isset( $_GET['customize_changeset_uuid'] ) ) {
            return;
        }

        if ( isset( $_GET['elementor-preview'] ) || isset( $_GET['et_fb'] ) || isset( $_GET['vc_editable'] ) ) {
            return;
        }

        // Check if get_plugins() function exists. This is required on the front end, since it is in a file that is normally only loaded in the admin.
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $plugins = get_plugins();
        if ( dppp_get_option( 'dppp-debug-box-state-' . get_current_user_id() ) === 'hidden' ) {
            $debug_box_class = 'dppp-debug-hidden';
        } else {
            $debug_box_class = 'dppp-debug-shown';
        }

        if ( ! is_admin() && is_page() ) {
            global $post;
            $page_id = intval( $post->ID );
        } else {
            $page_id = 0;
        }

        ?>
        <div id="dppp-debug-box" class="<?php echo esc_attr( $debug_box_class ); ?>">
            <img id="dppp-show-debug-logo"  class="dppp-logo-image dppp-display-none" src="<?php echo esc_url( dppp_image_url( 'dppp-logo-32.png' ) ); ?>">
            <span id="dppp-show-debug" onclick="dpppShowDebug()" class="dashicons dashicons-arrow-up-alt2 dppp-display-none"></span>
            <div id="dppp-debug-content">
                <h1>
                    <span id="dppp-hide-debug" onclick="dpppHideDebug()" class="dashicons dashicons-arrow-down-alt2"></span>
                    <img class="dppp-logo-image" src="<?php echo esc_url( dppp_image_url( 'dppp-logo-32.png' ) ); ?>">
                    <?php esc_html_e( 'Deactivate Plugins Per Page', 'dppp' ); ?>
                    <br>
                    <span class="dppp-debug-information"><?php esc_html_e( 'Debug Information', 'dppp' ); ?></span>
                </h1>
                <?php
                if ( dppp_get_option( 'dppp-local-mode' ) === 'on' ) {
                    if ( dppp_is_local_device() ) {
                        $local_color = 'dppp-green';
                        $local_message = __( 'This is a local device.', 'dppp' );
                    } else {
                        $local_color = 'dppp-red';
                        $local_message = __( 'This is not a local device.', 'dppp' );
                    }
                    echo '<p class="' . esc_attr( $local_color ) . '">';
                    esc_html_e( 'Local mode is enabled!', 'dppp' );
                    echo ' ';
                    echo esc_html( $local_message );
                    echo '</p>';
                }
                ?>
                <div class="dppp-debug-uri">
                    <?php esc_html_e( 'Current URI', 'dppp' ); ?>
                    <a title="<?php esc_attr_e( 'Reload Page', 'dppp' ); ?>" class="dppp-debug-reload-page" href="javascript:dpppDebugReloadPage()">
                        <span class="dashicons dashicons-image-rotate dppp-flip-character"></span>
                    </a>
                    <br>
                    <input type="text" id="dppp-debug-current-uri" value="<?php echo esc_attr( rawurldecode( $_SERVER['REQUEST_URI'] ) ); ?>" />
                    <input type="hidden" id="dppp-debug-current-page-id" value="<?php echo esc_attr( $page_id ); ?>" />
                </div>
                <?php
                $uri_for_page_rules = dppp_get_uri_for_page_rules( rawurldecode( $_SERVER['REQUEST_URI'] ) );
                if ( $uri_for_page_rules !== rawurldecode( $_SERVER['REQUEST_URI'] ) && ! is_admin() ) {
                    ?>
                    <div class="dppp-debug-uri">
                        <?php esc_html_e( 'Current URI for Page URI selection rules', 'dppp' ); ?>
                        <span onclick="dpppShowHideDebugHint('debug-current-uri-page')" class="dashicons dashicons-info"></span>
                        <i id="dppp-hint-debug-current-uri-page" class="dppp-hint-text">
                            <?php
                                esc_html_e( 'When applying rules with Page URI selection, we will use this for a Current URI. '
                                    . 'If this field is not showing, it means the value is the same as the Current URI field above.', 'dppp' );
                            ?>
                        </i>
                        <br class="dppp-break-debug-current-uri-page">
                        <input type="text" value="<?php echo esc_attr( $uri_for_page_rules ); ?>" />
                    </div>
                    <?php
                }
                $uri_for_custom_rules = dppp_get_uri_for_custom_rules( rawurldecode( $_SERVER['REQUEST_URI'] ) );
                if ( $uri_for_custom_rules !== rawurldecode( $_SERVER['REQUEST_URI'] ) ) {
                    ?>
                    <div class="dppp-debug-uri">
                        <?php esc_html_e( 'Current URI for Custom URI selection rules', 'dppp' ); ?>
                        <span onclick="dpppShowHideDebugHint('debug-current-uri-custom')" class="dashicons dashicons-info"></span>
                        <i id="dppp-hint-debug-current-uri-custom" class="dppp-hint-text">
                            <?php
                                esc_html_e( 'When applying rules with Custom URI selection, we will use this for a Current URI. '
                                    . 'If this field is not showing, it means the value is the same as the Current URI field above.', 'dppp' );
                            ?>
                        </i>
                        <br class="dppp-break-debug-current-uri-custom">
                        <input type="text" value="<?php echo esc_attr( $uri_for_custom_rules ); ?>" />
                    </div>
                    <?php
                }
                ?>
                <div class="dppp-plugin-list-title">
                    <?php esc_html_e( 'Plugin List', 'dppp' ); ?>
                    (<span class="dppp-green"><?php esc_html_e( 'Active', 'dppp' ); ?></span> /
                    <span class="dppp-red"><?php esc_html_e( 'Inactive', 'dppp' ); ?></span>)
                </div>
                <div class="dppp-plugin-list">
                    <?php
                    $number = 1;
                    $number_dppp_deactivated = 0;
                    foreach ( $plugins as $plugin_file => $plugin_data ) {
                        if ( is_plugin_active( $plugin_file ) ) {
                            echo '<div class="dppp-green">' . esc_html( $number . '. ' . $plugin_data['Name'] );
                        } else {
                            if ( is_array( $GLOBALS['dppp_deactivated_plugins'] ) && in_array( $plugin_file, $GLOBALS['dppp_deactivated_plugins'] ) ) {
                                echo '<div class="dppp-red">' . esc_html( $number . '. ' . $plugin_data['Name'] )
                                    . ' <img title="' . esc_attr__( 'Deactivated by Deactivate Plugins Per Page', 'dppp' )
                                    . '" class="dppp-deactivated-logo-image" src="' . esc_url( dppp_image_url( 'dppp-logo-32.png' ) ) . '">';
                                $number_dppp_deactivated++;
                            } elseif ( array_key_exists( 'dsppp_deactivated_plugins', $GLOBALS ) && is_array( $GLOBALS['dsppp_deactivated_plugins'] )
                                && in_array( $plugin_file, $GLOBALS['dsppp_deactivated_plugins'] ) ) {
                                echo '<div class="dppp-red">' . esc_html( $number . '. ' . $plugin_data['Name'] )
                                    . ' <img title="' . esc_attr__( 'Deactivated by Deactivate Site Plugins Per Page', 'dppp' )
                                    . '" class="dppp-deactivated-logo-image dppp-dsppp-icon" src="' . esc_url( dppp_image_url( 'dppp-logo-32.png' ) ) . '">';
                            } else {
                                echo '<div class="dppp-red">' . esc_html( $number . '. ' . $plugin_data['Name'] );
                            }
                        }
                        if ( 'deactivate-plugins-per-page/deactivate-plugins-per-page.php' !== $plugin_file ) {
                            echo '<span onclick="dpppDebugOpenPluginActions( \'' . esc_attr( $plugin_file ) . '\', \'' . esc_attr( md5( $plugin_file ) )
                                . '\', \'' . esc_attr( $plugin_data['Name'] ) . '\')" title="' . esc_attr__( 'Quick Actions', 'dppp' )
                                . '" class="dashicons dashicons-menu"></span>';
                        }

                        echo '</div>';
                        $number++;
                    }
                    ?>
                </div>
                <?php

                $show_backend_notice = 'no';
                if ( dppp_get_option( 'dppp-affect-back-end' ) === 'off' && is_admin() ) {
                    $show_backend_notice = 'yes';
                }

                foreach ( $plugins as $plugin_file => $plugin_data ) {

                    if ( 'deactivate-plugins-per-page/deactivate-plugins-per-page.php' !== $plugin_file ) {

                        echo '<div id="dppp-plugin-actions-' . esc_attr( md5( $plugin_file ) ) . '" class="dppp-plugin-actions dppp-display-none">';
                        echo '<b class="dppp-debug-quick-title">' . esc_html( sprintf( __( 'Quick actions for %s', 'dppp' ), $plugin_data['Name'] ) ) . '</b>';

                        if ( 'yes' === $show_backend_notice ) {
                            echo '<p class="dppp-red">' . esc_html__( 'Important notice! The deactivation rules are currently not affecting the backend. '
                                . 'You can change this from the plugin settings.', 'dppp' ) . '</p>';
                        }

                        echo '<p><a class="dppp-debug-deactivate-current-' . esc_attr( md5( $plugin_file ) )
                            . '" href="javascript:dpppDebugQuickAction(\'deactivate-current\', \''
                            . esc_attr( $plugin_file ) . '\', \'' . esc_attr( md5( $plugin_file ) ) . '\')"><span class="dashicons dashicons-yes"></span>'
                            . esc_html__( 'Deactivate on the Current URI', 'dppp' ) . '</a></p>';
                        echo '<p><a class="dppp-debug-deactivate-except-current-' . esc_attr( md5( $plugin_file ) )
                            . '" href="javascript:dpppDebugQuickAction(\'deactivate-except-current\', \''
                            . esc_attr( $plugin_file ) . '\', \'' . esc_attr( md5( $plugin_file ) ) . '\')"><span class="dashicons dashicons-yes-alt"></span>'
                            . esc_html__( 'Deactivate on All Except the Current URI', 'dppp' ) . '</a></p>';
                        echo '<p><a class="dppp-debug-add-to-group-' . esc_attr( md5( $plugin_file ) )
                            . '" href="javascript:dpppDebugQuickAction(\'add-to-group\', \''
                            . esc_attr( $plugin_file ) . '\', \'' . esc_attr( md5( $plugin_file ) )
                            . '\')"><span class="dashicons dashicons-screenoptions"></span>' . esc_html__( 'Add to Group:', 'dppp' ) . '</a>';

                        echo '<span class="dppp-debug-done-add-to-group dppp-display-none dppp-green" id="dppp-debug-done-add-to-group-'
                            . esc_attr( md5( $plugin_file ) ) . '" ><span class="dashicons dashicons-screenoptions"></span>'
                            . esc_html__( 'Add to Group:', 'dppp' ) . ' ' . esc_html__( '(Done)', 'dppp' ) . '</span>';

                        $plugin_groups = dppp_get_option( 'dppp-plugin-groups' );
                        if ( ! is_array( $plugin_groups ) || $plugin_groups === false || empty( $plugin_groups ) ) {
                            echo esc_html__( 'There are no groups.', 'dppp' );
                        } else {
                            echo ' <select class="dppp-debug-add-to-group" id="dppp-debug-select-add-to-group-' . esc_attr( md5( $plugin_file ) ). '">';
                            foreach ( $plugin_groups as $group_hash => $plugin_group ) {
                                echo '<option value="' . esc_attr( $group_hash ) . '">' . esc_html( $plugin_group['name'] ) . '</option>';
                            }
                            echo '</select>';
                        }
                        echo '</p>';
                        echo '<p><a href="javascript:dpppDebugClosePluginActions(\'' . esc_attr( md5( $plugin_file ) )
                            . '\')"><span class="dashicons dashicons-arrow-left-alt"></span>'
                            . esc_html__( 'Go back', 'dppp' ) . '</a></p>';
                        echo '<p class="dppp-debug-plugin-list-hint"><em>'
                            . esc_html__( 'Hint: To see changes in the plugin list, you will need to reload the page.', 'dppp' ) . '</em></p>';
                        echo '</div>';
                    }
                }
                echo "<p class='dppp-deactivated-by-rules'>" . sprintf( esc_html__( 'Deactivated by rules: %d', 'dppp' ), $number_dppp_deactivated ) . "</p>";

                ?>

            </div>
        </div>
        <?php
    }
}

/**
 * Adds the Settings action link on the plugins admin page
 * @param array $actions
 * @param string $plugin_file
 * @return array
 */
function dppp_add_settings_plugin_action_link( $actions, $plugin_file ) {
    if ( 'deactivate-plugins-per-page/deactivate-plugins-per-page.php' === $plugin_file ) {
        $settings_action = '<a href="' . esc_url( admin_url( 'plugins.php?page=deactivate-plugins-per-page' ) ) . '">'
            . esc_html__( 'Rules and Settings', 'dppp' ) . '</a>';
        $actions = dppp_add_element_to_array( $actions, 'dppp-settings', $settings_action, 'deactivate' );
    }
    return $actions;
}

// Adds the admin menu element for the plugin settings page
function dppp_add_admin_options_menu() {
    $options_page = add_submenu_page(
        'plugins.php',
        esc_html__( 'Deactivate Plugins Per Page', 'dppp' ),
        esc_html__( 'Deactivation Rules', 'dppp' ),
        'manage_options',
        'deactivate-plugins-per-page',
        'dppp_admin_options_page'
    );

    // Loads the style and script for the plugin settings page
    add_action( 'load-' . $options_page, 'dppp_add_options_assets' );
}

// Loads the style and script for the plugin settings page
function dppp_add_options_assets() {
    add_action( 'admin_enqueue_scripts', 'dppp_register_options_assets' );
}

// Registers, localizes, and enqueues the JavaScript file for the plugin settings page, and registers and enqueus the CSS file
function dppp_register_options_assets() {
    wp_register_script( 'dppp-options-script', dppp_plugin_dir_url() . 'scripts/options.js',
        array(), DPPP_DEACTIVATE_PLUGINS_PER_PAGE_VERSION, false );
    $localize = array(
        'ajaxNonce' => wp_create_nonce( 'dppp_options_ajax_nonce' ),
        'loading' => esc_js( __( 'Loading...', 'dppp' ) ),
        'hideHint' => esc_js( __( 'Hide hint', 'dppp' ) ),
        'readMore' => esc_js( __( 'Read more', 'dppp' ) ),
        'recreateCacheURL' => admin_url( 'plugins.php?page=deactivate-plugins-per-page&dppp-recreate-cache=yes' ),
        'settingsURL' => admin_url( 'plugins.php?page=deactivate-plugins-per-page#dppp-settings-section' ),
        'doneSitesText' => esc_js( __( 'Processed sites so far:', 'dppp' ) ),
        'confirmDeleteRule' => esc_js( __( 'Are you sure you want to delete this rule?', 'dppp' ) ),
        'confirmLocalDevice' => esc_js( __( 'This will create a small cookie file on your device, so the plugin can recognize '
            . 'it and apply the deactivations only for you (and other local devices if you have set any). It will be set to expire and be deleted in 1 year, '
            . 'but you can always delete it as well.', 'dppp' ) ),
        'confirmDeleteGroup' => esc_js( __( 'Are you sure you want to delete this plugin group and all its rules?', 'dppp' ) ),
        'confirmDeleteAllRules' => esc_js( __( 'Are you sure you want to DELETE ALL RULES?', 'dppp' ) ),
        'confirmPauseAllRules' => esc_js( __( 'Are you sure you want to pause all rules?', 'dppp' ) ),
        'confirmStartAllRules' => esc_js( __( 'Are you sure you want to start all rules?', 'dppp' ) ),
        'confirmDeletePluginRules' => esc_js( __( 'Are you sure you want to DELETE ALL RULES for this plugin?', 'dppp' ) ),
        'confirmPausePluginRules' => esc_js( __( 'Are you sure you want to pause all rules for this plugin?', 'dppp' ) ),
        'confirmStartPluginRules' => esc_js( __( 'Are you sure you want to start all rules for this plugin?', 'dppp' ) ),
        'confirmDeleteGroupRules' => esc_js( __( 'Are you sure you want to DELETE ALL RULES for this plugin group?', 'dppp' ) ),
        'confirmPauseGroupRules' => esc_js( __( 'Are you sure you want to pause all rules for this plugin group?', 'dppp' ) ),
        'confirmStartGroupRules' => esc_js( __( 'Are you sure you want to start all rules for this plugin group?', 'dppp' ) ),
        'confirmAbortImport' => esc_js( __( 'This will abort the import if it is not done after the current request. Are you sure?', 'dppp' ) ),
        'errorResponse' => esc_js( __( 'Error: We got an unexpected response.', 'dppp' ) ),
        'errorAccess' => esc_js( __( 'Error: You do not have sufficient permissions to make this request.', 'dppp' ) ),
        'errorNonce' => esc_js( __( 'Error: Invalid security nonce.', 'dppp' ) ),
        'errorData' => esc_js( __( 'Error: Invalid data sent.', 'dppp' ) ),
        'errorImportData' => esc_js( __( 'Error: Invalid import data sent.', 'dppp' ) ),
        'errorInvalidMultipleSettings' => esc_js( __( 'Error: Invalid import settings.', 'dppp' ) ),
        'errorCannotContinue' => esc_js( __( 'Error: This task was stopped due to an error. It could be partially completed.', 'dppp' ) ),
        'errorNoAffectedSites' => esc_js( __( 'Error: With these settings there will be no sites affected.', 'dppp' ) ),
        'errorSelectPage' => esc_js( __( 'Error: Select at least one page.', 'dppp' ) ),
        'errorRegex' => esc_js( __( 'Error: Invalid regular expression.', 'dppp' ) ),
        'errorEnterURI' => esc_js( __( 'Error: The custom URI value is empty.', 'dppp' ) ),
        'errorCheckExists' => esc_js( __( 'Error: Could not check if the rule exists.', 'dppp' ) ),
        'errorExists' => esc_js( __( 'Error: The rule already exists.', 'dppp' ) ),
        'errorExistsGroup' => esc_js( __( 'Error: The plugin group already exists.', 'dppp' ) ),
        'errorGroupNotExist' => esc_js( __( 'Error: The plugin group does not exist.', 'dppp' ) ),
        'errorAtLeastTwoPlugins' => esc_js( __( 'Error: Select at least two plugins.', 'dppp' ) ),
        'errorURL' => esc_js( __( 'Error: Please enter only the URI part after the hostname, not the full URL.', 'dppp' ) ),
        'errorInsert' => esc_js( __( 'Error: Could not insert rule data.', 'dppp' ) ),
        'errorEdit' => esc_js( __( 'Error: Could not edit the rule.', 'dppp' ) ),
        'errorEditGroup' => esc_js( __( 'Error: Could not edit the group.', 'dppp' ) ),
        'errorSetCookie' => esc_js( __( 'Error: We could not create the cookie. Make sure that cookies are allowed in your browser.', 'dppp' ) ),
        'errorRemoveCookie' => esc_js( __( 'Error: We could not remove the cookie.', 'dppp' ) ),
        'done' => esc_js( __( 'Done', 'dppp' ) ),
        'error' => esc_js( __( 'Error', 'dppp' ) ),
        'pasteRule' => esc_js( __( 'Paste 1 rule', 'dppp' ) ),
        'pasteRules' => esc_js( __( 'Paste %d rules', 'dppp' ) ),
        'donePastedRulesSkippedFailed' => esc_js( __( 'Some rules were pasted. Some rules were skipped because they already exist. '
            . 'Some rules failed to be pasted. The page will reload.', 'dppp' ) ),
        'donePastedRulesSkipped' => esc_js( __( 'Some rules were pasted. Other rules were skipped because they already exist. The page will reload.', 'dppp' ) ),
        'donePastedRulesFailed' => esc_js( __( 'Some rules were pasted. Other rules failed to be pasted. The page will reload.', 'dppp' ) ),
        'errorPastedRulesSkippedFailed' => esc_js( __( 'Error: No rules were pasted. Some rules failed to be pasted. '
            . 'Other rules were skipped because they already exist.', 'dppp' ) ),
        'errorPastedRulesSkipped' => esc_js( __( 'All rules were skipped because they already exist.', 'dppp' ) ),
        'errorPastedRulesFailed' => esc_js( __( 'Error: The rules failed to be pasted.', 'dppp' ) ),
        'errorPastedRulesError' => esc_js( __( 'Error: We could not paste the rules.', 'dppp' ) ),
        'errorServerFail' => esc_js( __( 'Error: Unexpected server error. If you have WordPress debugging and logging enabled, '
            . 'you should be able to see more details about the error in the /wp-content/debug.log file.', 'dppp' ) ),
    );
    wp_localize_script( 'dppp-options-script', 'dpppLocalizedOptions', $localize );
    wp_enqueue_script( 'dppp-options-script', '', array(), false, true );
    wp_register_style( 'dppp-options-style', dppp_plugin_dir_url() . 'styles/options.css',
        false, DPPP_DEACTIVATE_PLUGINS_PER_PAGE_VERSION );
    wp_enqueue_style( 'dppp-options-style' );
}

// Displays the plugin rules and settings in the plugin settings page in the admin panel
function dppp_admin_options_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'dppp' ) );
    }
    global $wpdb;
    if ( $wpdb->get_var( "SHOW TABLES LIKE '{$wpdb->prefix}dppp_plugin_deactivation_rules'" ) !== "{$wpdb->prefix}dppp_plugin_deactivation_rules" ) {
        dppp_create_database_table();
    }

    // We generate and save the unique cookie ID for this WordPress installation we will use the local mode cookie
    if ( dppp_get_option( 'dppp-local-mode-cookie' ) === false ) {
        update_option( 'dppp-local-mode-cookie', 'dppp-local-mode-' . dppp_random_string( 10 ) );
    }
    ?>
    <div class="wrap">
        <h1 class="dppp-h1-admin-page"></h1>
        <div class="dppp-logo-div dppp-white-box">
            <b>
                <img class="dppp-logo-image" src="<?php echo esc_url( dppp_image_url( 'dppp-logo-32.png' ) ); ?>" />
                <?php esc_html_e( 'Deactivate Plugins Per Page', 'dppp' ); ?>
            </b>
        </div>

        <?php
        $status = dppp_check_and_add_mu_plugin();
        if ( false === $status ) {
            if ( is_multisite() && ! current_user_can( 'manage_network' ) ) {
                ?>
                <div class="dppp-white-box dppp-full-width">
                    <b class="dppp-warning">
                        <span class="dashicons dashicons-warning"></span>
                        <?php esc_html_e( 'Warning!', 'dppp' ); ?>
                    </b>
                    <br>
                    <?php
                    printf(
                        esc_html__( 'We could not copy the required must use plugin file. Please tell the network administrator to do it manually. '
                            . 'The file %1$sdppp-mu-deactivate-plugins-per-page.php%2$s has to be copied in the %1$smu-plugins%2$s '
                            . 'folder of the multisite.', 'dppp' ),
                        '<i>', '</i>'
                    );
                    ?>
                </div>
                <?php
            } else {
                $mu_file_from_plugin = dppp_plugin_dir_path() . 'mu-plugin/dppp-mu-deactivate-plugins-per-page.php';
                $mu_file_path = dppp_get_mu_path() . '/dppp-mu-deactivate-plugins-per-page.php'
                ?>
                <div class="dppp-white-box dppp-full-width">
                    <b class="dppp-warning">
                        <span class="dashicons dashicons-warning"></span>
                        <?php esc_html_e( 'Warning!', 'dppp' ); ?>
                    </b>
                    <br>
                    <?php
                    printf(
                        esc_html__( 'We could not copy the required must use plugin file. Please do it manually. '
                            . 'Copy the file %s to %s', 'dppp' ),
                        '<i>' . esc_html( $mu_file_from_plugin ) . '</i>', '<i>' . esc_html( $mu_file_path ) . '</i>'
                    );
                    ?>
                </div>
                <?php
            }
        } elseif ( 'version-mismatch' === $status ) {
            if ( is_multisite() && ! current_user_can( 'manage_network' ) ) {
                ?>
                <div class="dppp-white-box dppp-full-width">
                    <b class="dppp-warning">
                        <span class="dashicons dashicons-warning"></span>
                        <?php esc_html_e( 'Warning!', 'dppp' ); ?>
                    </b>
                    <br>
                    <?php
                    printf(
                        esc_html__( 'The version of the required must use plugin file is not correct. Please tell the network '
                            . 'administrator to upload the correct version manually. '
                            . 'The correct version of the file %1$sdppp-mu-deactivate-plugins-per-page.php%2$s has to be copied in the %1$smu-plugins%2$s '
                            . 'folder of the multisite.', 'dppp' ),
                        '<i>', '</i>'
                    );
                    ?>
                </div>
                <?php
            } else {
                $mu_file_from_plugin = dppp_plugin_dir_path() . 'mu-plugin/dppp-mu-deactivate-plugins-per-page.php';
                $mu_file_path = dppp_get_mu_path() . '/dppp-mu-deactivate-plugins-per-page.php';
                ?>
                <div class="dppp-white-box dppp-full-width">
                    <b class="dppp-warning">
                        <span class="dashicons dashicons-warning"></span>
                        <?php esc_html_e( 'Warning!', 'dppp' ); ?>
                    </b>
                    <br>
                    <?php
                    printf(
                        esc_html__( 'The version of the required must use plugin file is not correct. This probably means that we could '
                            . 'not update it after a new version of the plugin was uploaded. Please do it manually. '
                            . 'Copy the file %s to %s and overwrite the existing file there.', 'dppp' ),
                        '<i>' . esc_html( $mu_file_from_plugin ) . '</i>', '<i>' . esc_html( $mu_file_path ) . '</i>'
                    );
                    ?>
                </div>
                <?php
            }
        }
        if ( dppp_get_option( 'dppp-local-mode' ) === 'on' ) {
            ?>
            <div class="dppp-white-box dppp-full-width">
                <b class="dppp-warning">
                    <span class="dashicons dashicons-warning"></span>
                    <?php esc_html_e( 'Notice!', 'dppp' ); ?>
                </b>
                <br>
                <?php
                esc_html_e( 'Local mode is enabled. All deactivations made by this plugin will only affect local devices.', 'dppp' );
                echo ' ';
                if ( dppp_is_local_device() ) {
                    esc_html_e( 'Your device is a local device. You can disable local mode in the settings located at the bottom of this screen.', 'dppp' );
                } else {
                    esc_html_e( 'Your device is not a local device. You can make it local from the settings located at the bottom of this screen.', 'dppp' );
                }
                ?>
            </div>
            <div class="dppp-white-box dppp-full-width">
                <b class="dppp-warning">
                    <span class="dashicons dashicons-warning"></span>
                    <?php esc_html_e( 'Warning!', 'dppp' ); ?>
                </b>
                <br>
                <?php
                $cookie = dppp_get_option( 'dppp-local-mode-cookie' );
                echo esc_html( sprintf( __( 'If you are using a caching plugin, it can make local mode useless by caching the changes you make and '
                    . 'serving them to everyone. To avoid this it is best to disable caching while testing in local mode. '
                    . 'Alternatively you can disable caching only for visitors with the cookie named %s (if your caching plugin allows that), '
                    . 'which indicates it is a local device.', 'dppp' ), $cookie ) );
                ?>
            </div>
            <?php
        }
        ?>

        <div id="dppp-top-actions" class="dppp-white-box dppp-full-width">
            <span class="dashicons dashicons-search"></span>
            <span class="dppp-hide-under-751"><?php esc_html_e( 'Find plugins:', 'dppp' ); ?></span>
            <input id="dppp-find-plugins" type="text" oninput="dpppFindPlugins()" />
            <a title="<?php esc_attr_e( 'Clear', 'dppp' ); ?>" id="dppp-clear-find-plugins" class="dppp-display-none" href="javascript:dpppClearFindPlugins()">
                <span class="dashicons dashicons-dismiss"></span>
            </a>
            <span class="dppp-global-actions-contain dppp-hide-on-copy-rule">
                <span class="dashicons dashicons-admin-site-alt"></span>
                <span class="dppp-hide-under-751"><?php esc_html_e( 'Global actions:', 'dppp' ); ?></span>
                <a title="<?php esc_attr_e( 'Export Rules and Settings', 'dppp' ); ?>" id="dppp-export-rules-link" class="dppp-global-action-link"
                    href="javascript:dpppGlobalActions( 'export-rules', 'none' )"><span class="dashicons dashicons-download"></span><span
                    class="dppp-global-action-text"><?php printf( esc_html__( 'Export Rules%sand Settings', 'dppp' ), '<br>' ); ?></span></a>
                <a title="<?php esc_attr_e( 'Import Rules and Settings', 'dppp' ); ?>" id="dppp-import-rules-link" class="dppp-global-action-link"
                    href="javascript:dpppGlobalActions( 'import-rules', 'none' )"><span class="dashicons dashicons-upload"></span><span
                    class="dppp-global-action-text"><?php printf( esc_html__( 'Import Rules%sand Settings', 'dppp' ), '<br>' ); ?></span></a>
                <a title="<?php esc_attr_e( 'Start All Rules (except front-end/back-end global rules)', 'dppp' ); ?>"
                    id="dppp-start-all-rules-link" class="dppp-global-action-link"
                    href="javascript:dpppGlobalActions( 'start-all-rules', 'none' )"><span class="dashicons dashicons-controls-play"></span></a>
                <a title="<?php esc_attr_e( 'Pause All Rules (except front-end/back-end global rules)', 'dppp' ); ?>"
                    id="dppp-pause-all-rules-link" class="dppp-global-action-link"
                    href="javascript:dpppGlobalActions( 'pause-all-rules', 'none' )"><span class="dashicons dashicons-controls-pause"></span></a>
                <a title="<?php esc_attr_e( 'Delete All Rules (except front-end/back-end global rules)', 'dppp' ); ?>"
                    id="dppp-delete-all-rules-link" class="dppp-global-action-link"
                    href="javascript:dpppGlobalActions( 'delete-all-rules', 'none' )"><span class="dashicons dashicons-no"></span></a>
            </span>
        </div>
        <?php
        if ( dppp_get_option( 'dppp-groups-first' ) === 'on' ) {
            ?>
            <div id="dppp-plugin-groups-section" class="dppp-white-box dppp-full-width">
                <?php dppp_output_plugin_groups_section(); ?>
            </div>
            <?php
        }
        $plugins = get_plugins();
        $plugins_to_hide = Array();
        $hidden_active_plugins = 0;
        $hidden_inactive_plugins = 0;
        if ( 'on' === dppp_get_option( 'dppp-hide-plugins-in-group' ) ) {
            $plugin_groups = dppp_get_option( 'dppp-plugin-groups' );
            foreach ( $plugins as $plugin_file => $plugin_data ) {
                if ( ! dppp_plugin_has_rules_or_global( $plugin_file ) && dppp_plugin_in_any_group( $plugin_file, $plugin_groups ) ) {
                    $plugins_to_hide[] = $plugin_file;
                    if ( is_plugin_active( $plugin_file ) ) {
                        $hidden_active_plugins++;
                    } else {
                        $hidden_inactive_plugins++;
                    }
                }
            }
        }
        ?>
        <div id="dppp-active-plugins-section" class="dppp-white-box dppp-full-width">
            <b class="dppp-subheading">
                <span class="dashicons dashicons-admin-plugins"></span>
                <?php
                esc_html_e( 'Active Plugins', 'dppp' );
                if ( $hidden_active_plugins > 0 ) {
                    ?>
                    <span class="dppp-hidden-plugin-count"
                        title="<?php esc_attr_e( 'You have enabled the setting to hide plugins without rules if they are in a group.', 'dppp' ) ?>">
                        <?php echo esc_html( sprintf( __( '(%d hidden)', 'dppp' ), $hidden_active_plugins ) ); ?>
                    </span>
                    <?php
                }
                ?>
            </b>
            <a id="dppp-show-hide-active-plugins" title="<?php esc_attr_e( 'Show/Hide Section', 'dppp' ); ?>"
                class="dppp-show-hide-section" href="javascript:dpppShowHideSection( 'active-plugins' )">
                <?php
                if ( dppp_get_option( 'dppp-section-show-hide-active-plugins' ) !== 'hide' ) {
                    $add_class_active_plugins = '';
                    echo '<span class="dashicons dashicons-arrow-up-alt2"></span>';
                } else {
                    $add_class_active_plugins = ' dppp-display-none';
                    echo '<span class="dashicons dashicons-arrow-down-alt2"></span>';
                }
                ?>
            </a>
            <?php
            if ( dppp_get_option( 'dppp-rules-first' ) === 'on' ) {
                $plugins_with_rules = Array();
                $plugins_without_rules = Array();
                foreach ( $plugins as $plugin_file => $plugin_data ) {
                    $has_rules = dppp_plugin_has_rules_or_global( $plugin_file );
                    if ( false === $has_rules ) {
                        $plugins_without_rules[ $plugin_file ] = $plugin_data;
                    } else {
                        $plugins_with_rules[ $plugin_file ] = $plugin_data;
                    }
                }
                $loop_count = 1;
                foreach ( $plugins_with_rules as $plugin_file => $plugin_data ) {
                    if ( is_plugin_active( $plugin_file ) && ! in_array( $plugin_file, $plugins_to_hide )
                        && 'deactivate-plugins-per-page/deactivate-plugins-per-page.php' !== $plugin_file ) {
                        dppp_output_plugin_row( $loop_count, $plugin_file, $plugin_data, $add_class_active_plugins );
                        $loop_count++;
                    }
                }
                foreach ( $plugins_without_rules as $plugin_file => $plugin_data ) {
                    if ( is_plugin_active( $plugin_file ) && ! in_array( $plugin_file, $plugins_to_hide )
                        && 'deactivate-plugins-per-page/deactivate-plugins-per-page.php' !== $plugin_file ) {
                        dppp_output_plugin_row( $loop_count, $plugin_file, $plugin_data, $add_class_active_plugins );
                        $loop_count++;
                    }
                }
            } else {
                $loop_count = 1;
                foreach ( $plugins as $plugin_file => $plugin_data ) {
                    if ( is_plugin_active( $plugin_file ) && ! in_array( $plugin_file, $plugins_to_hide )
                        && 'deactivate-plugins-per-page/deactivate-plugins-per-page.php' !== $plugin_file ) {
                        dppp_output_plugin_row( $loop_count, $plugin_file, $plugin_data, $add_class_active_plugins );
                        $loop_count++;
                    }
                }
            }
            if ( $loop_count < 2 ) {
                ?>
                <div class="dppp-inner-box<?php echo esc_attr( $add_class_active_plugins ); ?>">
                    <div class="dppp-section-row">
                        <span class="dppp-section-row-message">
                            <?php esc_html_e( 'You do not have any other active plugins right now.', 'dppp' ); ?>
                        </span>
                    </div>
                </div>
                <?php
            }
            if ( dppp_get_option( 'dppp-inactive-plugins' ) === 'on' ) {
                $inactive_plugins_class = "";
            } else {
                $inactive_plugins_class = "dppp-display-none";
            }
            ?>
        </div>
        <div id="dppp-inactive-plugins-section" class="dppp-white-box dppp-full-width <?php echo esc_attr( $inactive_plugins_class ); ?>">
            <b class="dppp-subheading">
                <span class="dashicons dashicons-admin-plugins"></span>
                <?php
                esc_html_e( 'Inactive Plugins', 'dppp' );
                if ( $hidden_inactive_plugins > 0 ) {
                    ?>
                    <span class="dppp-hidden-plugin-count"
                        title="<?php esc_attr_e( 'You have enabled the setting to hide plugins without rules if they are in a group.', 'dppp' ) ?>">
                        <?php echo esc_html( sprintf( __( '(%d hidden)', 'dppp' ), $hidden_inactive_plugins ) ); ?>
                    </span>
                    <?php
                }
                ?>
            </b>
            <a id="dppp-show-hide-inactive-plugins" title="<?php esc_attr_e( 'Show/Hide Section', 'dppp' ); ?>"
                class="dppp-show-hide-section" href="javascript:dpppShowHideSection( 'inactive-plugins' )">
                <?php
                if ( dppp_get_option( 'dppp-section-show-hide-inactive-plugins' ) !== 'hide' ) {
                    $add_class_inactive_plugins = '';
                    echo '<span class="dashicons dashicons-arrow-up-alt2"></span>';
                } else {
                    $add_class_inactive_plugins = ' dppp-display-none';
                    echo '<span class="dashicons dashicons-arrow-down-alt2"></span>';
                }
                ?>
            </a>
            <?php
            if ( dppp_get_option( 'dppp-rules-first' ) === 'on' ) {
                $loop_count = 1;
                foreach ( $plugins_with_rules as $plugin_file => $plugin_data ) {
                    if ( ! is_plugin_active( $plugin_file ) && ! in_array( $plugin_file, $plugins_to_hide ) ) {
                        dppp_output_plugin_row( $loop_count, $plugin_file, $plugin_data, $add_class_inactive_plugins );
                        $loop_count++;
                    }
                }
                foreach ( $plugins_without_rules as $plugin_file => $plugin_data ) {
                    if ( ! is_plugin_active( $plugin_file ) && ! in_array( $plugin_file, $plugins_to_hide ) ) {
                        dppp_output_plugin_row( $loop_count, $plugin_file, $plugin_data, $add_class_inactive_plugins );
                        $loop_count++;
                    }
                }
            } else {
                $loop_count = 1;
                foreach ( $plugins as $plugin_file => $plugin_data ) {
                    if ( ! is_plugin_active( $plugin_file ) && ! in_array( $plugin_file, $plugins_to_hide ) ) {
                        dppp_output_plugin_row( $loop_count, $plugin_file, $plugin_data, $add_class_inactive_plugins );
                        $loop_count++;
                    }
                }
            }
            if ( $loop_count < 2 ) {
                ?>
                <div class="dppp-inner-box<?php echo esc_attr( $add_class_inactive_plugins ); ?>">
                    <div class="dppp-section-row">
                        <span class="dppp-section-row-message">
                            <?php esc_html_e( 'You do not have any inactive plugins right now.', 'dppp' ); ?>
                        </span>
                    </div>
                </div>
                <?php
            }
            ?>
        </div>
        <?php
        if ( dppp_get_option( 'dppp-groups-first' ) !== 'on' ) {
            ?>
            <div id="dppp-plugin-groups-section" class="dppp-white-box dppp-full-width">
                <?php dppp_output_plugin_groups_section(); ?>
            </div>
            <?php
        }
        ?>
        <div id="dppp-settings-section" class="dppp-white-box dppp-full-width dppp-settings-box">
            <b class="dppp-subheading">
                <span class="dashicons dashicons-admin-generic"></span> <?php esc_html_e( 'Settings', 'dppp' ); ?>
            </b>
            <a id="dppp-show-hide-settings" title="<?php esc_attr_e( 'Show/Hide Section', 'dppp' ); ?>"
                class="dppp-show-hide-section" href="javascript:dpppShowHideSection( 'settings' )">
                <?php
                if ( dppp_get_option( 'dppp-section-show-hide-settings' ) !== 'hide' ) {
                    $add_class_settings = '';
                    echo '<span class="dashicons dashicons-arrow-up-alt2"></span>';
                } else {
                    $add_class_settings = ' dppp-display-none';
                    echo '<span class="dashicons dashicons-arrow-down-alt2"></span>';
                }
                ?>
            </a>
            <div id="dppp-settings-inner" class="dppp-inner-box dppp-full-width<?php echo esc_attr( $add_class_settings ); ?>">
                <?php dppp_output_settings_content( $add_class_settings ); ?>
            </div>
        </div>
    </div>
    <div id="dppp-ajax-results-container" class="dppp-div-contain-results">
        <table class="dppp-table-container">
            <tr>
                <td class="dppp-middle">
                    <div id="dppp-results">
                    </div>
                </td>
            </tr>
        </table>
    </div>
    <?php
    dppp_output_export_rules_form_hidden();
    dppp_output_import_rules_form_hidden();
}

// Handles the ajax request for deleting a rule
function dppp_delete_rule_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }
    $rule_id = intval( $_POST['rule_id'] );
    global $wpdb;
    $wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE rule_id = %d", $rule_id ) );
    wp_die( 'done' );
}

// Handles the ajax request for pausing a rule
function dppp_pause_rule_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }
    $rule_id = intval( $_POST['rule_id'] );
    global $wpdb;
    $wpdb->update(
        "{$wpdb->prefix}dppp_plugin_deactivation_rules",
        array(
            'status' => 'paused',
        ),
        array(
            'rule_id' => $rule_id,
        ),
        array( '%s' ),
        array( '%d' )
    );
    wp_die( 'done' );
}

// Handles the ajax request for starting a rule
function dppp_start_rule_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }
    $rule_id = intval( $_POST['rule_id'] );
    global $wpdb;
    $wpdb->update(
        "{$wpdb->prefix}dppp_plugin_deactivation_rules",
        array(
            'status' => 'started',
        ),
        array(
            'rule_id' => $rule_id,
        ),
        array( '%s' ),
        array( '%d' )
    );
    wp_die( 'done' );
}

// Handles the ajax request for refreshing the plugin rules layer displayed for a given plugin
function dppp_update_rules_content_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( '<p class="dppp-clear">' . esc_html__( 'Error: Invalid security nonce.', 'dppp' ) . '</p>' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( '<p class="dppp-clear">' . esc_html__( 'Error: You do not have sufficient permissions to make this request.', 'dppp' ) . '</p>' );
    }
    $plugin_file = dppp_sanitize_plugin_file( $_POST['plugin_file'] );
    if ( ! dppp_is_valid_plugin_file( $plugin_file ) && 'dppp-group' !== $plugin_file ) {
        wp_die( '<p class="dppp-clear">' . esc_html__( 'Error: Invalid plugin file data.', 'dppp' ) . '</p>' );
    }
    if ( 'dppp-group' === $plugin_file ) {
        $hash = sanitize_html_class( $_POST['hash'] );
        $plugin_groups = dppp_get_option( 'dppp-plugin-groups' );
        $plugin_files_list = dppp_get_plugin_files_list_from_array( $plugin_groups[ $hash ]['plugins'] );
        $results_multi_array = dppp_get_results_plugin_rules( $plugin_files_list );
        dppp_output_plugin_rules_content( 'dppp-group', $hash, $results_multi_array );
    } else {
        $hash = md5( $plugin_file );
        $results_multi_array = dppp_get_results_plugin_rules( $plugin_file );
        dppp_output_plugin_rules_content( $plugin_file, $hash, $results_multi_array );
    }
    wp_die();
}

/**
 * Returns a multi-array with database query results with the rules data for a plugin
 * @param string $plugin_file
 * @return mixed
 */
function dppp_get_results_plugin_rules( $plugin_file ) {
    global $wpdb;
    $results_multi_array = $wpdb->get_results( $wpdb->prepare( "SELECT rule_id, status, devices, page_ids, uri_type, deactivation_type, "
        . "uri_condition, uri_value FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE plugin_file = %s AND uri_type != 'global' "
        . "ORDER BY rule_id ASC", $plugin_file ), ARRAY_A );
    return $results_multi_array;
}

/**
 * Checks if the plugin has any rules, including paused or global deactivations
 * @param string $plugin_file
 * @return bool
 */
function dppp_plugin_has_rules_or_global( $plugin_file ) {
    global $wpdb;
    $result = $wpdb->get_var( $wpdb->prepare( "SELECT count( rule_id ) FROM {$wpdb->prefix}dppp_plugin_deactivation_rules "
        . "WHERE plugin_file = %s AND ( uri_type != 'global' OR ( uri_type = 'global' AND status = 'started' ) )", $plugin_file ) );
    if ( $result > 0 ) {
        return true;
    }
    return false;
}

// Handles the ajax request for displaying the box with the form for adding a new plugin rule
function dppp_add_new_rule_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( '<h2>' . esc_html__( 'Add New Rule', 'dppp' ) . '</h2>'
            . '<input type="button" class="dppp-close-button button" id="dppp-button-close-add-new" value="' . esc_attr__( 'Close', 'dppp' ) . '" />'
            . '<p class="dppp-clear">' . esc_html__( 'Error: Invalid security nonce.', 'dppp' ) . '</p>' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( '<h2>' . esc_html__( 'Add New Rule', 'dppp' ) . '</h2>'
            . '<input type="button" class="dppp-close-button button" id="dppp-button-close-add-new" value="' . esc_attr__( 'Close', 'dppp' ) . '" />'
            . '<p class="dppp-clear">' . esc_html__( 'Error: You do not have sufficient permissions to make this request.', 'dppp' ) . '</p>' );
    }
    $plugin_file = dppp_sanitize_plugin_file( $_POST['plugin_file'] );
    if ( ! dppp_is_valid_plugin_file( $plugin_file ) && 'dppp-group' !== $plugin_file ) {
        wp_die( '<h2>' . esc_html__( 'Add New Rule', 'dppp' ) . '</h2>'
            . '<input type="button" class="dppp-close-button button" id="dppp-button-close-add-new" value="' . esc_attr__( 'Close', 'dppp' ) . '" />'
            . '<p class="dppp-clear">' . esc_html__( 'Error: Invalid plugin file data.', 'dppp' ) . '</p>' );
    }
    if ( 'dppp-group' !== $plugin_file ) {
        $plugin_data = get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin_file );
        $add_rule_for = $plugin_data['Name'];
    } else {
        $group_hash = sanitize_html_class( $_POST['hash'] );
        $plugin_groups = dppp_get_option( 'dppp-plugin-groups' );
        if ( ! is_array( $plugin_groups ) || ! array_key_exists( $group_hash, $plugin_groups ) ) {
            wp_die( '<h2>' . esc_html__( 'Add New Rule', 'dppp' ) . '</h2>'
                . '<input type="button" class="dppp-close-button button" id="dppp-button-close-add-new" value="' . esc_attr__( 'Close', 'dppp' ) . '" />'
                . '<p class="dppp-clear">' . esc_html__( 'Error: The plugin group does not exist.', 'dppp' ) . '</p>' );
        }
        $add_rule_for = stripslashes( $plugin_groups[ $group_hash ]['name'] );
    }
    echo '<h2>' . esc_html( sprintf( __( 'Add New Rule for %s', 'dppp' ), $add_rule_for ) ) . '</h2>';
    ?>
    <input type="button" class="dppp-close-button button" id="dppp-button-close-add-new" value="<?php esc_attr_e( 'Close', 'dppp' ); ?>" />
    <div class="dppp-clear">
        <?php esc_html_e( '1. Deactivation type', 'dppp' ); ?><br>
        <select name="dppp-select-deactivation-type" id="dppp-select-deactivation-type">
            <option value="selected" selected><?php esc_html_e( 'Deactivate on selected URI', 'dppp' ); ?></option>
            <option value="except"><?php esc_html_e( 'Deactivate on all URI except selected', 'dppp' ); ?></option>
        </select>
    </div>
    <div>
        <?php esc_html_e( '2. URI selection type', 'dppp' ); ?><br>
        <select name="dppp-select-uri-type" id="dppp-select-uri-type">
            <option value="page" selected><?php esc_html_e( 'Page', 'dppp' ); ?></option>
            <option value="custom"><?php esc_html_e( 'Custom', 'dppp' ); ?></option>
        </select>
    </div>
    <div id="dppp-page-uri-type">
        <?php

        esc_html_e( '3. Select pages to get URI from (updates automatically if the page slug changes)', 'dppp' );

        // If the WPML plugin is active, we show a small menu to filter pages by language
        if ( is_plugin_active( 'sitepress-multilingual-cms/sitepress.php' ) ) {
            global $sitepress;
            $languages = apply_filters( 'wpml_active_languages', NULL );
            if ( ! empty( $languages ) ) {
                echo '<br>';
                esc_html_e( 'Filter by WPML language: ', 'dppp' );
                echo '<a id="dppp-wpml-filter-pages-all" class="dppp-display-none dppp-wpml-filter-pages" '
                    . 'href="javascript:dpppFilterPagesByLanguageWPML( \'dppp-all\' )">'
                    . esc_html__( 'All', 'dppp' ) . '</a><span id="dppp-wpml-filter-pages-all-active" class="dppp-wpml-filter-pages-active">'
                    . esc_html__( 'All', 'dppp' ) . '</span> | ';
                $number = 1;
                foreach( $languages as $language ) {
                    echo '<a id="dppp-wpml-filter-pages-' . esc_attr( sanitize_html_class( $language['language_code'] ) )
                        . '" class="dppp-wpml-filter-pages" href="javascript:dpppFilterPagesByLanguageWPML( \''
                        . esc_attr( esc_js( sanitize_html_class( $language['language_code'] ) ) ) . '\' )">'
                        . esc_html( strtoupper( $language['language_code'] ) ) . '</a><span id="dppp-wpml-filter-pages-'
                        . esc_attr( sanitize_html_class( $language['language_code'] ) ) . '-active" class="dppp-display-none dppp-wpml-filter-pages-active">'
                        . esc_html( strtoupper( $language['language_code'] ) ) . '</span>';
                    if ( $number !== count( $languages ) ) {
                        echo ' | ';
                    }
                    $number++;
                }
            }
        }
        ?>

        <div class="dppp-choose-elements" onscroll="dpppPositionFindSelectables()">

            <div class="dppp-find-selectables">
                <label>
                    <span title="<?php esc_attr_e( 'Find', 'dppp' ); ?>" class="dashicons dashicons-search"></span>
                    <input class="dppp-input-find-selectables" type="text" oninput="dpppFindSelectables()" />
                    <a title="Clear" id="dppp-clear-find-selectables" class="dppp-display-none" href="javascript:dpppClearFindSelectables()">
                        <span class="dashicons dashicons-dismiss"></span>
                    </a>
                </label>
            </div>

            <?php
            $args = array(
                'sort_order' => 'asc',
                'sort_column' => 'post_title',
                'hierarchical' => 1,
                'exclude' => '',
                'include' => '',
                'meta_key' => '',
                'meta_value' => '',
                'authors' => '',
                'child_of' => 0,
                'parent' => -1,
                'exclude_tree' => '',
                'number' => '',
                'offset' => 0,
                'post_type' => 'page',
                'post_status' => 'publish'
            );

            // If the WPML plugin is active, we show the pages in sections by language
            if ( is_plugin_active( 'sitepress-multilingual-cms/sitepress.php' ) ) {
                if ( ! empty( $languages ) ) {
                    foreach ( $languages as $language ) {
                        $sitepress->switch_lang( $language['language_code'] );
                        echo '<div class="dppp-pages-language" id="dppp-pages-language-' . esc_attr( sanitize_html_class( $language['language_code'] ) ) . '">';
                        echo '<p>&nbsp;&nbsp; ' . esc_html( sprintf( __( 'WPML (%s)', 'dppp' ), $language['translated_name'] ) ) . '</p>';
                        $pages = get_pages( $args );
                        if ( is_array( $pages ) && count( $pages ) > 0 ) {
                            foreach ( $pages as $page ) {
                                echo '<p class="dppp-page-paragraphs" id="dppp-page-paragraph-' . esc_attr( $page->ID )
                                    . '"><label><input type="checkbox" class="dppp-page-checkbox" id="dppp-page-checkbox-' . esc_attr( $page->ID ) . '" /> ';
                                if ( $page->post_title === '' ) {
                                    echo '<span class="dppp-page-title">';
                                    esc_html_e( '(no title)', 'dppp' );
                                    echo '</span></label></p>';
                                } else {
                                    echo '<span class="dppp-page-title">' . esc_html( $page->post_title ) . '</span></label></p>';
                                }
                            }
                        } else {
                            echo '<p class="dppp-no-pages">' . esc_html__( 'There are no published pages at the moment.', 'dppp' ) . '</p>';
                        }
                        echo '</div>';
                    }
                }

            // If the WPML plugin is not active, we just show all published pages
            } else {
                $pages = get_pages( $args );
                if ( is_array( $pages ) && count( $pages ) > 0 ) {
                    foreach ( $pages as $page ) {
                        echo '<p class="dppp-page-paragraphs" id="dppp-page-paragraph-' . esc_attr( $page->ID )
                            . '"><label><input type="checkbox" class="dppp-page-checkbox" id="dppp-page-checkbox-' . esc_attr( $page->ID ) . '" /> ';
                        if ( $page->post_title === '' ) {
                            echo '<span class="dppp-page-title">';
                            esc_html_e( '(no title)', 'dppp' );
                            echo '</span></label></p>';
                        } else {
                            echo '<span class="dppp-page-title">' . esc_html( $page->post_title ) . '</span></label></p>';
                        }
                    }
                } else {
                    echo '<p class="dppp-no-pages">' . esc_html__( 'There are no published pages at the moment.', 'dppp' ) . '</p>';
                }
            }
            ?>
        </div>
        <?php

        // If the WPML plugin is active, we show a checkbox option to also select translated pages of selected pages
        if ( is_plugin_active( 'sitepress-multilingual-cms/sitepress.php' ) ) {
            ?>
            <div class="dppp-select-wpml-translated-pages-contain">
                <label><input type="checkbox" id="dppp-select-wpml-translated-pages" name="dppp-select-wpml-translated-pages" />
                <?php esc_html_e( 'Also select all translated with WPML pages of selected pages', 'dppp' ); ?></label>
            </div>
            <?php
        }
        ?>

        <div class="dppp-under-pages-links">
            <div class="dppp-under-pages-links-left">
                <?php esc_html_e( 'Selected:', 'dppp' ); ?> <span></span>
            </div>
            <div class="dppp-under-pages-links-right">
                <a href="javascript:dpppSelectAllPages()"><?php esc_html_e( 'Select all', 'dppp' ); ?></a> /
                <a href="javascript:dpppDeselectAllPages()"><?php esc_html_e( 'Deselect all', 'dppp' ); ?></a>
            </div>
        </div>
    </div>
    <div id="dppp-custom-uri-type">
        <?php esc_html_e( '3. Custom URI condition', 'dppp' ); ?><br>
        <select name="dppp-custom-uri-select-condition" id="dppp-custom-uri-select-condition">
            <option value="equals" selected><?php esc_html_e( 'URI equals: ', 'dppp' ); ?></option>
            <option value="does-not-equal"><?php esc_html_e( 'URI does not equal: ', 'dppp' ); ?></option>
            <option value="contains"><?php esc_html_e( 'URI contains: ', 'dppp' ); ?></option>
            <option value="does-not-contain"><?php esc_html_e( 'URI does not contain: ', 'dppp' ); ?></option>
            <option value="starts-with"><?php esc_html_e( 'URI starts with: ', 'dppp' ); ?></option>
            <option value="does-not-start-with"><?php esc_html_e( 'URI does not start with: ', 'dppp' ); ?></option>
            <option value="ends-with"><?php esc_html_e( 'URI ends with: ', 'dppp' ); ?></option>
            <option value="does-not-end-with"><?php esc_html_e( 'URI does not end with: ', 'dppp' ); ?></option>
            <option value="regular-expression"><?php esc_html_e( 'URI matches the regular expression: ', 'dppp' ); ?></option>
        </select>
        <input type="text" name="dppp-custom-uri-value" id="dppp-custom-uri-value" />
        <div id="dppp-more-uri-conditions" class="dppp-display-none"></div>
        <div id="dppp-uri-condition-template" class="dppp-display-none">
            <div id="dppp-template-uri-condition-container">
                <div class="dppp-uri-conditions-inner">
                    <div class="dppp-uri-conditions-fields">
                        <select name="dppp-template-custom-uri-select-condition" id="dppp-template-custom-uri-select-condition">
                            <option value="equals" selected><?php esc_html_e( 'or URI equals: ', 'dppp' ); ?></option>
                            <option value="does-not-equal"><?php esc_html_e( 'or URI does not equal: ', 'dppp' ); ?></option>
                            <option value="contains"><?php esc_html_e( 'or URI contains: ', 'dppp' ); ?></option>
                            <option value="does-not-contain"><?php esc_html_e( 'or URI does not contain: ', 'dppp' ); ?></option>
                            <option value="starts-with"><?php esc_html_e( 'or URI starts with: ', 'dppp' ); ?></option>
                            <option value="does-not-start-with"><?php esc_html_e( 'or URI does not start with: ', 'dppp' ); ?></option>
                            <option value="ends-with"><?php esc_html_e( 'or URI ends with: ', 'dppp' ); ?></option>
                            <option value="does-not-end-with"><?php esc_html_e( 'or URI does not end with: ', 'dppp' ); ?></option>
                            <option value="regular-expression"><?php esc_html_e( 'or URI matches the regular expression: ', 'dppp' ); ?></option>
                        </select>
                        <input type="text" name="dppp-template-custom-uri-value" id="dppp-template-custom-uri-value" />
                    </div>
                    <div class="dppp-contain-remove-condition">
                        <a title="<?php esc_html_e( 'Remove condition', 'dppp' ); ?>" class="dppp-remove-condition-link"
                            href="javascript:dpppRemoveCondition( 'dppp-number-condition' )">
                            <span class="dashicons dashicons-no"></span>
                        </a>
                    </div>
                </div>
            </div>
        </div>
        <div class="dppp-align-right">
            <a id="dppp-add-more-conditions-link" href="javascript:void( 0 )" onclick="dpppAddMoreConditions( 2 )">
                <span class="dashicons dashicons-plus"></span><?php esc_html_e( 'Add more URI conditions', 'dppp' ); ?>
            </a>
        </div>
    </div>
    <a id="dppp-show-advanced" href="javascript:dpppShowHidden('dppp-contain-advanced', 'dppp-show-advanced')"><?php esc_html_e( 'Show Advanced Settings', 'dppp' ); ?></a>
    <div id="dppp-contain-advanced" class="dppp-display-none">
        <div class="dppp-contain-devices">
            <?php esc_html_e( '4. Affected devices', 'dppp' ); ?>
            <a id="dppp-read-more-devices"
                href="javascript:dpppShowHidden('dppp-notice-devices', 'dppp-read-more-devices')"><?php esc_html_e( 'Read more', 'dppp' ); ?></a>
            <br>
            <select name="dppp-select-devices" id="dppp-select-devices">
                <option value="all" selected><?php esc_html_e( 'All', 'dppp' ); ?></option>
                <option value="desktop"><?php esc_html_e( 'Desktop', 'dppp' ); ?></option>
                <option value="mobile"><?php esc_html_e( 'Mobile', 'dppp' ); ?></option>
            </select>
            <p id="dppp-notice-devices" class="dppp-display-none dppp-margin-top-5">
                <?php esc_html_e( 'Notice: For values different from "All", if you are using a caching plugin, it needs to be able to generate '
                    . 'different cache for different devices. '
                    . 'Otherwise the device of the first visitor to the page will determine the content of the cache and the deactivation rules for '
                    . 'that device would be applied to everyone. Also keep in mind that mobile includes tablets.', 'dppp' ); ?>
            </p>
        </div>
        <div class="dppp-contain-custom-function">
            <?php esc_html_e( '5. Custom condition function (for developers) (optional)', 'dppp' ); ?>
            <a id="dppp-read-more-custom-function"
                href="javascript:dpppShowHidden('dppp-hint-custom-function', 'dppp-read-more-custom-function')"><?php esc_html_e( 'Read more', 'dppp' ); ?></a>
            <input type="text" name="dppp-custom-condition-function" id="dppp-custom-condition-function" />
            <p id="dppp-hint-custom-function" class="dppp-display-none dppp-margin-top-5">
                <?php printf( esc_html__( 'Add the name of your own PHP function (without brackets) for an additional condition check. The deactivation rule will only be applied if '
                    . 'your function returns true. This is an additional condition, so the regular conditions of the rule also have to be met. '
                    . 'Create your own %smust-use plugin file%s inside wp-content/mu-plugins to add your function, so it is available before the normal plugins are loaded. '
                    . 'Keep in mind that in a must-use plugin some WordPress functions will not work, because they are loaded at a later point of the loading process.', 'dppp' ),
                    '<a href="https://wordpress.org/support/article/must-use-plugins/" target="_blank">', '</a>' );
                ?>
            </p>
        </div>
        <div class="dppp-contain-add-note">
            <?php esc_html_e( '6. Note (optional)', 'dppp' ); ?>
            <textarea name="dppp-rule-note" id="dppp-rule-note"></textarea>
        </div>
    </div>
    <div>
        <input type="button" id="dppp-button-add-selected-pages"
            class="button button-primary dppp-full-width" value="<?php esc_attr_e( 'Add New Rule', 'dppp' ); ?>" />
    </div>
    <?php
    wp_die();
}

// Handles the ajax request for displaying the box with the explanation of the rule
function dppp_explain_rule_callback() {
    $rule_id = intval( $_POST['rule_id'] );
    echo '<h2>';
    echo esc_html( sprintf( __( 'Explanation of the rule with ID %d', 'dppp' ), $rule_id ) );
    echo '</h2>';
    echo '<input type="button" class="dppp-close-button button" id="dppp-button-close-explain" value="' . esc_attr__( 'Close', 'dppp' ) . '" />';
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( '<p class="dppp-clear">' . esc_html__( 'Error: Invalid security nonce.', 'dppp' ) . '</p>' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( '<p class="dppp-clear">' . esc_html__( 'Error: You do not have sufficient permissions to make this request.', 'dppp' ) . '</p>' );
    }
    global $wpdb;
    $result = $wpdb->get_row( $wpdb->prepare( "SELECT plugin_file, deactivation_type, uri_condition, uri_value, devices, note, uri_type, page_ids, status, custom_function FROM "
        . "{$wpdb->prefix}dppp_plugin_deactivation_rules WHERE rule_id = %d", $rule_id ), ARRAY_A );
    if ( ! empty( $result ) ) {

        if ( 'selected' === $result['deactivation_type'] ) {
            $deactivation_type = __( 'if the URI ', 'dppp' );
        } elseif ( 'except' === $result['deactivation_type'] ) {
            $deactivation_type = __( 'always except if the URI ', 'dppp' );
        } else {
            $deactivation_type = '[' . __( 'Error', 'dppp' ) . ']';
        }
        if ( 'mobile' === $result['devices'] ) {
            $devices_text = __( 'on mobile devices ', 'dppp' );
        } elseif ( 'all' === $result['devices'] ) {
            $devices_text = __( 'on all devices ', 'dppp' );
        } elseif ( 'desktop' === $result['devices'] ) {
            $devices_text = __( 'on desktop devices ', 'dppp' );
        } else {
            $devices_text = '[' . __( 'Error', 'dppp' ) . ']';
        }

        if ( strpos( $result['plugin_file'], ';;' ) !== false ) {
            $plugins = explode( ';;', $result['plugin_file'] );
            sort( $plugins );
            $group_hash = dppp_get_plugin_group_hash_from_array( $plugins );
            $plugin_groups = dppp_get_option( 'dppp-plugin-groups' );
            echo '<p class="dppp-clear">' . esc_html__( 'Deactivates the group of plugins named', 'dppp' ) . ' '
                . '<strong>' . esc_html( stripslashes( $plugin_groups[ $group_hash ]['name'] ) ) . '</strong> ';
        } else {
            $plugin_data = get_plugin_data( WP_PLUGIN_DIR . '/' . $result['plugin_file'] );
            echo '<p class="dppp-clear">' . esc_html__( 'Deactivates the plugin', 'dppp' ) . ' ' . '<strong>' . esc_html( $plugin_data['Name'] ) . '</strong> ';
        }

        echo esc_html( $devices_text ) . esc_html( $deactivation_type ) . '<span class="dppp-explain-uri-value">';

        dppp_output_uri_conditions_and_values( $result['uri_type'], $result['uri_condition'], $result['uri_value'] );

        echo '</span></p>';

        if ( 'page' === $result['uri_type'] ) {
            if ( strpos( $result['page_ids'], ',' ) !== false ) {
                $page_ids = str_replace( ',', ', ', $result['page_ids'] );
                echo '<p>' .
                    sprintf(
                        esc_html__( 'The URI value will update automatically for this rule if the permalink of any of the pages with ID %s changes.', 'dppp' ),
                        esc_html( $page_ids )
                    )
                    . '</p>';
            } else {
                echo '<p>' .
                    sprintf(
                        esc_html__( 'The URI value will update automatically for this rule if the permalink of the %spage with ID %s%s changes.', 'dppp' ),
                        '<a target="_blank" href="' . esc_url( admin_url( 'post.php?post=' . intval( $result['page_ids'] ) . '&action=edit' ) ) . '">',
                        esc_html( $result['page_ids'] ), '</a>'
                    )
                    . '</p>';
            }
            if ( get_option( 'permalink_structure' ) !== '' && dppp_get_option( 'dppp-uri-parameters' ) === 'ignore-all' ) {
                echo '<p>' . esc_html__( 'When this rule is applied, all URI parameters in the current server URI are ignored.', 'dppp' ) . '</p>';
            }
            if ( dppp_get_option( 'dppp-uri-parameters' ) === 'ignore-chosen' ) {
                echo '<p>' . esc_html__( 'When this rule is applied, some chosen URI parameters in the current server URI are ignored.', 'dppp' ) . '</p>';
            }
        } elseif ( 'custom' === $result['uri_type'] ) {
            if ( get_option( 'permalink_structure' ) !== '' && dppp_get_option( 'dppp-uri-parameters-custom' ) === 'ignore-all' ) {
                echo '<p>' . esc_html__( 'When this rule is applied, all URI parameters in the current server URI are ignored.', 'dppp' ) . '</p>';
            }
            if ( dppp_get_option( 'dppp-uri-parameters-custom' ) === 'ignore-chosen' ) {
                echo '<p>' . esc_html__( 'When this rule is applied, some chosen URI parameters in the current server URI are ignored.', 'dppp' ) . '</p>';
            }
        } else {
            echo '[' . esc_html__( 'Error', 'dppp' ) . ']';
        }
        if ( ! empty( $result['custom_function'] ) ) {
            echo '<p>' . sprintf( esc_html__( 'This rule will only be applied if the PHP function %s returns true.', 'dppp' ), esc_html( $result['custom_function'] ) )
                . '</p>';
        }
        if ( 'started' === $result['status'] ) {
            // Nothing
        } elseif ( 'paused' === $result['status'] ) {
            echo '<p>' . esc_html__( 'This rule is paused at the moment, and therefor it has no effect.', 'dppp' ) . '</p>';
        } else {
            echo '[' . esc_html__( 'Error', 'dppp' ) . ']';
        }
        if ( dppp_get_option( 'dppp-affect-back-end' ) === 'off' || ( dppp_get_option( 'dppp-affect-back-end' ) !== 'off' && 'custom' !== $result['uri_type'] ) ) {
            echo '<p>' .
                sprintf(
                    esc_html__( 'This rule affects only %sGET requests%s on the front-end.', 'dppp' ),
                    '<a href="https://www.w3schools.com/tags/ref_httpmethods.asp" target="_blank">', '</a>'
                )
                . '</p>';
        } else {
            echo '<p>' .
                sprintf(
                    esc_html__( 'This rule affects only %sGET requests%s on both the front-end and back-end.', 'dppp' ),
                    '<a href="https://www.w3schools.com/tags/ref_httpmethods.asp" target="_blank">', '</a>'
                )
                . '</p>';
        }

        if ( '' !== $result['note'] && null !== $result['note'] ) {
            echo '<p class="dppp-explain-note-paragraph"><b>' . esc_html__( 'User note:', 'dppp' ) . '</b></p><div class="dppp-explain-note">' . nl2br( esc_html(  $result['note'] ) ) . '</div>';
        }

    } else {
        echo '<p class="dppp-clear">' . esc_html__( 'Error: Rule not found', 'dppp' ) . '</p>';
    }
    wp_die();
}

// Handles the ajax request for saving the plugin settings
function dppp_save_settings_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }
    $chosen_uri_parameters = sanitize_textarea_field( $_POST['chosen_uri_parameters'] );
    if ( ! dppp_is_comma_separated_uri_parameters( $chosen_uri_parameters ) && ! empty( $chosen_uri_parameters ) ) {
        wp_die( 'invalid-data' );
    }

    $previous_trailing_slash = dppp_get_option( 'dppp-trailing-slash' );
    $previous_affect_back_end = dppp_get_option( 'dppp-affect-back-end' );

    $debug_mode = sanitize_html_class( $_POST['debug_mode'] );
    $inactive_plugins = sanitize_html_class( $_POST['inactive_plugins'] );
    $completely_remove = sanitize_html_class( $_POST['completely_remove'] );
    $groups_first = sanitize_html_class( $_POST['groups_first'] );
    $trailing_slash = sanitize_html_class( $_POST['trailing_slash'] );
    $affect_back_end = sanitize_html_class( $_POST['affect_back_end'] );
    $uri_parameters = sanitize_html_class( $_POST['uri_parameters'] );
    $uri_parameters_custom = sanitize_html_class( $_POST['uri_parameters_custom'] );
    $plugin_icon_images = sanitize_html_class( $_POST['plugin_icon_images'] );
    $local_mode = sanitize_html_class( $_POST['local_mode'] );
    $rules_first = sanitize_html_class( $_POST['rules_first'] );
    $hide_in_group = sanitize_html_class( $_POST['hide_in_group'] );
    $priority = intval( $_POST['priority'] );
    $visual_editors = sanitize_html_class( $_POST['affect_visual_editors'] );
    $edit_post_parameter = sanitize_html_class( $_POST['edit_post_parameter'] );
    $admin_menu_cache_time = intval( $_POST['admin_menu_cache_time'] );
    update_option( 'dppp-debug-mode', $debug_mode );
    update_option( 'dppp-inactive-plugins', $inactive_plugins );
    update_option( 'dppp-completely-remove', $completely_remove );
    update_option( 'dppp-groups-first', $groups_first );
    update_option( 'dppp-trailing-slash', $trailing_slash );
    update_option( 'dppp-affect-back-end', $affect_back_end );
    update_option( 'dppp-uri-parameters', $uri_parameters );
    update_option( 'dppp-uri-parameters-custom', $uri_parameters_custom );
    update_option( 'dppp-chosen-uri-parameters', $chosen_uri_parameters );
    update_option( 'dppp-plugin-icons', $plugin_icon_images );
    update_option( 'dppp-local-mode', $local_mode );
    update_option( 'dppp-rules-first', $rules_first );
    update_option( 'dppp-hide-plugins-in-group', $hide_in_group );
    update_option( 'dppp-priority', $priority );
    update_option( 'dppp-affect-visual-editors', $visual_editors );
    update_option( 'dppp-edit-post-parameter', $edit_post_parameter );
    update_option( 'dppp-admin-menu-cache-time', $admin_menu_cache_time );

    // If the trailing slash option has changed, we refresh the page URIs
    if ( $previous_trailing_slash !== $_POST['trailing_slash'] ) {
        dppp_refresh_page_uri_values();
    }

    // If the option for affecting the back-end has changed to using admin menu cache, we recreate the cache.
    if ( $previous_affect_back_end !== $affect_back_end && 'cache' === $affect_back_end ) {
        wp_die( 'done-recreate-cache' );
    }

    // If the option for affecting the back-end has changed to not using admin menu cache, we delete the cache.
    if ( $previous_affect_back_end !== $affect_back_end && 'cache' === $previous_affect_back_end ) {
        dppp_delete_admin_menus_cache();
    }

    wp_die( 'done' );
}

// Handles the ajax request for saving the state of the debug box (to be fully shown or minimized)
function dppp_save_debug_box_callback() {
    if ( ! check_ajax_referer( 'dppp_debug_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }
    $state = sanitize_html_class( $_POST['state'] );
    update_option( 'dppp-debug-box-state-' . intval( get_current_user_id() ), $state );
    wp_die( 'done' );
}

// Handles the ajax request for adding a new rule to the database
function dppp_save_rule_data_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }
    $deactivation_type = sanitize_html_class( $_POST['deactivation_type'] );
    $uri_type = sanitize_html_class( $_POST['uri_type'] );
    $custom_condition = sanitize_html_class( $_POST['custom_condition'] );
    if ( 'custom' === $uri_type ) {
        if ( 'regular-expression' === $custom_condition ) {
            $custom_value = stripslashes( $_POST['custom_value'] );
            if ( ! dppp_is_valid_regular_expression( $custom_value ) ) {
                wp_die( 'invalid-regex' );
            }
        } else {
            $custom_value = dppp_escape_uri( $_POST['custom_value'] );
            if ( $custom_value !== $_POST['custom_value'] ) {
                wp_die( 'invalid-data' );
            }
        }
        $more_custom_conditions = sanitize_html_class( $_POST['more_custom_conditions'] );
        if ( strpos( $more_custom_conditions, 'regular-expression' ) !== false ) {
            $more_custom_values = stripslashes( $_POST['more_custom_values'] );
        } else {
            $more_custom_values = dppp_escape_uri( $_POST['more_custom_values'] );
            if ( $more_custom_values !== $_POST['more_custom_values'] ) {
                wp_die( 'invalid-data' );
            }
        }
        if ( ! in_array( $custom_condition, Array( 'equals', 'does-not-equal', 'contains', 'does-not-contain', 'starts-with', 'does-not-start-with', 'ends-with',
            'does-not-end-with', 'regular-expression' ) ) || ! dppp_are_more_conditions_and_values_valid( $more_custom_conditions, $more_custom_values ) ) {
            wp_die( 'invalid-data' );
        }
    }
    $devices = sanitize_html_class( $_POST['devices'] );
    $note = sanitize_textarea_field( stripslashes_deep( $_POST['note'] ) );
    $custom_function = sanitize_text_field( $_POST['custom_function'] );
    $select_translated_pages_wpml = sanitize_html_class( $_POST['select_translated_pages_wpml'] );
    $pages = dppp_sanitize_comma_separated_numbers( $_POST['pages'] );

    // If this is equal to 'dppp-group', then it is a group rule with a group hash in $_POST['hash']
    $plugin_file = dppp_sanitize_plugin_file( $_POST['plugin_file'] );

    if ( ! empty( $pages ) ) {
        $pages = substr_replace( $pages, "", -1 );
    }
    if ( ! in_array( $deactivation_type, Array( 'selected', 'except' ) ) || ! in_array( $uri_type, Array( 'page', 'custom' ) )
        || ! in_array( $devices, Array( 'all', 'desktop', 'mobile' ) )
        || ( ! empty( $pages ) && ! dppp_is_comma_separated_numbers( $pages ) )
        || ( ! dppp_is_valid_plugin_file( $plugin_file ) && 'dppp-group' !== $plugin_file )
        || dppp_strip_whitespace( $_POST['custom_function'] ) !== $_POST['custom_function']
        || ( ! empty( $_POST['custom_function'] ) && ! dppp_is_valid_php_function_name( $custom_function ) ) ) {
        wp_die( 'invalid-data' );
    }
    if ( 'page' === $uri_type && empty( $pages ) ) {
        wp_die( 'select-page' );
    }
    if ( 'custom' === $uri_type && empty( $custom_value ) ) {
        wp_die( 'enter-uri' );
    }
    if ( 'custom' === $uri_type && ! dppp_are_more_conditions_and_values_regex_valid( $more_custom_conditions, $more_custom_values ) ) {
        wp_die( 'invalid-regex' );
    }

    // If we are adding a rule to a plugin group, we generate the plugin file string from the group hash
    if ( 'dppp-group' === $plugin_file ) {
        $group_hash = sanitize_html_class( $_POST['hash'] );
        $plugin_groups = dppp_get_option( 'dppp-plugin-groups' );
        if ( ! is_array( $plugin_groups ) || ! array_key_exists( $group_hash, $plugin_groups ) ) {
            wp_die( 'group-does-not-exist' );
        }
        $plugin_file = dppp_get_plugin_files_list_from_array( $plugin_groups[ $group_hash ]['plugins'] );
    }

    global $wpdb;
    $inserted_ids = Array();

    // It is a custom rule
    if ( 'custom' === $uri_type ) {
        $more_custom_conditions = str_replace( '--dppp-or--', '[dppp-or]', $more_custom_conditions );
        $more_custom_values = str_replace( '--dppp-or--', '[dppp-or]', $more_custom_values );
        $custom_condition .= $more_custom_conditions;
        $custom_value .= $more_custom_values;
        $status = $wpdb->query( $wpdb->prepare( "SELECT rule_id FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE plugin_file = %s "
            . "AND deactivation_type = %s AND uri_type = 'custom' AND page_ids = '0' AND uri_condition = %s AND uri_value = %s AND devices = %s",
            $plugin_file, $deactivation_type, $custom_condition, $custom_value, $devices ) );
        if ( $status === false ) {
            wp_die( 'could-not-check-if-exists' );
        }
        if ( $status !== 0 ) {
            wp_die( 'already-exists' );
        }
        if ( dppp_is_valid_url( $custom_value ) ) {
            wp_die( 'uri-not-url' );
        }
        $status_insert = $wpdb->insert(
        	"{$wpdb->prefix}dppp_plugin_deactivation_rules",
        	array(
        		'status' => 'started',
        		'plugin_file' => $plugin_file,
        		'deactivation_type' => $deactivation_type,
        		'uri_type' => 'custom',
        		'page_ids' => '0',
        		'uri_condition' => $custom_condition,
        		'uri_value' => $custom_value,
        		'devices' => $devices,
        		'custom_function' => $custom_function,
                'note' => $note,
        		'unix_time_added' => time(),
        	),
        	array(
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%d',
        	)
        );
        if ( $status_insert === false ) {
            wp_die( 'could-not-insert' );
        }
        $inserted_ids[] = $wpdb->insert_id;

    // It is a page rule
    } else {
        $page_uris = '';
        $page_ids = explode( ',', $pages );

        // If the WPML plugin is active, and the option to also select translated pages of selected pages was checked, we do that
        if ( is_plugin_active( 'sitepress-multilingual-cms/sitepress.php' ) && 'yes' === $select_translated_pages_wpml ) {
            global $sitepress;
            $add_page_ids = Array();
            foreach ( $page_ids as $page_id ) {
                $translation_group_id = apply_filters( 'wpml_element_trid', NULL, $page_id, 'post_page' );
                $translations = apply_filters( 'wpml_get_element_translations', NULL, $translation_group_id, 'post_page' );
                foreach ( $translations as $translation ) {
                    $add_page_ids[] = intval( $translation->element_id );
                }
            }
            foreach ( $add_page_ids as $add_page_id ) {
                if ( ! in_array( $add_page_id, $page_ids ) ) {
                    $page_ids[] = $add_page_id;
                }
            }
        }

        // We sort the array and set the $pages var so the order is the same always and we can properly check if the rule exists
        sort( $page_ids );
        $pages = implode( ',', $page_ids );

        $loop = 1;
        foreach ( $page_ids as $page_id ) {
            $page_uri = dppp_escape_uri( dppp_get_page_uri_or_uris( $page_id ) );
            $page_uris .= $page_uri;
            if ( $loop !== count( $page_ids ) ) {
                $page_uris .= '[dppp-or]';
            }
            $loop++;
        }
        $status = $wpdb->query( $wpdb->prepare( "SELECT rule_id FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE plugin_file = %s AND "
            . "deactivation_type = %s AND uri_type = 'page' AND page_ids = %s AND uri_condition = 'equals' AND uri_value = %s AND devices = %s",
            $plugin_file, $deactivation_type, $pages, $page_uris, $devices ) );
        if ( $status !== 0 ) {
            wp_die( 'already-exists' );
        }
        $status_insert = $wpdb->insert(
        	"{$wpdb->prefix}dppp_plugin_deactivation_rules",
        	array(
        		'status' => 'started',
        		'plugin_file' => $plugin_file,
        		'deactivation_type' => $deactivation_type,
        		'uri_type' => 'page',
        		'page_ids' => $pages,
        		'uri_condition' => 'equals',
        		'uri_value' => $page_uris,
        		'devices' => $devices,
        		'custom_function' => $custom_function,
                'note' => $note,
        		'unix_time_added' => time(),
        	),
        	array(
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%d',
        	)
        );
        if ( $status_insert === false ) {
            wp_die( 'could-not-insert' );
        }
        $inserted_ids[] = $wpdb->insert_id;
    }
    $inserted_count = count( $inserted_ids );
    $last_index = $inserted_count - 1;
    $last_inserted_id = $inserted_ids[ $last_index ];
    echo '[done]';
    foreach ( $inserted_ids as $inserted_id ) {
        echo esc_attr( intval( $inserted_id ) );
        if ( $last_inserted_id !== $inserted_id ) {
            echo ',';
        }
    }
    wp_die();
}

// When the permalink structure is updated (even without change), we update the URI values for the page type rules we have
function dppp_permalinks_updated() {

    /*
     * This updated the page URI values when the permalink structure is updated even if there is no change.
     * It is useful since when a user changes an option of some plugin that affects the permalinks,
     * it usually suggests a permalink update (re-save), and we can catch the change too.
     */
    if ( isset( $_SERVER['REQUEST_URI'] ) && strpos( $_SERVER['REQUEST_URI'], 'options-permalink.php' ) !== false && isset( $_GET['settings-updated'] )
        && ( true === $_GET['settings-updated'] || 'true' === $_GET['settings-updated'] ) && current_user_can( 'manage_options') ) {
        dppp_refresh_page_uri_values();

    // If the previous detection method fails for some reason, we will absolutely update the page URIs if the permalink structure has actually changed
    } else {
        add_action( 'permalink_structure_changed', 'dppp_permalinks_changed', 10, 2 );
    }
}

/**
 * When the permalink structure changes, we update the URI values for the page type rules we have
 * @param string $old_permalink_structure
 * @param string $permalink_structure
 */
function dppp_permalinks_changed( $old_permalink_structure, $permalink_structure ) {
    dppp_refresh_page_uri_values();
}

/**
 * When a page is updated, we update the URI values of the
 * @param int $post_ID
 * @param object $post_after
 * @param object $post_before
 */
function dppp_update_page_rules( $post_ID, $post_after, $post_before ){
    if ( 'page' === $post_before->post_type && 'publish' === $post_after->post_status ) {
        dppp_refresh_page_uri_values();
    }
}

/**
 * When a page is trashed, we update the URI values of the rules
 * @param int $post_id
 */
function dppp_refresh_rules_after_trashed( $post_id ) {
    if ( get_post_type( $post_id ) === 'page' ) {
        dppp_refresh_page_uri_values();
    }
}

/**
 * When the option that says which page to be a front page is created, we update the URI values of the page type rules
 * @param string $option_name
 * @param mixed $option_value
 */
function dppp_add_option_front_page_refresh( $option_name, $option_value ) {
    dppp_refresh_page_uri_values();
}

/**
 * When the option that says which page to be a front page is updated, we update the URI values of the page type rules
 * @param mixed $old_value
 * @param mixed $option_value
 */
function dppp_update_option_front_page_refresh( $old_value, $option_value ) {
    dppp_refresh_page_uri_values();
}

/**
 * Adds a new element in an array on the exact place we want (if possible).
 * We use this when adding a custom column or an action link somewhere in the admin panel.
 * @param array $original_array
 * @param string $add_element_key
 * @param mixed $add_element_value
 * @param string $add_before_key
 * @return array
 */
function dppp_add_element_to_array( $original_array, $add_element_key, $add_element_value, $add_before_key ) {
    $is_added = 0;
    $new_array = array();
    foreach( $original_array as $key => $value ) {
        if ( $key === $add_before_key ) {
      	    $new_array[ $add_element_key ] = $add_element_value;
            $is_added = 1;
        }
        $new_array[ $key ] = $value;
    }
    if ( 0 === $is_added ) {
        $new_array[ $add_element_key ] = $add_element_value;
    }
    return $new_array;
}

/**
 * Returns the path to the must use plugins folder
 * @return string
 */
function dppp_get_mu_path() {
    if ( defined( 'WPMU_PLUGIN_DIR' ) ) {
        return WPMU_PLUGIN_DIR;
    } else {
        return WP_CONTENT_DIR . '/mu-plugins';
    }
}

// Deletes the must use plugin file that we added
function dppp_remove_mu_plugin() {
    $mu_plugins = get_mu_plugins();
    if ( is_array( $mu_plugins ) && array_key_exists( 'dppp-mu-deactivate-plugins-per-page.php', $mu_plugins ) ) {
        $file_path = dppp_get_mu_path() . '/dppp-mu-deactivate-plugins-per-page.php';
        if ( file_exists( $file_path ) ) {
            @unlink( $file_path );
        }
    }
}

/**
 * Adds the must use plugin to the must use plugins folder and returns true on success
 * @return bool
 */
function dppp_check_and_add_mu_plugin() {
    $mu_plugins = get_mu_plugins();
    if ( ! is_array( $mu_plugins ) || ! array_key_exists( 'dppp-mu-deactivate-plugins-per-page.php', $mu_plugins ) ) {
        $mu_path = dppp_get_mu_path();
        $mu_from_plugin_folder = dppp_plugin_dir_path() . 'mu-plugin/dppp-mu-deactivate-plugins-per-page.php';
        if ( file_exists( $mu_from_plugin_folder ) ) {
            if ( ! file_exists( $mu_path ) ) {
                $status = @mkdir( $mu_path, 0755, true );
                if ( $status ) {
                    @copy( $mu_from_plugin_folder, $mu_path . '/dppp-mu-deactivate-plugins-per-page.php' );
                }
            } else {
                @copy( $mu_from_plugin_folder, $mu_path . '/dppp-mu-deactivate-plugins-per-page.php' );
            }
        }
        $mu_plugins = get_mu_plugins();
        if ( ! is_array( $mu_plugins ) || ! array_key_exists( 'dppp-mu-deactivate-plugins-per-page.php', $mu_plugins ) ) {
            return false;
        }
    }

    // The mu plugin is there, but the version is not correct
    if ( $mu_plugins['dppp-mu-deactivate-plugins-per-page.php']['Version'] !== DPPP_DEACTIVATE_PLUGINS_PER_PAGE_VERSION ) {
        return 'version-mismatch';
    }

    return true;
}

// Adds the new must use plugin file (overwriting the old one) to the must use plugins folder
function dppp_update_mu_plugin() {
    $mu_path = dppp_get_mu_path();
    $mu_from_plugin_folder = dppp_plugin_dir_path() . 'mu-plugin/dppp-mu-deactivate-plugins-per-page.php';
    if ( file_exists( $mu_from_plugin_folder ) ) {
        if ( ! file_exists( $mu_path ) ) {
            $status = @mkdir( $mu_path, 0755, true );
            if ( $status ) {
                @copy( $mu_from_plugin_folder, $mu_path . '/dppp-mu-deactivate-plugins-per-page.php' );
            }
        } else {
            @copy( $mu_from_plugin_folder, $mu_path . '/dppp-mu-deactivate-plugins-per-page.php' );
        }
    }
}

// Creates the database table where we store the plugin deactivation rules
function dppp_create_database_table() {
    global $wpdb;
    $collate = '';
    if ( $wpdb->has_cap( 'collation' ) ) {
        $collate = $wpdb->get_charset_collate();
    }
    $sql = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}dppp_plugin_deactivation_rules ( "
        . "rule_id bigint(20) NOT NULL AUTO_INCREMENT, "
        . "status varchar(100) DEFAULT NULL, "
        . "plugin_file longtext DEFAULT NULL, "
        . "deactivation_type varchar(100) DEFAULT NULL, "
        . "uri_type varchar(100) DEFAULT NULL, "
        . "page_ids longtext DEFAULT NULL, "
        . "uri_condition longtext DEFAULT NULL, "
        . "uri_value longtext DEFAULT NULL, "
        . "devices varchar(100) DEFAULT NULL, "
        . "note longtext DEFAULT NULL, "
        . "custom_function longtext DEFAULT NULL, "
        . "unix_time_added int(11) NOT NULL, "
        . "PRIMARY KEY  (rule_id) "
        . ") " . $collate . ";";
    require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
    dbDelta( $sql );
    update_option( 'dppp-database-table-created', 'yes' );
}

/*
 * Outputs the HTML code needed to display the plugin settings form
 * @param string $add_class_inner_section
 */
function dppp_output_settings_content( $add_class_inner_section ) {
    ?>
    <div class="dppp-inner-settings-section<?php echo esc_attr( $add_class_inner_section ); ?>">
        <h2>
            <span>
                <?php esc_html_e( 'Rules Settings', 'dppp' ); ?>
            </span>
        </h2>
        <div class="dppp-setting-row">
            <div class="dppp-setting-left">
                <h3><?php esc_html_e( 'Back-end deactivation', 'dppp' ); ?></h3>
                <p>
                    <?php esc_html_e( 'Affect also the back-end by rules with Custom URI selection type.', 'dppp' ); ?>
                    <a id="dppp-hint-link-affect-back-end" href="javascript:dpppShowHideHint('affect-back-end')">
                        <?php esc_html_e( 'Read more', 'dppp' ); ?>
                    </a>
                    <i id="dppp-hint-affect-back-end" class="dppp-hint-text">
                        <?php
                            esc_html_e( 'If this is enabled, all deactvation rules with Custom URI selection type will also affect the back-end dashboard area. '
                                . 'To avoid problems, some dashboard pages will still not be affected, like the ones related to controlling, '
                                . 'installing, or updating plugins. When admin menu caching is also enabled, a cached version of the admin menus will '
                                . 'be loaded on admin pages where we have deactivated at least one plugin via a rule with Custom URI selection. '
                                . 'This way you will still see the menus created by deactivated plugins. Cached admin menus affect only admin '
                                . 'users and the cache is separate per user. When you activate/deactivate a plugin or activate a theme '
                                . 'the cache will be cleared.', 'dppp' );
                        ?>
                    </i>
                </p>
            </div>
            <div class="dppp-setting-right">
                <?php
                dppp_setting_select(
                    'dppp-affect-back-end',
                    Array( 'on', 'cache', 'off' ),
                    Array(
                        __( 'Enable', 'dppp' ),
                        __( 'Enable and cache admin menus', 'dppp' ),
                        __( 'Disable', 'dppp' ),
                    )
                );
                ?>
            </div>
        </div>
    </div>
    <div class="dppp-inner-settings-section<?php echo esc_attr( $add_class_inner_section ); ?>">
        <h2>
            <span>
                <?php esc_html_e( 'Admin Page Settings', 'dppp' ); ?>
            </span>
        </h2>
        <div class="dppp-setting-row">
            <div class="dppp-setting-left">
                <h3><?php esc_html_e( 'Inactive plugins', 'dppp' ); ?></h3>
                <p>
                    <?php esc_html_e( 'Show the rules for inactive plugins.', 'dppp' ); ?>
                    <a id="dppp-hint-link-inactive-plugins" href="javascript:dpppShowHideHint('inactive-plugins')">
                        <?php esc_html_e( 'Read more', 'dppp' ); ?>
                    </a>
                    <i id="dppp-hint-inactive-plugins" class="dppp-hint-text">
                        <?php
                            esc_html_e( 'Above on the current page you will be able to see and edit the rules for currently inactive plugins.', 'dppp' );
                        ?>
                    </i>
                </p>
            </div>
            <div class="dppp-setting-right">
                <?php
                dppp_setting_select(
                    'dppp-inactive-plugins',
                    Array( 'on', 'off' ),
                    Array(
                        __( 'Enable', 'dppp' ),
                        __( 'Disable', 'dppp' ),
                    )
                );
                ?>
            </div>
        </div>
        <div class="dppp-setting-row">
            <div class="dppp-setting-left">
                <h3><?php esc_html_e( 'Plugin icons', 'dppp' ); ?></h3>
                <p>
                    <?php esc_html_e( 'Show the plugin icon image next to the plugin name.', 'dppp' ); ?>
                </p>
            </div>
            <div class="dppp-setting-right">
                <?php
                dppp_setting_select(
                    'dppp-plugin-icons',
                    Array( 'on', 'off' ),
                    Array(
                        __( 'Enable', 'dppp' ),
                        __( 'Disable', 'dppp' ),
                    )
                );
                ?>
            </div>
        </div>
        <div class="dppp-setting-row">
            <div class="dppp-setting-left">
                <h3><?php esc_html_e( 'Plugin groups first', 'dppp' ); ?></h3>
                <p>
                    <?php esc_html_e( 'Show the Plugin Groups section above the Active Plugins section.', 'dppp' ); ?>
                </p>
            </div>
            <div class="dppp-setting-right">
                <?php
                dppp_setting_select(
                    'dppp-groups-first',
                    Array( 'on', 'off' ),
                    Array(
                        __( 'Enable', 'dppp' ),
                        __( 'Disable', 'dppp' ),
                    )
                );
                ?>
            </div>
        </div>
        <div class="dppp-setting-row">
            <div class="dppp-setting-left">
                <h3><?php esc_html_e( 'Plugins with rules first', 'dppp' ); ?></h3>
                <p>
                    <?php esc_html_e( 'Show the plugins that have rules at the top in the Active / Inactive Plugins sections above.', 'dppp' ); ?>
                </p>
            </div>
            <div class="dppp-setting-right">
                <?php
                dppp_setting_select(
                    'dppp-rules-first',
                    Array( 'on', 'off' ),
                    Array(
                        __( 'Enable', 'dppp' ),
                        __( 'Disable', 'dppp' ),
                    )
                );
                ?>
            </div>
        </div>
        <div class="dppp-setting-row">
            <div class="dppp-setting-left">
                <h3><?php esc_html_e( 'Hide plugins part of a group', 'dppp' ); ?></h3>
                <p>
                    <?php esc_html_e( 'The Active / Inactive Plugins sections will hide plugins part of a group, unless they have rules.',
                        'dppp' ); ?>
                </p>
            </div>
            <div class="dppp-setting-right">
                <?php
                dppp_setting_select(
                    'dppp-hide-plugins-in-group',
                    Array( 'on', 'off' ),
                    Array(
                        __( 'Enable', 'dppp' ),
                        __( 'Disable', 'dppp' ),
                    )
                );
                ?>
            </div>
        </div>
    </div>
    <div class="dppp-inner-settings-section<?php echo esc_attr( $add_class_inner_section ); ?>">
        <h2>
            <span>
                <?php esc_html_e( 'General Settings', 'dppp' ); ?>
            </span>
        </h2>
        <div class="dppp-setting-row">
            <div class="dppp-setting-left">
                <h3><?php esc_html_e( 'Debug mode', 'dppp' ); ?></h3>
                <p>
                    <?php esc_html_e( 'Show to admins a useful debug box in the bottom right corner.', 'dppp' ); ?>
                    <a id="dppp-hint-link-debug-mode" href="javascript:dpppShowHideHint('debug-mode')">
                        <?php esc_html_e( 'Read more', 'dppp' ); ?>
                    </a>
                    <i id="dppp-hint-debug-mode" class="dppp-hint-text">
                        <?php
                            esc_html_e( 'While debug mode is enabled, administrators will see a debug box in the bottom right corner on the'
                                . ' front-end (and possibly back-end depending on the settings) of the site. '
                                . 'It shows the status of the plugins and the URI for the current page. On some back-end pages, '
                                . 'it will not show even if the setting to affect back-end is enabled, since we have excluded them by the effect of this plugin.',
                                'dppp' );
                        ?>
                    </i>
                </p>
            </div>
            <div class="dppp-setting-right">
                <?php
                dppp_setting_select(
                    'dppp-debug-mode',
                    Array( 'on', 'back', 'front', 'off' ),
                    Array(
                        __( 'Enable', 'dppp' ),
                        __( 'Enable on back-end', 'dppp' ),
                        __( 'Enable on front-end', 'dppp' ),
                        __( 'Disable', 'dppp' ),
                    )
                );
                ?>
            </div>
        </div>
        <div class="dppp-setting-row">
            <div class="dppp-setting-left">
                <h3><?php esc_html_e( 'Local mode', 'dppp' ); ?></h3>
                <p>
                    <?php esc_html_e( 'All deactivations made by this plugin will only affect your device (useful for testing). '
                        . 'Make your device local to be affected. Warning: Caching plugins can make changes public!', 'dppp' ); ?>
                    <a id="dppp-hint-link-local-mode" href="javascript:dpppShowHideHint('local-mode')">
                        <?php esc_html_e( 'Read more', 'dppp' ); ?>
                    </a>
                    <i id="dppp-hint-local-mode" class="dppp-hint-text">
                        <?php
                            esc_html_e( 'Enable local mode and also make your device local so you are the only one that is affected by this plugin. '
                                . 'A cookie will be created on your device so we recognize it as a local device.', 'dppp' );
                            echo ' ';
                        $cookie = dppp_get_option( 'dppp-local-mode-cookie' );
                        echo esc_html( sprintf( __( 'If you are using a caching plugin, it can make local mode useless by caching the changes you make and '
                            . 'serving them to everyone. To avoid this it is best to disable caching while testing in local mode. '
                            . 'Alternatively you can disable caching only for visitors with the cookie named %s (if your caching plugin allows that), '
                            . 'which indicates it is a local device.', 'dppp' ), $cookie ) );
                        ?>
                    </i>
                </p>
            </div>
            <div class="dppp-setting-right">
                <?php
                dppp_setting_select(
                    'dppp-local-mode',
                    Array( 'on', 'off' ),
                    Array(
                        __( 'Enable', 'dppp' ),
                        __( 'Disable', 'dppp' ),
                    )
                );
                if ( dppp_is_local_device() ) {
                    ?>
                    <p>
                        <?php esc_html_e( 'Your device is a local device.', 'dppp' ) ?>
                        <a id="dppp-link-local-mode" href="javascript:dpppRemoveLocalDevice()">
                            <?php esc_html_e( 'Remove device', 'dppp' ); ?>
                        </a>
                    </p>
                    <?php
                } else {
                    ?>
                    <p>
                        <a id="dppp-link-local-mode" href="javascript:dpppMakeDeviceLocal()">
                            <?php esc_html_e( 'Make device local', 'dppp' ); ?>
                        </a>
                    </p>
                    <?php
                }
                ?>
            </div>
        </div>
        <div class="dppp-setting-row<?php if ( is_multisite() ) { echo ' dppp-display-none'; } ?>">
            <div class="dppp-setting-left">
                <h3><?php esc_html_e( 'Complete deletion', 'dppp' ); ?></h3>
                <p>
                    <?php esc_html_e( 'Remove all its data when this plugin is deleted via the dashboard.', 'dppp' ); ?>
                    <a id="dppp-hint-link-completely-remove" href="javascript:dpppShowHideHint('completely-remove')">
                        <?php esc_html_e( 'Read more', 'dppp' ); ?>
                    </a>
                    <i id="dppp-hint-completely-remove" class="dppp-hint-text">
                        <?php
                            esc_html_e( 'When this plugin is deleted (via the admin dashboard), all plugin rules, settings, and '
                                . 'additional files will also be deleted. This option is not available and not visible for multisites.', 'dppp' );
                        ?>
                    </i>
                </p>
            </div>
            <div class="dppp-setting-right">
                <?php
                dppp_setting_select(
                    'dppp-completely-remove',
                    Array( 'on', 'off' ),
                    Array(
                        __( 'Enable', 'dppp' ),
                        __( 'Disable', 'dppp' ),
                    )
                );
                ?>
            </div>
        </div>
    </div>
    <div class="dppp-inner-settings-section<?php echo esc_attr( $add_class_inner_section ); ?>">
        <h2>
            <span>
                <?php esc_html_e( 'Advanced Settings', 'dppp' ); ?>
            </span>
        </h2>
        <div class="dppp-setting-row">
            <?php esc_html_e( 'Most users will not need to change the advanced settings. They are only useful in more specific and rare situations.', 'dppp' ); ?>
        </div>
        <div class="dppp-setting-row">
            <div class="dppp-setting-left">
                <h3><?php esc_html_e( 'URI parameters for page rules', 'dppp' ); ?></h3>
                <p>
                    <?php esc_html_e( 'How to handle URI parameters when applying rules with Page URI selection type?', 'dppp' ); ?>
                    <a id="dppp-hint-link-uri-parameters" href="javascript:dpppShowHideHint('uri-parameters')">
                        <?php esc_html_e( 'Read more', 'dppp' ); ?>
                    </a>
                    <i id="dppp-hint-uri-parameters" class="dppp-hint-text">
                        <?php
                            esc_html_e( 'If the option to ignore the URI parameters (aka the query string) is chosen, when we apply the rules '
                                . 'with Page URI selection, we will use a version of the current server URI without parameters. So everything '
                                . 'starting from the question mark and until the end of the URI will be ignored. This means that even with tracking '
                                . 'parameters in the URL (for example ?utm_source=facebook&utm_medium=post) the page will be detected correctly. '
                                . 'If you are using a plain permalink structure, we will not ignore any parameters. If the option to ignore the '
                                . 'chosen URI parameters is chosen, only the parameters listed in the text box below will be ignored (to allow '
                                . 'other parameters potentially used by plugins to still be part of the page URI when applying rules). ', 'dppp' );
                        ?>
                    </i>
                </p>
            </div>
            <div class="dppp-setting-right">
                <?php
                dppp_setting_select(
                    'dppp-uri-parameters',
                    Array( 'ignore-all', 'ignore-chosen', 'do-not-ignore' ),
                    Array(
                        __( 'Ignore all URI parameters', 'dppp' ),
                        __( 'Ignore chosen URI parameters', 'dppp' ),
                        __( 'Do not ignore URI parameters', 'dppp' ),
                    )
                );
                ?>
            </div>
        </div>
        <div class="dppp-setting-row">
            <div class="dppp-setting-left">
                <h3><?php esc_html_e( 'URI parameters for custom rules', 'dppp' ); ?></h3>
                <p>
                    <?php esc_html_e( 'How to handle URI parameters when applying rules with Custom URI selection type?', 'dppp' ); ?>
                    <a id="dppp-hint-link-uri-parameters-custom" href="javascript:dpppShowHideHint('uri-parameters-custom')">
                        <?php esc_html_e( 'Read more', 'dppp' ); ?>
                    </a>
                    <i id="dppp-hint-uri-parameters-custom" class="dppp-hint-text">
                        <?php
                            esc_html_e( 'If the option to ignore all URI parameters (aka the query string) is chosen, when we apply the rules '
                                . 'with Custom URI selection, we will use a version of the current server URI without parameters. So everything '
                                . 'starting from the question mark and until the end of the URI will be ignored. If the option to ignore the chosen '
                                . 'URI parameters is chosen, only the parameters listed in the text box below will be ignored.', 'dppp' );
                        ?>
                    </i>
                </p>
            </div>
            <div class="dppp-setting-right">
                <?php
                dppp_setting_select(
                    'dppp-uri-parameters-custom',
                    Array( 'ignore-all', 'ignore-chosen', 'do-not-ignore' ),
                    Array(
                        __( 'Ignore all URI parameters', 'dppp' ),
                        __( 'Ignore chosen URI parameters', 'dppp' ),
                        __( 'Do not ignore URI parameters', 'dppp' ),
                    )
                );
                ?>
            </div>
        </div>
        <div class="dppp-setting-row">
            <div class="dppp-setting-left">
                <h3><?php esc_html_e( 'Chosen URI parameters', 'dppp' ); ?></h3>
                <p>
                    <?php esc_html_e( 'A comma-separated list of chosen URI parameters to ignore.', 'dppp' ); ?>
                    <a id="dppp-hint-link-chosen-uri-parameters" href="javascript:dpppShowHideHint('chosen-uri-parameters')">
                        <?php esc_html_e( 'Read more', 'dppp' ); ?>
                    </a>
                    <i id="dppp-hint-chosen-uri-parameters" class="dppp-hint-text">
                        <?php
                            esc_html_e( 'See the hint for the previous setting above.', 'dppp' );
                        ?>
                    </i>
                </p>
            </div>
            <div class="dppp-setting-right">
                <?php dppp_setting_textarea( 'dppp-chosen-uri-parameters' ); ?>
            </div>
        </div>
        <div class="dppp-setting-row">
            <div class="dppp-setting-left">
                <h3><?php esc_html_e( 'Trailing slash', 'dppp' ); ?></h3>
                <p>
                    <?php esc_html_e( 'How to handle the trailing slash of page URIs in rules with page URI selection?', 'dppp' ); ?>
                    <a id="dppp-hint-link-trailing-slash-pages" href="javascript:dpppShowHideHint('trailing-slash-pages')">
                        <?php esc_html_e( 'Read more', 'dppp' ); ?>
                    </a>
                    <i id="dppp-hint-trailing-slash-pages" class="dppp-hint-text">
                        <?php
                            esc_html_e( 'This setting affects the generated URI conditions of deactivation rules with Page URI selection '
                                . '(it changes the rules, not the actual URIs of pages). '
                                . 'If the first option is chosen, the selected URI equals the original URI extracted from the Page permalink. '
                                . 'If the second option is chosen, and the original URI does not have a trailing slash, we will add it. '
                                . 'If the third option is chosen, and the original URI has a trailing slash, we will remove it. '
                                . 'If the fourth option is chosen, two conditions will be added for each page, so both URIs with and '
                                . 'without trailing slash are selected. If the WordPress permalink structure is set to Plain, we will '
                                . 'always use the original URI. Also if the URI is only one slash, it will not be changed.', 'dppp' );
                        ?>
                    </i>
                </p>
            </div>
            <div class="dppp-setting-right">
                <?php
                dppp_setting_select(
                    'dppp-trailing-slash',
                    Array( 'original', 'add', 'remove', 'both' ),
                    Array(
                        __( 'Original URI from permalink', 'dppp' ),
                        __( 'If there is no trailing slash, add one', 'dppp' ),
                        __( 'If there is a trailing slash, remove it', 'dppp' ),
                        __( 'Use both versions', 'dppp' )
                    )
                );
                ?>
            </div>
        </div>
        <div class="dppp-setting-row">
            <div class="dppp-setting-left">
                <h3><?php esc_html_e( 'Visual editors deactivation', 'dppp' ); ?></h3>
                <p>
                    <?php esc_html_e( 'Affect also the front-end visual editor page of Elementor, Divi, and WPBakery Page Builder.', 'dppp' ); ?>
                    <a id="dppp-hint-link-affect-visual-editors" href="javascript:dpppShowHideHint('affect-visual-editors')">
                        <?php esc_html_e( 'Read more', 'dppp' ); ?>
                    </a>
                    <i id="dppp-hint-affect-visual-editors" class="dppp-hint-text">
                        <?php
                            esc_html_e( 'If this is enabled, the page where you visually edit pages with the mentioned plugins, '
                                . 'will also be possible to be affected by the deactivation rules. Keep in mind that some visual editors '
                                . 'are loaded in an iframe with a URL address not visible in the browser. To correctly target them '
                                . 'with a deactivation rule, you need to see the URL in the source code iframe. For example to target '
                                . 'the visual editor frame of Elementor you can use '
                                . 'the phrase elementor-preview in a rule that checks if it is contained in the URI.', 'dppp' );
                        ?>
                    </i>
                </p>
            </div>
            <div class="dppp-setting-right">
                <?php
                dppp_setting_select(
                    'dppp-affect-visual-editors',
                    Array( 'on', 'off' ),
                    Array(
                        __( 'Enable', 'dppp' ),
                        __( 'Disable', 'dppp' ),
                    )
                );
                ?>
            </div>
        </div>
        <div class="dppp-setting-row">
            <div class="dppp-setting-left">
                <h3><?php esc_html_e( 'Edit post parameter', 'dppp' ); ?></h3>
                <p>
                    <?php esc_html_e( 'Adds the dppp_post_type parameter to the edit post links so back-end '
                        . 'pages like this post.php?post=XXXX&action=edit can be targeted by rules per post type.', 'dppp' ); ?>
                    <a id="dppp-hint-link-edit-post-parameter" href="javascript:dpppShowHideHint('edit-post-parameter')">
                        <?php esc_html_e( 'Read more', 'dppp' ); ?>
                    </a>
                    <i id="dppp-hint-edit-post-parameter" class="dppp-hint-text">
                        <?php
                            esc_html_e( 'If this is enabled, the links which point to editing a post or a page, '
                                . 'will receive an extra URL parameter dppp_post_type with a value equal to the post type. '
                                . 'This parameter will not change anything but you can use it to better target deactivation rules '
                                . 'for these back-end pages based on post type.', 'dppp' );
                        ?>
                    </i>
                </p>
            </div>
            <div class="dppp-setting-right">
                <?php
                dppp_setting_select(
                    'dppp-edit-post-parameter',
                    Array( 'on', 'off' ),
                    Array(
                        __( 'Enable', 'dppp' ),
                        __( 'Disable', 'dppp' ),
                    )
                );
                ?>
            </div>
        </div>
        <div class="dppp-setting-row">
            <div class="dppp-setting-left">
                <h3><?php esc_html_e( 'Admin menu cache time (if enabled)', 'dppp' ); ?></h3>
                <p>
                    <?php esc_html_e( 'It is only used if back-end deactivation with admin menu cache is enabled. Default value: 30 minutes.', 'dppp' ); ?>
                    <a id="dppp-hint-link-admin-menu-cache-time" href="javascript:dpppShowHideHint('admin-menu-cache-time')">
                        <?php esc_html_e( 'Read more', 'dppp' ); ?>
                    </a>
                    <i id="dppp-hint-admin-menu-cache-time" class="dppp-hint-text">
                        <?php
                            esc_html_e( 'If the admin menu cache is enabled, after this time we will rebuild the cache. '
                                . 'But it will also be rebuilt even before that if certain actions trigger it, like activating a plugin. '
                                . 'Usually you will not need to change this setting.', 'dppp' );
                        ?>
                    </i>
                </p>
            </div>
            <div class="dppp-setting-right">
                <?php
                dppp_setting_select(
                    'dppp-admin-menu-cache-time',
                    Array( 5*60, 10*60, 30*60, 2*60*60, 8*60*60, 24*60*60, 48*60*60, 7*24*60*60, 30*24*60*60, 365*24*60*60, 5*365*24*60*60, 10*365*24*60*60 ),
                    Array(
                        __( '5 minutes', 'dppp' ),
                        __( '10 minutes', 'dppp' ),
                        __( '30 minutes', 'dppp' ),
                        __( '2 hours', 'dppp' ),
                        __( '8 hours', 'dppp' ),
                        __( '24 hours', 'dppp' ),
                        __( '48 hours', 'dppp' ),
                        __( '7 days', 'dppp' ),
                        __( '30 days', 'dppp' ),
                        __( '365 days', 'dppp' ),
                        __( '5 years', 'dppp' ),
                        __( '10 years', 'dppp' ),
                    )
                );
                ?>
            </div>
        </div>
        <div class="dppp-setting-row">
            <div class="dppp-setting-left">
                <h3><?php esc_html_e( 'Priority', 'dppp' ); ?></h3>
                <p>
                    <?php esc_html_e( 'Changing this value can solve some rare plugin conflicts. Default value: 10.', 'dppp' ); ?>
                </p>
            </div>
            <div class="dppp-setting-right">
                <?php
                dppp_setting_select(
                    'dppp-priority',
                    Array( 0, 1, 5, 10, 20, 99, 999999999 ),
                    'same-as-values'
                );
                ?>
            </div>
        </div>
    </div>
    <div class="dppp-inner-settings-section<?php echo esc_attr( $add_class_inner_section ); ?>">
        <p>
            <input id="dppp-save-settings" onclick="dpppSaveSettings()" name="dppp-save-settings" class="button button-primary" type="button"
                value="<?php esc_attr_e( 'Save Settings', 'dppp' ); ?>" />
            <span id="dppp-status-save"></span>
            <span id="dppp-loading-settings" class="dashicons dashicons-update"></span>
        </p>
    </div>
    <?php
}

/**
 * Returns an url of an image in the images folder based on a given file name
 * @param string $filename
 * @return string
 */
function dppp_image_url( $filename ) {
    return dppp_plugin_dir_url() . 'images/' . $filename;
}

// Returns the url of the plugin folder. Do not move this function to a file in another folder.
function dppp_plugin_dir_url() {
    return plugin_dir_url( __FILE__ );
}

// Returns the path of the plugin folder. Do not move this function to a file in another folder.
function dppp_plugin_dir_path() {
    return plugin_dir_path( __FILE__ );
}

/**
 * Shortens a long string to a chosen length and adds three dots at the end
 * @param string $string
 * @param int $length
 * @return string
 */
function dppp_truncate_string( $string, $length ) {
    if ( mb_strlen( $string ) > $length ) {
        $string = mb_substr( $string, 0, $length ) . '...';
    }
    return $string;
}

/**
 * Outputs the HTML code for displaying the whole row and rules data for a plugin on the plugin settings page
 * @param int $loop_count
 * @param string $plugin_file
 * @param string $add_class_section
 * @param array $plugin_data
 */
function dppp_output_plugin_row( $loop_count, $plugin_file, $plugin_data, $add_class_section ) {
    if ( ! dppp_is_valid_plugin_file( $plugin_file ) ) {
        echo '<div class="dppp-inner-box' . esc_attr( $add_class_section ) . '">';
        echo esc_html__( 'Error: Invalid plugin file name', 'dppp' );
        echo '</div>';
    } else {
        $plugin_hash = md5( $plugin_file );
        $results_multi_array = dppp_get_results_plugin_rules( $plugin_file );
        $add_class = '';
        if ( empty( $results_multi_array ) ) {
            $add_class = ' dppp-display-none';
        }

        $plugin_global_status = dppp_get_plugin_or_group_global_status( $plugin_file );
        if ( 'deactivated' === $plugin_global_status ) {
            $add_class_section .= ' dppp-front-end-deactivated dppp-back-end-deactivated';
        } elseif ( 'front-end-deactivated' === $plugin_global_status ) {
            $add_class_section .= ' dppp-front-end-deactivated';
        } elseif ( 'back-end-deactivated' === $plugin_global_status ) {
            $add_class_section .= ' dppp-back-end-deactivated';
        }

        echo '<div id="dppp-inner-box-' . esc_attr( $plugin_hash ) . '" class="dppp-inner-box' . esc_attr( $add_class_section ) . '">';
        echo '<div class="dppp-section-row">';

        if ( dppp_get_option( 'dppp-plugin-icons' ) == 'on' ) {
            $plugin_icon_option = get_option( 'dppp-plugin-icon-' . $plugin_hash );
            if ( $plugin_icon_option !== false ) {
                if ( dppp_is_valid_url( $plugin_icon_option ) && dppp_get_host_from_url( $plugin_icon_option ) === 'ps.w.org' ) {
                    echo '<span id="dppp-plugin-icon-' . esc_attr( $plugin_hash ) . '" class="dppp-plugin-icon" data-plugin-slug="'
                        . esc_attr( dppp_get_slug_from_plugin_file( $plugin_file ) ) . '" data-plugin-file="' . esc_attr( $plugin_file )
                        . '"><img src="' . esc_url( $plugin_icon_option ) . '" onerror="this.classList.add(\'dppp-error-img\')" /></span>';
                } else  {
                    echo '<span class="dppp-plugin-icon"><span class="dashicons dashicons-admin-plugins"></span></span>';
                }
            } else {
                echo '<span id="dppp-plugin-icon-' . esc_attr( $plugin_hash ) . '" class="dppp-plugin-icon dppp-not-set" data-plugin-slug="'
                    . esc_attr( dppp_get_slug_from_plugin_file( $plugin_file ) ) . '" data-plugin-file="'
                    . esc_attr( $plugin_file ) . '"><span class="dashicons dashicons-admin-plugins"></span></span>';
            }
            $plugin_icon_class = 'dppp-icons-on';
        } else {
            $plugin_icon_class = '';
        }

        echo '<span class="dppp-row-name ' . esc_attr( $plugin_icon_class ) . '">' . esc_html( $loop_count ) . '. <span class="dppp-row-name-only">'
            . esc_html( dppp_truncate_string( $plugin_data['Name'], 40 ) ) . '</span></span>';
        echo ' <a id="dppp-add-rule-link-' . esc_attr( $plugin_hash )
            . '" class="dppp-add-rule-link dppp-hide-on-copy-rule dppp-one-line" href="javascript:dpppAddNewRule( \''
            . esc_attr( esc_js( $plugin_file ) ) . '\', \''
            . esc_attr( esc_js( $plugin_hash ) ) . '\' )"><span class="dashicons dashicons-plus"></span><span class="dppp-add-new-rule">'
            . esc_html__( 'Add New Rule', 'dppp' ) . '</span></a>';
        echo ' <a id="dppp-paste-rules-link-' . esc_attr( $plugin_hash )
            . '" class="dppp-paste-rules-link dppp-show-on-copy-rule dppp-hide-while-pasting-rule dppp-display-none dppp-one-line" '
            . 'href="javascript:dpppPasteRules( \'' . esc_attr( esc_js( $plugin_file ) ) . '\', \'' . esc_attr( esc_js( $plugin_hash ) )
            . '\' )"><span class="dashicons dashicons-clipboard"></span><span class="dppp-paste-rules">'
            . '' . esc_html__( 'Paste 1 rule', 'dppp' ) . '</span></a>';
        echo ' <a id="dppp-cancel-paste-rules-link-' . esc_attr( $plugin_hash )
            . '" class="dppp-cancel-paste-rules-link dppp-show-on-copy-rule dppp-hide-while-pasting-rule dppp-display-none dppp-one-line" '
            . 'href="javascript:dpppCancelPasteRules()"><span class="dashicons dashicons-no"></span><span class="dppp-cancel-paste-rules">'
            . esc_html__( 'Cancel', 'dppp' ) . '</span></a>';
        echo '<span class="dppp-plugin-or-group-action-links ' . esc_attr( $plugin_icon_class ) . ' dppp-hide-on-copy-rule dppp-one-line">';
        echo ' <a title="' . esc_attr__( 'Front-end Deactivate', 'dppp' ) . '" id="dppp-plugin-or-group-front-end-deactivate-link-' . esc_attr( $plugin_hash )
            . '" class="dppp-plugin-or-group-front-end-deactivate-link" href="javascript:dpppPluginAction( \'front-end-deactivate\', \''
            . esc_attr( esc_js( $plugin_file ) ) . '\', \''
            . esc_attr( esc_js( $plugin_hash ) ) . '\' )"><span class="dashicons dashicons-welcome-view-site"></span>'
            . '<span class="dppp-plugin-action-text">' . sprintf( esc_html__( 'Front-end%sDeactivate', 'dppp' ), '<br>' ) . '</span></a>';
        echo ' <a title="' . esc_attr__( 'Front-end Activate', 'dppp' ) . '" id="dppp-plugin-or-group-front-end-activate-link-' . esc_attr( $plugin_hash )
            . '" class="dppp-plugin-or-group-front-end-activate-link" href="javascript:dpppPluginAction( \'front-end-activate\', \''
            . esc_attr( esc_js( $plugin_file ) ) . '\', \''
            . esc_attr( esc_js( $plugin_hash ) ) . '\' )"><span class="dashicons dashicons-welcome-view-site"></span>'
            . '<span class="dppp-plugin-action-text">' . sprintf( esc_html__( 'Front-end%sActivate', 'dppp' ), '<br>' ) . '</span></a>';
        echo ' <a title="' . esc_attr__( 'Back-end Deactivate', 'dppp' ) . '" id="dppp-plugin-or-group-back-end-deactivate-link-' . esc_attr( $plugin_hash )
            . '" class="dppp-plugin-or-group-back-end-deactivate-link" href="javascript:dpppPluginAction( \'back-end-deactivate\', \''
            . esc_attr( esc_js( $plugin_file ) ) . '\', \''
            . esc_attr( esc_js( $plugin_hash ) ) . '\' )"><span class="dashicons dashicons-welcome-widgets-menus"></span>'
            . '<span class="dppp-plugin-action-text">' . sprintf( esc_html__( 'Back-end%sDeactivate', 'dppp' ), '<br>' ) . '</span></a>';
        echo ' <a title="' . esc_attr__( 'Back-end Activate', 'dppp' ) . '" id="dppp-plugin-or-group-back-end-activate-link-' . esc_attr( $plugin_hash )
            . '" class="dppp-plugin-or-group-back-end-activate-link" href="javascript:dpppPluginAction( \'back-end-activate\', \''
            . esc_attr( esc_js( $plugin_file ) ) . '\', \''
            . esc_attr( esc_js( $plugin_hash ) ) . '\' )"><span class="dashicons dashicons-welcome-widgets-menus"></span>'
            . '<span class="dppp-plugin-action-text">' . sprintf( esc_html__( 'Back-end%sActivate', 'dppp' ), '<br>' ) . '</span></a>';
        echo ' <a title="' . esc_attr__( 'Start Plugin Rules', 'dppp' ) . '" id="dppp-plugin-or-group-start-rules-link-' . esc_attr( $plugin_hash )
            . '" class="dppp-plugin-or-group-start-rules-link' . esc_attr( $add_class ) . '" href="javascript:dpppPluginAction( \'start-rules\', \''
            . esc_attr( esc_js( $plugin_file ) ) . '\', \''
            . esc_attr( esc_js( $plugin_hash ) ) . '\' )"><span class="dashicons dashicons dashicons-controls-play"></span></a>';
        echo ' <a title="' . esc_attr__( 'Pause Plugin Rules', 'dppp' ) . '" id="dppp-plugin-or-group-pause-rules-link-' . esc_attr( $plugin_hash )
            . '" class="dppp-plugin-or-group-pause-rules-link' . esc_attr( $add_class ) . '" href="javascript:dpppPluginAction( \'pause-rules\', \''
            . esc_attr( esc_js( $plugin_file ) ) . '\', \''
            . esc_attr( esc_js( $plugin_hash ) ) . '\' )"><span class="dashicons dashicons dashicons-controls-pause"></span></a>';
        echo ' <a title="' . esc_attr__( 'Delete Plugin Rules', 'dppp' ) . '" id="dppp-plugin-or-group-delete-rules-link-' . esc_attr( $plugin_hash )
            . '" class="dppp-plugin-or-group-delete-rules-link' . esc_attr( $add_class ) . '" href="javascript:dpppPluginAction( \'delete-rules\', \''
            . esc_attr( esc_js( $plugin_file ) ) . '\', \''
            . esc_attr( esc_js( $plugin_hash ) ) . '\' )"><span class="dashicons dashicons dashicons-no"></span></a>';
        echo '</span>';
        echo '</div>';
        echo '<div class="dppp-plugin-rules-layer' . esc_attr( $add_class ) . '" id="dppp-plugin-rules-' . esc_attr( $plugin_hash ) . '">';
        dppp_output_plugin_rules_content( $plugin_file, md5( $plugin_file ), $results_multi_array );
        echo '</div>';
        echo '</div>';
    }
}

/**
 * Outputs the HTML code for displaying the table with rules for a plugin on the plugin settings page
 * @param string $plugin_file
 * @param array $results_multi_array
 */
function dppp_output_plugin_rules_content( $plugin_file, $plugin_or_group_hash, $results_multi_array ) {
    if ( ! empty( $results_multi_array ) ) {
        echo '<table class="dppp-table dppp-full-width">';
        echo '<thead>'
            . '<tr>'
            . '<th>' . esc_html__( 'Rule ID', 'dppp' ) . '</th>'
            . '<th>' . esc_html__( 'Deactivation Type', 'dppp' ) . '</th>'
            . '<th>' . esc_html__( 'URI Type', 'dppp' ) . '</th>'
            . '<th>' . esc_html__( 'URI Conditions', 'dppp' ) . '</th>'
            . '<th>' . esc_html__( 'Devices', 'dppp' ) . '</th>'
            . '<th>' . esc_html__( 'Actions', 'dppp' ) . '</th>'
            . '</tr>'
            . '</thead>';
        echo '<tbody>';
        foreach ( $results_multi_array as $results ) {
            echo '<tr id="dppp-rule-row-' . esc_attr( $results['rule_id'] ) . '" class="dppp-rule-row dppp-rule-row-' . esc_attr( $results['status'] ) . '">';
            echo '<td data-title="' . esc_attr__( 'Rule ID', 'dppp' ) . '" class="dppp-rule-id">' . esc_html( $results['rule_id'] ) . '</td>';
            echo '<td data-title="' . esc_attr__( 'Deactivation Type', 'dppp' ) . '" class="dppp-deactivation-type">';

            if ( 'selected' === $results['deactivation_type'] ) {
                echo '<span class="dppp-deactivation-type-icons"><span title="' . esc_attr__( 'selected URI', 'dppp' )
                    . '" class="dashicons dashicons-yes"></span> ' . esc_html__( 'selected URI', 'dppp' ) . '</span>';
            } elseif ( 'except' === $results['deactivation_type'] ) {
                echo '<span class="dppp-deactivation-type-icons"><span title="' . esc_attr__( 'all except selected URI', 'dppp' )
                    . '" class="dashicons dashicons-yes-alt"></span> ' . esc_html__( 'all except selected URI', 'dppp' ) . '</span>';
            } else {
                echo esc_html__( 'Error', 'dppp' );
            }

            echo '</td>';
            echo '<td data-title="' . esc_attr__( 'URI Type', 'dppp' ) . '">';

            if ( 'page' === $results['uri_type'] ) {
                echo '<span class="dppp-uri-type-icons dppp-one-line"><span title="' . esc_attr__( 'Page', 'dppp' )
                    . '" class="dashicons dashicons-media-default"></span> ';
                if ( strpos( $results['page_ids'], ',' ) !== false ) {
                    echo esc_html__( 'Pages', 'dppp' );
                } else {
                    echo esc_html( sprintf( __( 'Page (ID %d)', 'dppp' ), intval( $results['page_ids'] ) ) );
                }
                echo '</span>';
            } elseif ( 'custom' === $results['uri_type'] ) {
                echo '<span class="dppp-uri-type-icons dppp-one-line"><span title="' . esc_attr__( 'Custom', 'dppp' )
                    . '" class="dashicons dashicons-admin-settings"></span> ' . esc_html__( 'Custom', 'dppp' ) . '</span>';
            } else {
                echo esc_html__( 'Error', 'dppp' );
            }

            echo '</td>';

            echo '<td data-title="' . esc_attr__( 'URI Conditions', 'dppp' ) . '" class="dppp-break-words dppp-uri-condition-and-value">';

            dppp_output_uri_conditions_and_values( $results['uri_type'], $results['uri_condition'], $results['uri_value'] );

            echo '</td>';
            echo '<td data-title="' . esc_attr__( 'Devices', 'dppp' ) . '">';

            if ( 'all' === $results['devices'] ) {
                echo '<span class="dppp-device-icons dppp-one-line"><span title="'
                    . esc_attr__( 'Affect desktop devices', 'dppp' ) . '" class="dashicons dashicons-desktop"></span>'
                    . '<span title="' . esc_attr__( 'Affect mobile devices', 'dppp' ) . '" class="dashicons dashicons-smartphone"></span></span>';
            } elseif ( 'desktop' === $results['devices'] ) {
                echo '<span class="dppp-device-icons dppp-one-line"><span title="'
                    . esc_attr__( 'Affect desktop devices', 'dppp' ) . '" class="dashicons dashicons-desktop"></span>'
                    . '<span title="' . esc_attr__( 'Do not affect mobile devices', 'dppp' )
                    . '" class="dashicons dashicons-smartphone dppp-barely-visible"></span></span>';
            } elseif ( 'mobile' === $results['devices'] ) {
                echo '<span class="dppp-device-icons dppp-one-line"><span title="' . esc_attr__( 'Do not affect desktop devices', 'dppp' ) . '" '
                    . 'class="dashicons dashicons-desktop dppp-barely-visible"></span>'
                    . '<span title="' . esc_attr__( 'Affect mobile devices', 'dppp' ) . '" class="dashicons dashicons-smartphone"></span></span>';
            } else {
                echo esc_html__( 'Error', 'dppp' );
            }

            echo '</td>';
            echo '<td data-title="' . esc_attr__( 'Actions', 'dppp' ) . '"><span class="dppp-one-line">';

            if ( 'paused' === $results['status'] ) {
                echo '<a class="dppp-dashicon-action-link dppp-start-rule dppp-hide-on-copy-rule" id="dppp-start-rule-'
                    . esc_attr( $results['rule_id'] )
                    . '" href="javascript:dpppStartRule( ' . esc_attr( esc_js( $results['rule_id'] ) ) . ', \''
                    . esc_attr( esc_js( $plugin_file ) ) . '\', \'' . esc_attr( esc_js( $plugin_or_group_hash ) ) . '\''
                    . ' )" title="' . esc_attr__( 'Start Rule', 'dppp' ) . '"><span class="dashicons dashicons-controls-play"></span></a>';
            } elseif ( 'started' === $results['status'] ) {
                echo '<a class="dppp-dashicon-action-link dppp-pause-rule dppp-hide-on-copy-rule" id="dppp-pause-rule-'
                    . esc_attr( $results['rule_id'] )
                    . '" href="javascript:dpppPauseRule( ' . esc_attr( esc_js( $results['rule_id'] ) ) . ', \''
                    . esc_attr( esc_js( $plugin_file ) ) . '\', \'' . esc_attr( esc_js( $plugin_or_group_hash ) ) . '\''
                    . ' )" title="' . esc_attr__( 'Pause Rule', 'dppp' ) . '"><span class="dashicons dashicons-controls-pause"></span></a>';
            } else {
                echo esc_html__( 'Error', 'dppp' );
            }

            echo ' <a class="dppp-dashicon-action-link dppp-hide-on-copy-rule dppp-explain-rule" id="dppp-explain-rule-'
                    . esc_attr( $results['rule_id'] )
                    . '" href="javascript:dpppExplainRule( ' . esc_attr( esc_js( $results['rule_id'] ) )
                    . ' )" title="' . esc_attr__( 'Explain Rule', 'dppp' ) . '"><span class="dashicons dashicons-info"></span></a>'
                . ' <a class="dppp-dashicon-action-link dppp-edit-rule dppp-hide-on-copy-rule" id="dppp-edit-rule-'
                    . esc_attr( $results['rule_id'] )
                    . '" href="javascript:dpppEditRule( ' . esc_attr( esc_js( $results['rule_id'] ) ) . ', \''
                    . esc_attr( esc_js( $results['uri_type'] ) ) . '\', \''
                    . esc_attr( esc_js( $plugin_file ) ) . '\', \'' . esc_attr( esc_js( $plugin_or_group_hash ) ) . '\' )" title="'
                    . esc_attr__( 'Edit Rule', 'dppp' ) . '"><span class="dashicons dashicons-edit"></span></a>'
                . ' <a class="dppp-dashicon-action-link dppp-copy-rule dppp-hide-while-pasting-rule" id="dppp-copy-rule-'
                    . esc_attr( $results['rule_id'] ) . '" href="javascript:dpppCopyRule( ' . esc_attr( esc_js( $results['rule_id'] ) ) . ' )" title="'
                    . esc_attr__( 'Copy Rule', 'dppp' ) . '"><span class="dashicons dashicons-admin-page"></span></a>'
                . '<span id="dppp-copied-rule-' . esc_attr( $results['rule_id'] ) . '" class="dppp-copied-rule dppp-display-none">'
                    . esc_html__( 'Copied', 'dppp' ) . '</span>'
                . ' <span id="dppp-contain-undo-copied-rule-' . esc_attr( $results['rule_id'] ) . '" class="dppp-contain-undo-copied-rule dppp-display-none">'
                    . '(<a class="dppp-undo-copy-rule" href="javascript:dpppUndoCopyRule( ' . esc_attr( esc_js( $results['rule_id'] ) ) . ' )">'
                    . esc_html__( 'Undo', 'dppp' ) . '</a>)</span>'
                . ' <a class="dppp-dashicon-action-link dppp-delete-rule dppp-hide-on-copy-rule" id="dppp-delete-rule-'
                    . esc_attr( $results['rule_id'] )
                    . '" href="javascript:dpppDeleteRule( ' . esc_attr( esc_js( $results['rule_id'] ) ) . ', \''
                    . esc_attr( esc_js( $plugin_file ) ) . '\', \'' . esc_attr( esc_js( $plugin_or_group_hash ) ) . '\''
                . ' )" title="' . esc_attr__( 'Delete Rule', 'dppp' ) . '"><span class="dashicons dashicons-no"></span></a>'
                . '</span></td>';
            echo '</tr>';
        }
        echo '</table>';
    }
}

/**
 * Returns the translated condition text based on the database value.
 * @param string $database_value
 * @return string
 */
function dppp_get_rule_condition_text( $database_value ) {
    if ( 'equals' === $database_value ) {
        return __( 'equals: ', 'dppp' );
    } elseif ( 'does-not-equal' === $database_value ) {
        return __( 'does not equal: ', 'dppp' );
    } elseif ( 'contains' === $database_value ) {
        return __( 'contains: ', 'dppp' );
    } elseif ( 'does-not-contain' === $database_value ) {
        return __( 'does not contain: ', 'dppp' );
    } elseif ( 'starts-with' === $database_value ) {
        return __( 'starts with: ', 'dppp' );
    } elseif ( 'does-not-start-with' === $database_value ) {
        return __( 'does not start with: ', 'dppp' );
    } elseif ( 'ends-with' === $database_value ) {
        return __( 'ends with: ', 'dppp' );
    } elseif ( 'does-not-end-with' === $database_value ) {
        return __( 'does not end with: ', 'dppp' );
    } elseif ( 'regular-expression' === $database_value ) {
        return __( 'matches the regular expression: ', 'dppp' );
    } else {
        return __( 'Error', 'dppp' );
    }
}

/**
 * Checks if the provided variable contains only comma separated numbers
 * @param string $string
 * @return int
 */
function dppp_is_comma_separated_numbers( $string ) {
    return preg_match( '/^\d+(?:,\d+)*$/', $string );
}

/**
 * Returns only the original URI of a page by ID, regadless of settings.
 * @param int $page_id
 * @return string
 */
function dppp_get_original_page_uri( $page_id ) {

    if ( ! function_exists( 'is_plugin_active' ) ) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }

    // If the WPML plugin is active, we switch to the language of the current page, so we can get the correct URI
    if ( is_plugin_active( 'sitepress-multilingual-cms/sitepress.php' ) ) {
        global $sitepress;
        $page_language_details = apply_filters( 'wpml_post_language_details', NULL, $page_id );
        if ( is_array( $page_language_details ) && array_key_exists( 'language_code', $page_language_details ) ) {
            $sitepress->switch_lang( $page_language_details['language_code'] );
        }
    }

    $page_permalink = get_permalink( $page_id );
    $url_parts = parse_url( $page_permalink );
    $page_uri = '';
    if ( is_array( $url_parts ) ) {
        if ( array_key_exists( 'path', $url_parts ) ) {
            $page_uri .= $url_parts['path'];
        }
        if ( array_key_exists( 'query', $url_parts ) ) {
            $page_uri .= '?' . $url_parts['query'];
        }
    }

    $page_uri = rawurldecode( $page_uri );

    return $page_uri;
}

/**
 * Returns the URI of a page by ID. It could also return two URIs with a separator based on the settings.
 * @param int $page_id
 * @return string
 */
function dppp_get_page_uri_or_uris( $page_id ) {

    $page_uri = dppp_get_original_page_uri( $page_id );
    $trailing_slash = dppp_get_option( 'dppp-trailing-slash' );
    $permalink_structure = get_option( 'permalink_structure' );

    // If the permalink structure is not Plain, and the URI is not the home page, we could add, remove the trailing slash, or keep it unchanged
    if ( $permalink_structure !== '' && $page_uri !== '/' ) {
        if ( 'add' === $trailing_slash && substr( $page_uri, -1 ) !== '/' ) {
            $page_uri .= '/';
        } elseif ( 'remove' === $trailing_slash && substr( $page_uri, -1 ) === '/' ) {
            $page_uri = substr_replace( $page_uri, "", -1 );
        } elseif ( 'both' === $trailing_slash ) {
            if ( substr( $page_uri, -1 ) === '/' ) {
                $page_uri_with_slash = $page_uri;
                $page_uri_without_slash = substr_replace( $page_uri, "", -1 );
            } else {
                $page_uri_with_slash = $page_uri . '/';
                $page_uri_without_slash = $page_uri;
            }
            $page_uri = $page_uri_with_slash . '[dppp-or]' . $page_uri_without_slash;
        }
    }

    return $page_uri;
}

// Updates the URI values of all the page type rules to the current values
function dppp_refresh_page_uri_values() {
    global $wpdb;
    $results_multi_array = $wpdb->get_results( "SELECT rule_id, uri_value, page_ids FROM {$wpdb->prefix}dppp_plugin_deactivation_rules "
        . "WHERE uri_type = 'page'", ARRAY_A );
    if ( ! empty( $results_multi_array ) ) {
        foreach ( $results_multi_array as $results ) {
            $new_page_ids = $results['page_ids'];

            // Multiple page ids
            if ( strpos( $results['page_ids'], ',' ) !== false ) {
                $remove_page_ids = array();
                $page_ids = explode( ',', $results['page_ids'] );
                $new_uri = '';
                $loop = 1;
                foreach ( $page_ids as $page_id ) {
                    if ( get_post_status( $page_id ) !== false ) {
                        $page_uri = dppp_escape_uri( dppp_get_page_uri_or_uris( $page_id ) );
                        $new_uri .= $page_uri;
                        if ( $loop !== count( $page_ids ) ) {
                            $new_uri .= '[dppp-or]';
                        }
                    } else {

                        // A page was deleted it seems, so we need to later update the page ids by removing these ones
                        $remove_page_ids[] = $page_id;
                    }
                    $loop++;
                }
                $new_page_ids_array = array_diff( $page_ids, $remove_page_ids );

                if ( ! empty( $new_page_ids_array ) ) {
                    $new_page_ids = implode( ',', $new_page_ids_array );
                    if ( substr( $new_uri, -9 ) === '[dppp-or]' ) {
                        $new_uri = substr_replace( $new_uri, "", -9 );
                    }

                    // If the old uri value is different from the new, we update the rule
                    if ( $results['uri_value'] !== $new_uri ) {
                        $wpdb->update(
                            "{$wpdb->prefix}dppp_plugin_deactivation_rules",
                            array(
                                'uri_value' => $new_uri,
                                'page_ids' => $new_page_ids,
                            ),
                            array(
                                'rule_id' => intval( $results['rule_id'] ),
                            ),
                            array( '%s', '%s' ),
                            array( '%d' )
                        );
                    }

                // If all of the pages were deleted we delete the rule
                } else {
                    $wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE rule_id = %d",
                        intval( $results['rule_id'] ) ) );
                }

            // Single page id
            } else {
                if ( get_post_status ( intval( $results['page_ids'] ) ) !== false ) {
                    $new_uri = dppp_escape_uri( dppp_get_page_uri_or_uris( intval( $results['page_ids'] ) ) );

                    // If the old uri value is different from the new, we update the rule
                    if ( $results['uri_value'] !== $new_uri ) {
                        $wpdb->update(
                            "{$wpdb->prefix}dppp_plugin_deactivation_rules",
                            array(
                                'uri_value' => $new_uri,
                                'page_ids' => $new_page_ids,
                            ),
                            array(
                                'rule_id' => intval( $results['rule_id'] ),
                            ),
                            array( '%s', '%s' ),
                            array( '%d' )
                        );
                    }

                // If the page does not exist we delete the rule
                } else {
                    $wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE rule_id = %d",
                        intval( $results['rule_id'] ) ) );
                }
            }
        }
    }
}

/**
 * Checks if the string is a valid plugin file identifier (folder name plus file name or only file name)
 * @param string $string
 * @return bool
 */
function dppp_is_valid_plugin_file( $string ) {
    if ( substr_count( $string, '/' ) > 1 || substr( $string, -4, 4 ) !== '.php' ) {
        return false;
    }
    $string_no_separator = str_replace( '/', '', $string );
    $specials = array( '?', '[', ']', '/', '\\', '=', '<', '>', ':', ';', ',', "'", '"', '&', '$', '#', '*', '(', ')', '|', '~', '`', '!',
        '{', '}', '%', '+', chr( 0 ) );
    foreach ( $specials as $special ) {
        if ( strpos( $string_no_separator, $special ) !== false ) {
            return false;
        }
    }
    return true;
}

/**
 * Sanitizes a string by removing invalid characters for a plugin file identifier (folder name plus file name or only file name)
 * @param string $string
 * @return string
 */
function dppp_sanitize_plugin_file( $string ) {
    $string_no_separator = str_replace( '/', 'dppp-separator', $string );
    $specials = array( '?', '[', ']', '/', '\\', '=', '<', '>', ':', ';', ',', "'", '"', '&', '$', '#', '*', '(', ')', '|', '~', '`', '!',
        '{', '}', '%', '+', chr( 0 ) );
    foreach ( $specials as $special ) {
        $string_no_separator = str_replace( $special, '', $string_no_separator );
    }
    $string = str_replace( 'dppp-separator', '/', $string_no_separator );
    return $string;
}

/**
 * Escapes a URI string for adding in the database, but allows my delimiter [dppp-or]
 * @param string $uri
 * @return string
 */
function dppp_escape_uri( $uri ) {
    $uri_without_delimiters = str_replace( '[dppp-or]', 'nf75dxog67ax3nba99xq', $uri );
    $example_url = esc_url_raw( 'http://example.com/a' . $uri_without_delimiters );
    $example_url_with_delimiters = str_replace( 'nf75dxog67ax3nba99xq', '[dppp-or]', $example_url );
    return str_replace( 'http://example.com/a', '', $example_url_with_delimiters );
}

/**
 * Sanitizes a comma-separated string of numbers.
 * @param string $string
 * @return string
 */
function dppp_sanitize_comma_separated_numbers( $string ) {
    return preg_replace( '/[^0-9, ]/', '', $string );
}

/**
 * Sanitizes a comma-separated string of slugs.
 * @param string $string
 * @return string
 */
function dppp_sanitize_comma_separated_slugs( $string ) {
    return preg_replace( '/[^0-9,a-z_\- ]/', '', $string );
}

/**
 * Returns a string with the plugins files separated by ;; based on an array with plugin files
 * @param array $sorted_plugin_files
 * @return string
 */
function dppp_get_plugin_files_list_from_array( $sorted_plugin_files ) {
    $plugin_files_string = '';
    foreach ( $sorted_plugin_files as $group_plugin_file ) {
        $plugin_files_string .= $group_plugin_file . ';;';
    }
    return substr_replace( $plugin_files_string, "", -2 );
}

/**
 * Returns the group hash based on the sorted array of plugin files
 * @param array $sorted_plugin_files
 * @return string
 */
function dppp_get_plugin_group_hash_from_array( $sorted_plugin_files ) {
    $plugin_hashes_string = '';
    foreach ( $sorted_plugin_files as $plugin_file ) {
        $plugin_hashes_string .= md5( $plugin_file );
    }
    return md5( $plugin_hashes_string );
}

/**
 * Outputs the HTML code for displaying the whole row and rules data for a plugin group on the plugin settings page
 * @param int $loop_count
 * @param array $plugin_group
 * @param string $add_class_section
 */
function dppp_output_plugin_group_row( $loop_count, $plugin_group, $add_class_section ) {
    $group_hash = dppp_get_plugin_group_hash_from_array( $plugin_group['plugins'] );
    $plugin_files_string = dppp_get_plugin_files_list_from_array( $plugin_group['plugins'] );
    $results_multi_array = dppp_get_results_plugin_rules( $plugin_files_string );
    $add_class = '';
    if ( empty( $results_multi_array ) ) {
        $add_class = ' dppp-display-none';
    }
    $hide_group_plugins = '';
    $showing_plugins = ' dppp-showing-plugins';
    if ( dppp_get_option( 'dppp-hide-group-plugins' ) === 'on' ) {
        $hide_group_plugins = ' dppp-display-none';
        $showing_plugins = '';
    }

    $plugin_global_status = dppp_get_plugin_or_group_global_status( $plugin_files_string );
    if ( 'deactivated' === $plugin_global_status ) {
        $add_class_section .= ' dppp-front-end-deactivated dppp-back-end-deactivated';
    } elseif ( 'front-end-deactivated' === $plugin_global_status ) {
        $add_class_section .= ' dppp-front-end-deactivated';
    } elseif ( 'back-end-deactivated' === $plugin_global_status ) {
        $add_class_section .= ' dppp-back-end-deactivated';
    }

    echo '<div id="dppp-inner-box-' . esc_attr( $group_hash ) . '" class="dppp-inner-box' . esc_attr( $add_class_section ) . '">';
    echo '<div class="dppp-section-row' . esc_attr( $showing_plugins ) . '"><span class="dppp-row-name">' . esc_html( $loop_count )
        . '. <span class="dppp-row-name-only">' . esc_html( dppp_truncate_string( stripslashes( $plugin_group['name'] ), 40 ) ) . '</span></span>';
    echo ' <span class="dppp-group-actions">';
    echo '<a id="dppp-add-rule-link-' . esc_attr( $group_hash )
        . '" class="dppp-add-rule-link dppp-hide-on-copy-rule dppp-one-line" href="javascript:dpppAddNewRule( \'dppp-group\', \'' . esc_attr( esc_js( $group_hash ) ) . '\' )">'
        . '<span class="dashicons dashicons-plus"></span><span class="dppp-add-new-rule">' . esc_html__( 'Add New Rule', 'dppp' ) . '</span></a>';
    echo ' <a title="' . esc_attr__( 'Edit the plugin group', 'dppp' ) . '" id="dppp-edit-group-link-' . esc_attr( $group_hash )
        . '" class="dppp-edit-group-link dppp-hide-on-copy-rule" href="javascript:dpppEditPluginGroup( \'' . esc_attr( esc_js( $group_hash ) ) . '\' )">'
        . '<span class="dashicons dashicons-edit"></span></a>';
    echo ' <a title="' . esc_attr__( 'Delete the plugin group and its rules', 'dppp' ) . '" id="dppp-delete-group-link-' . esc_attr( $group_hash )
        . '" class="dppp-delete-group-link dppp-hide-on-copy-rule" href="javascript:dpppDeletePluginGroup( \'' . esc_attr( esc_js( $group_hash ) ) . '\' )">'
        . '<span class="dashicons dashicons-trash"></span></a>';
    echo ' <a id="dppp-paste-rules-link-' . esc_attr( $group_hash )
        . '" class="dppp-paste-rules-link dppp-show-on-copy-rule dppp-hide-while-pasting-rule dppp-display-none dppp-one-line" '
        . 'href="javascript:dpppPasteRules( \'dppp-group\', \'' . esc_attr( esc_js( $group_hash ) )
        . '\' )"><span class="dashicons dashicons-clipboard"></span><span class="dppp-paste-rules">'
        . '' . esc_html__( 'Paste 1 rule', 'dppp' ) . '</span></a>';
    echo ' <a id="dppp-cancel-paste-rules-link-' . esc_attr( $group_hash )
        . '" class="dppp-cancel-paste-rules-link dppp-show-on-copy-rule dppp-hide-while-pasting-rule dppp-display-none dppp-one-line" '
        . 'href="javascript:dpppCancelPasteRules()"><span class="dashicons dashicons-no"></span><span class="dppp-cancel-paste-rules">'
        . esc_html__( 'Cancel', 'dppp' ) . '</span></a>';
    echo '</span>';

    echo '<span class="dppp-plugin-or-group-action-links dppp-hide-on-copy-rule dppp-one-line">';
    echo ' <a title="' . esc_attr__( 'Front-end Deactivate', 'dppp' ) . '" id="dppp-plugin-or-group-front-end-deactivate-link-' . esc_attr( $group_hash )
        . '" class="dppp-plugin-or-group-front-end-deactivate-link" href="javascript:dpppGroupAction( \'front-end-deactivate\', \''
        . esc_attr( esc_js( $group_hash ) ) . '\' )"><span class="dashicons dashicons-welcome-view-site"></span>'
        . '<span class="dppp-plugin-action-text">' . sprintf( esc_html__( 'Front-end%sDeactivate', 'dppp' ), '<br>' ) . '</span></a>';
    echo ' <a title="' . esc_attr__( 'Front-end Activate', 'dppp' ) . '" id="dppp-plugin-or-group-front-end-activate-link-' . esc_attr( $group_hash )
        . '" class="dppp-plugin-or-group-front-end-activate-link" href="javascript:dpppGroupAction( \'front-end-activate\', \''
        . esc_attr( esc_js( $group_hash ) ) . '\' )"><span class="dashicons dashicons-welcome-view-site"></span>'
        . '<span class="dppp-plugin-action-text">' . sprintf( esc_html__( 'Front-end%sActivate', 'dppp' ), '<br>' ) . '</span></a>';
    echo ' <a title="' . esc_attr__( 'Back-end Deactivate', 'dppp' ) . '" id="dppp-plugin-or-group-back-end-deactivate-link-' . esc_attr( $group_hash )
        . '" class="dppp-plugin-or-group-back-end-deactivate-link" href="javascript:dpppGroupAction( \'back-end-deactivate\', \''
        . esc_attr( esc_js( $group_hash ) ) . '\' )"><span class="dashicons dashicons-welcome-widgets-menus"></span>'
        . '<span class="dppp-plugin-action-text">' . sprintf( esc_html__( 'Back-end%sDeactivate', 'dppp' ), '<br>' ) . '</span></a>';
    echo ' <a title="' . esc_attr__( 'Back-end Activate', 'dppp' ) . '" id="dppp-plugin-or-group-back-end-activate-link-' . esc_attr( $group_hash )
        . '" class="dppp-plugin-or-group-back-end-activate-link" href="javascript:dpppGroupAction( \'back-end-activate\', \''
        . esc_attr( esc_js( $group_hash ) ) . '\' )"><span class="dashicons dashicons-welcome-widgets-menus"></span>'
        . '<span class="dppp-plugin-action-text">' . sprintf( esc_html__( 'Back-end%sActivate', 'dppp' ), '<br>' ) . '</span></a>';
    echo ' <a title="' . esc_attr__( 'Start Group Rules', 'dppp' ) . '" id="dppp-plugin-or-group-start-rules-link-' . esc_attr( $group_hash )
        . '" class="dppp-plugin-or-group-start-rules-link' . esc_attr( $add_class ) . '" href="javascript:dpppGroupAction( \'start-rules\', \''
        . esc_attr( esc_js( $group_hash ) ) . '\' )"><span class="dashicons dashicons dashicons-controls-play"></span></a>';
    echo ' <a title="' . esc_attr__( 'Pause Group Rules', 'dppp' ) . '" id="dppp-plugin-or-group-pause-rules-link-' . esc_attr( $group_hash )
        . '" class="dppp-plugin-or-group-pause-rules-link' . esc_attr( $add_class ) . '" href="javascript:dpppGroupAction( \'pause-rules\', \''
        . esc_attr( esc_js( $group_hash ) ) . '\' )"><span class="dashicons dashicons dashicons-controls-pause"></span></a>';
    echo ' <a title="' . esc_attr__( 'Delete Group Rules', 'dppp' ) . '" id="dppp-plugin-or-group-delete-rules-link-' . esc_attr( $group_hash )
        . '" class="dppp-plugin-or-group-delete-rules-link' . esc_attr( $add_class ) . '" href="javascript:dpppGroupAction( \'delete-rules\', \''
        . esc_attr( esc_js( $group_hash ) ) . '\' )"><span class="dashicons dashicons dashicons-no"></span></a>';
    echo '</span>';

    echo '</div>';

    echo '<span class="dppp-plugins-of-group">';
    foreach ( $plugin_group['plugin-names'] as $plugin_name ) {
        echo '<span class="dppp-plugin-in-group' . esc_attr( $hide_group_plugins )
            . '"><span class="dashicons dashicons-admin-plugins"></span> <span class="dppp-plugin-name-in-group">'
            . esc_html( dppp_truncate_string( $plugin_name, 40 ) ) . '</span></span>';
    }
    echo '</span>';
    echo '<div class="dppp-plugin-rules-layer ' . esc_attr( $add_class ) . '" id="dppp-plugin-rules-' . esc_attr( $group_hash ) . '">';
    dppp_output_plugin_rules_content( 'dppp-group', $group_hash, $results_multi_array );
    echo '</div>';
    echo '</div>';
}

// Handles the ajax request for adding a new plugin group to the database
function dppp_save_plugin_group_data_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }
    $plugin_hashes = sanitize_html_class( $_POST['plugins'] );
    $group_name = sanitize_text_field( $_POST['group_name'] );
    if ( empty( $plugin_hashes ) || empty( $group_name ) ) {
        wp_die( 'invalid-data' );
    }

    // We turn the hashes into an array of plugin files
    $plugins = get_plugins();
    $selected_plugins = Array();
    foreach ( $plugins as $plugin_file => $plugin_data ) {
        $plugin_hash = md5( $plugin_file );
        if ( strpos( $plugin_hashes, $plugin_hash ) !== false ) {
            $selected_plugins[] = $plugin_file;
        }
    }

    if ( count( $selected_plugins ) < 2 ) {
        wp_die( 'at-least-two-plugins' );
    }

    // We add the selected plugins as a new group in a multi-multi-array option along with the other groups already there
    if ( ! empty( $selected_plugins ) ) {
        sort( $selected_plugins );
        $plugin_names = Array();
        foreach ( $selected_plugins as $group_plugin_file ) {
            $plugin_data = get_plugin_data( WP_PLUGIN_DIR . '/' . $group_plugin_file );
            $plugin_names[] = $plugin_data['Name'];
        }
        $group_hash = dppp_get_plugin_group_hash_from_array( $selected_plugins );
        $plugin_groups = dppp_get_option( 'dppp-plugin-groups' );
        if ( ! is_array( $plugin_groups ) && false === $plugin_groups ) {
            $plugin_groups = Array( $group_hash => Array( 'name' => $group_name, 'plugins' => $selected_plugins, 'plugin-names' => $plugin_names ) );
        } else {
            if ( array_key_exists( $group_hash, $plugin_groups ) ) {
                wp_die( 'already-exists-group' );
            }
            $plugin_groups[ $group_hash ] = Array( 'name' => $group_name, 'plugins' => $selected_plugins, 'plugin-names' => $plugin_names );
        }
        update_option( 'dppp-plugin-groups', $plugin_groups );
    }
    wp_die( '[done]' );
}

// Handles the ajax request for displaying the box with the form for adding a new plugin group
function dppp_add_new_plugin_group_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( '<h2>' . esc_html__( 'Add New Plugin Group', 'dppp' ) . '</h2>'
            . '<input type="button" class="dppp-close-button button" id="dppp-button-close-add-new" value="' . esc_attr__( 'Close', 'dppp' ) . '" />'
            . '<p class="dppp-clear">' . esc_html__( 'Error: Invalid security nonce.', 'dppp' ) . '</p>' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( '<h2>' . esc_html__( 'Add New Plugin Group', 'dppp' ) . '</h2>'
            . '<input type="button" class="dppp-close-button button" id="dppp-button-close-add-new" value="' . esc_attr__( 'Close', 'dppp' ) . '" />'
            . '<p class="dppp-clear">' . esc_html__( 'Error: You do not have sufficient permissions to make this request.', 'dppp' ) . '</p>' );
    }
    ?>
    <h2>
        <?php echo esc_html__( 'Add New Plugin Group', 'dppp' ); ?>
    </h2>
    <input type="button" class="dppp-close-button button" id="dppp-button-close-add-new" value="<?php esc_attr_e( 'Close', 'dppp' ); ?>" />
    <div class="dppp-clear">
        <?php esc_html_e( '1. Enter a name', 'dppp' ); ?><br>
        <input type="text" name="dppp-plugin-group-name" id="dppp-plugin-group-name" />
    </div>
    <div id="dppp-select-plugins">
        <?php esc_html_e( '2. Select plugins', 'dppp' ); ?>
        (<?php esc_html_e( 'Active', 'dppp' ); ?> / <span class="dppp-red"><?php esc_html_e( 'Inactive', 'dppp' ); ?></span>)

        <div class="dppp-choose-elements">
            <?php
            $plugins = get_plugins();
            foreach ( $plugins as $plugin_file => $plugin_data ) {
                if ( 'deactivate-plugins-per-page/deactivate-plugins-per-page.php' === $plugin_file ) {
                    continue;
                }
                $plugin_hash = md5( $plugin_file );
                echo '<p id="dppp-plugin-paragraph-' . esc_attr( $plugin_hash ) . '"><label><span class="dppp-plugin-span-checkbox">'
                    . '<input type="checkbox" class="dppp-plugin-checkbox" '
                    . 'id="dppp-plugin-checkbox-' . esc_attr( $plugin_hash ) . '" /> ';
                if ( is_plugin_active( $plugin_file ) ) {
                    echo esc_html( $plugin_data['Name'] );
                } else {
                    echo '<span class="dppp-red">' . esc_html( $plugin_data['Name'] ) . '</span>';
                }
                echo '</span>';

                // We show an icon if the plugin is in any group
                if ( dppp_plugin_in_any_group( $plugin_file, dppp_get_option( 'dppp-plugin-groups' ) ) ) {
                    echo '<span title="' . esc_attr__( 'The plugin is part of at least one group', 'dppp' )
                        . '" class="dashicons dashicons-screenoptions"></span>';
                }

                echo '</label></p>';
            }
            ?>
        </div>
        <div class="dppp-under-plugins-links">
            <div class="dppp-under-plugins-links-left">
                <?php esc_html_e( 'Selected:', 'dppp' ); ?> <span></span>
            </div>
            <div class="dppp-under-plugins-links-right">
                <a href="javascript:dpppSelectAllPlugins()"><?php esc_html_e( 'Select all', 'dppp' ); ?></a> /
                <a href="javascript:dpppDeselectAllPlugins()"><?php esc_html_e( 'Deselect all', 'dppp' ); ?></a>
            </div>
        </div>
    </div>
    <div>
        <input type="button" id="dppp-button-add-selected-plugins"
            class="button button-primary dppp-full-width" value="<?php esc_attr_e( 'Add New Plugin Group', 'dppp' ); ?>" />
    </div>
    <?php
    wp_die();
}

// Handles the ajax request for updating the plugin groups section with fresh data
function dppp_update_plugin_groups_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( '<p class="dppp-clear">' . esc_html__( 'Error: Invalid security nonce.', 'dppp' ) . '</p>' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( '<p class="dppp-clear">' . esc_html__( 'Error: You do not have sufficient permissions to make this request.', 'dppp' ) . '</p>' );
    }
    dppp_output_plugin_groups_section();
    wp_die();
}

// Handles the ajax request for deleting a plugin group and all its rules
function dppp_delete_plugin_group_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }
    $group_hash = sanitize_html_class( $_POST['group_hash'] );
    $plugin_groups = dppp_get_option( 'dppp-plugin-groups' );
    if ( ( ! is_array( $plugin_groups ) && false === $plugin_groups ) || ! array_key_exists( $group_hash, $plugin_groups ) ) {
        wp_die( 'group-does-not-exist' );
    } else {
        $plugin_files_string = dppp_get_plugin_files_list_from_array( $plugin_groups[ $group_hash ]['plugins'] );
        global $wpdb;

        // Delete the rules in the group
        $wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE plugin_file = %s", $plugin_files_string ) );

        // Delete the group
        unset( $plugin_groups[ $group_hash ] );
        update_option( 'dppp-plugin-groups', $plugin_groups );
    }
    wp_die( '[done]' );
}

/**
 * Returns the array of settings names that are exportable and importable.
 * @return array
 */
function dppp_get_settings_names_to_export_import() {
    return Array(
        'dppp-completely-remove',
        'dppp-inactive-plugins',
        'dppp-debug-mode',
        'dppp-hide-group-plugins',
        'dppp-groups-first',
        'dppp-trailing-slash',
        'dppp-affect-back-end',
        'dppp-uri-parameters',
        'dppp-uri-parameters-custom',
        'dppp-chosen-uri-parameters',
        'dppp-section-show-hide-settings',
        'dppp-section-show-hide-active-plugins',
        'dppp-section-show-hide-inactive-plugins',
        'dppp-section-show-hide-plugin-groups',
        'dppp-plugin-icons',
        'dppp-local-mode',
        'dppp-rules-first',
        'dppp-hide-plugins-in-group',
        'dppp-priority',
        'dppp-affect-visual-editors',
        'dppp-edit-post-parameter',
        'dppp-admin-menu-cache-time',
    );
}

// Outputs the HTML code used to show the form for imporing rules
function dppp_output_import_rules_form_hidden() {
    ?>
    <div id="dppp-import-rules-hidden-form" class="dppp-display-none">
        <h2><?php esc_html_e( 'Import Deactivation Rules and Settings', 'dppp' ); ?></h2>
        <input type="button" class="dppp-close-button button not-execute" id="dppp-button-close-import-rules" value="<?php esc_attr_e( 'Close', 'dppp' ); ?>" />
        <div class="dppp-clear">
            <p>
                <?php
                esc_html_e( 'Here you can import all the deactivation rules, plugin groups, and settings of "Deactivate Plugins Per Page" from another website. '
                    . 'To do that paste below the text you got after you clicked the "Export" button on the other site, and click "Import".', 'dppp' );
                ?>
            </p>
            <p>
                <?php
                printf( esc_html__( 'Keep in mind that rules, which have Page URI selection type, will only be imported if the affected pages '
                    . 'exist on both sites and have the same IDs and addresses (slugs). Otherwise these rules %swill be skipped%s.', 'dppp' ), '<b>', '</b>' );
                ?>
            </p>
            <p class="dppp-red">
                <?php
                printf( esc_html__( 'Importing will %sPERMANENTLY DELETE%s all current deactivation rules, plugin groups, '
                    . 'and settings of "Deactivate Plugins Per Page"!', 'dppp' ), '<b>', '</b>' );
                ?>
            </p>
        </div>
        <div class="dppp-clear">
            <textarea id="dppp-import-rules-textarea"></textarea>
        </div>

        <?php
        if ( is_multisite() && current_user_can( 'manage_network' ) ) {
            ?>
            <div class="dppp-clear dppp-import-multiple-contain">
                <label>
                    <input type="checkbox" id="dppp-import-multiple-sites"> <?php esc_html_e( 'Import into multiple sites', 'dppp' ); ?>
                </label>
            </div>
            <div class="dppp-clear dppp-import-affected-contain dppp-display-none">
                <label>
                    <?php esc_html_e( 'Affected sites:', 'dppp' ); ?><br>
                    <select name="dppp-import-affected-sites" id="dppp-import-affected-sites">
                        <option value="selected"><?php esc_html_e( 'Sites with these IDs:', 'dppp' ); ?></option>
                        <option value="except-selected"><?php esc_html_e( 'All sites except sites with these IDs:', 'dppp' ); ?></option>
                        <option value="all"><?php esc_html_e( 'All sites', 'dppp' ); ?></option>
                    </select>
                    <br>
                    <input type="text" name="dppp-import-affected-ids" id="dppp-import-affected-ids" value=""
                        placeholder="<?php esc_attr_e( 'Enter a comma-separated list of site IDs here', 'dppp' ); ?>">
                </label>
            </div>
            <?php
        }
        ?>

        <div>
            <input type="button" id="dppp-button-import-rules"
                class="button button-primary dppp-full-width" value="<?php esc_attr_e( 'Import', 'dppp' ); ?>" />
        </div>
    </div>
    <?php
}

// Outputs the HTML code used to show the form for exporting rules
function dppp_output_export_rules_form_hidden() {
    ?>
    <div id="dppp-export-rules-hidden-form" class="dppp-display-none">
        <h2><?php esc_html_e( 'Export Deactivation Rules and Settings', 'dppp' ); ?></h2>
        <input type="button" class="dppp-close-button button" id="dppp-button-close-export-rules" value="<?php esc_attr_e( 'Close', 'dppp' ); ?>" />
        <div class="dppp-clear">
            <p>
                <?php
                esc_html_e( 'Here you can export all deactivation rules, plugin groups, and settings of "Deactivate Plugins Per Page". '
                    . 'This includes rules for currently inactive plugins as well. After that you can import them on another site.', 'dppp' );
                ?>
            </p>
            <p>
                <?php
                printf( esc_html__( 'Keep in mind that rules, which have Page URI selection type, will only be imported if the affected pages '
                    . 'exist on both sites and have the same IDs and addresses (slugs). Otherwise these rules %swill be skipped%s.', 'dppp' ), '<b>', '</b>' );
                ?>
            </p>
            <p>
                <?php
                printf( esc_html__( 'Also keep in mind that later when you import the data on the destination site, all its previous '
                    . 'deactivation rules, plugin groups, and settings of "Deactivate Plugins Per Page" '
                    . '%swill be deleted%s and replaced with the new data.', 'dppp' ), '<b>', '</b>' );
                ?>
            </p>
        </div>
        <div>
            <input type="button" id="dppp-button-export-rules"
                class="button button-primary dppp-full-width" value="<?php esc_attr_e( 'Export', 'dppp' ); ?>" />
        </div>
    </div>
    <?php
}

/**
 * I use this function to debug the plugin sometimes (to see the value of some variable), it is not used in the actual plugin
 * @param string $variable_name
 * @param mixed $variable_name
 */
function dppp_log( $variable_name, $variable_data ) {
    $file_path = plugin_dir_path( __FILE__ ) . 'dppp_debug_log.txt';
    if ( ! file_exists( $file_path ) || filesize( $file_path ) < 10000000 ) {
        file_put_contents( $file_path, $variable_name . ':' . $variable_data . ";\r\n", FILE_APPEND | LOCK_EX );
    }
}

// Handles the ajax request for global actions (delete all rules, pause all rules, start all rules, hide group plugins, show group plugins)
function dppp_global_actions_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }
    $global_action = sanitize_html_class( $_POST['global_action'] );
    global $wpdb;
    if ( 'delete-all-rules' === $global_action ) {
        $wpdb->query( "DELETE FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE uri_type != 'global'" );
    } elseif ( 'pause-all-rules' === $global_action ) {
        $wpdb->query( "UPDATE {$wpdb->prefix}dppp_plugin_deactivation_rules SET status = 'paused' WHERE uri_type != 'global'" );
    } elseif ( 'start-all-rules' === $global_action ) {
        $wpdb->query( "UPDATE {$wpdb->prefix}dppp_plugin_deactivation_rules SET status = 'started' WHERE uri_type != 'global'" );
    } elseif ( 'hide-group-plugins' === $global_action ) {
        update_option( 'dppp-hide-group-plugins', 'on' );
    } elseif ( 'show-group-plugins' === $global_action ) {
        update_option( 'dppp-hide-group-plugins', 'off' );

    // Executing global action: import rules and settings
    } elseif ( 'execute-import-rules' === $global_action ) {

        if ( isset( $_POST['multiple'] ) && 'Yes' === $_POST['multiple'] ) {
            if ( ! current_user_can( 'manage_network' ) ) {
                wp_die( 'no-access' );
            }
        }

        $import = json_decode( base64_decode( $_POST['text'] ), true );

        if ( $import === null || ! is_array( $import )
            || ! array_key_exists( 'export-information', $import )
            || ! array_key_exists( 'deactivation-rules', $import )
            || ! array_key_exists( 'plugin-groups', $import )
            || ! array_key_exists( 'plugin-settings', $import )
            || ! array_key_exists( 'pages', $import )
            || ! array_key_exists( 'version', $import['export-information'] )
            || ! array_key_exists( 'rules-count', $import['export-information'] ) || count( $import ) !== 5 ) {
            wp_die( 'invalid-import-data' );
        }

        // Validate the data for importing the plugin settings
        $settings_names = dppp_get_settings_names_to_export_import();
        foreach ( $settings_names as $current_name ) {
            if ( ! array_key_exists( $current_name, $import['plugin-settings'] ) ) {
                wp_die( 'invalid-import-data' );
            }
            if ( 'dppp-chosen-uri-parameters' === $current_name ) {
                $chosen_uri_parameters = sanitize_textarea_field( $import['plugin-settings']['dppp-chosen-uri-parameters'] );
                if ( ! dppp_is_comma_separated_uri_parameters( $chosen_uri_parameters ) && ! empty( $chosen_uri_parameters ) ) {
                    wp_die( 'invalid-import-data' );
                }
            }
        }

        // Validate the array keys of the data for importing the deactivation rules
        foreach ( $import['deactivation-rules'] as $rule ) {
            if ( ! is_array( $rule ) || ! array_key_exists( 'rule_id', $rule ) || ! array_key_exists( 'status', $rule )
            || ! array_key_exists( 'deactivation_type', $rule ) || ! array_key_exists( 'uri_type', $rule )
            || ! array_key_exists( 'page_ids', $rule ) || ! array_key_exists( 'plugin_file', $rule )
            || ! array_key_exists( 'uri_condition', $rule ) || ! array_key_exists( 'uri_value', $rule ) || ! array_key_exists( 'devices', $rule ) ) {
                wp_die( 'invalid-import-data' );
            }
        }

        // Validate rules count
        if ( intval( $import['export-information']['rules-count'] ) !== count( $import['deactivation-rules'] ) ) {
            wp_die( 'invalid-import-data' );
        }

        // Validate the data for importing the plugin groups
        if ( is_array( $import['plugin-groups'] ) || empty( $import['plugin-groups'] ) ) {
            if ( is_array( $import['plugin-groups'] ) && count( $import['plugin-groups'] ) > 0 ) {
                foreach ( $import['plugin-groups'] as $group_hash => $group_data ) {
                    if ( ! array_key_exists( 'name', $group_data ) || ! array_key_exists( 'plugins', $group_data )
                        || ! array_key_exists( 'plugin-names', $group_data ) || count( $group_data ) !== 3 ) {
                        wp_die( 'invalid-import-data' );
                    }
                    if ( ! is_array( $group_data['plugins'] ) || ! is_array( $group_data['plugin-names'] )
                        || count( $group_data['plugins'] ) !== count( $group_data['plugin-names'] ) ) {
                        wp_die( 'invalid-import-data' );
                    }
                    foreach ( $group_data['plugin-names'] as $plugin_name ) {
                        if ( sanitize_text_field( dppp_strip_whitespace( $plugin_name ) ) !== dppp_strip_whitespace( $plugin_name ) ) {
                            wp_die( 'invalid-import-data' );
                        }
                    }
                    foreach ( $group_data['plugins'] as $plugin_file ) {
                        if ( ! dppp_is_valid_plugin_file( $plugin_file ) ) {
                            wp_die( 'invalid-import-data' );
                        }
                    }
                    $group_hash_import = dppp_get_plugin_group_hash_from_array( $group_data['plugins'] );
                    if ( $group_hash_import !== $group_hash ) {
                        wp_die( 'invalid-import-data' );
                    }
                }
            }
        } else {
            wp_die( 'invalid-import-data' );
        }

        // We are importing into multiple sites
        if ( 'Yes' === $_POST['multiple'] ) {

            if ( ! isset( $_POST['sites'] ) || ! isset( $_POST['ids'] ) || ! in_array( $_POST['sites'], Array( 'selected', 'except-selected', 'all' ) ) ) {
                wp_die( 'invalid-multiple-settings' );
            }

            if ( $_POST['sites'] !== 'all' && ( ! dppp_is_comma_separated_numbers( dppp_strip_whitespace( $_POST['ids'] ) )
                || dppp_strip_whitespace( $_POST['ids'] ) === '' ) ) {
                wp_die( 'invalid-multiple-settings' );
            }

            $continue_task = 'no';
            if ( isset( $_POST['task_id'] ) && 'none' !== $_POST['task_id'] ) {
                $continue_task = 'yes';
            }

            $start_time = time();

            // We are continuing a task started in a previous request
            if ( 'yes' === $continue_task ) {
                $task_id = sanitize_html_class( $_POST['task_id'] );
                $affect_site_ids = get_site_option( 'dppp-import-task-' . $task_id );
                if ( ! is_array( $affect_site_ids ) && false === $affect_site_ids ) {
                    wp_die( 'cannot-continue-task' );
                }

            // We are starting a new task
            } else {

                // We will get the IDs of all affected sites
                $get_sites_arguments = array(
                    'offset' => 0,
                    'number' => 200000,
                    'fields' => 'ids'
                );

                if ( 'all' !== $_POST['sites'] ) {
                    $site_ids = explode( ',', dppp_sanitize_comma_separated_numbers( dppp_strip_whitespace( $_POST['ids'] ) ) );
                    if ( 'selected' === $_POST['sites'] ) {
                        $get_sites_arguments['site__in'] = $site_ids;
                    }
                    if ( 'except-selected' === $_POST['sites'] ) {
                        $get_sites_arguments['site__not_in'] = $site_ids;
                    }
                }

                $affect_site_ids = get_sites( $get_sites_arguments );
            }

            if ( empty( $affect_site_ids ) ) {
                wp_die( 'no-affected-sites' );
            }

            $done = 0;
            $timeout = 'no';
            $all_count = count( $affect_site_ids );

            // We go through the sites to be affected, we switch to them and import the data if the plugin is activated
            foreach ( $affect_site_ids as $affect_site_id ) {
                switch_to_blog( $affect_site_id );
                if ( is_plugin_active( 'deactivate-plugins-per-page/deactivate-plugins-per-page.php' ) ) {

                    /*
                     * We do this because some sites may have never been visited since the plugin was network activated and they will not even have
                     * the database table created.
                     */
                    dppp_update_database_and_mu_plugin();

                    dppp_import( $import );
                }
                restore_current_blog();
                $done++;

                // If 10 seconds have passed we will stop and continue in the next request
                if ( ( time() -  $start_time ) > 10 && $all_count !== $done ) {
                    $timeout = 'yes';
                    break;
                }
            }

            // If the time for this request has passed we will save the progress and continue in the next one
            if ( 'yes' === $timeout ) {
                if ( 'yes' !== $continue_task ) {
                    $task_id = dppp_random_string( 10 );
                }
                $remaining_site_ids = array_slice( $affect_site_ids, $done );
                unset( $affect_site_ids );
                update_site_option( 'dppp-import-task-' . $task_id, $remaining_site_ids );
                unset( $remaining_site_ids );

                // This is the data to pass to the javascript function in json format
                $json_result = array(
                    'done' => $done,
                    'taskID' => $task_id,
                );

                echo 'dppp-timeout';
                echo json_encode( $json_result );
                wp_die();

            // We are done
            } else {
                unset( $affect_site_ids );
                if ( 'yes' === $continue_task ) {
                    delete_site_option( 'dppp-import-task-' . $task_id );
                }
                ?>

                <h2><?php esc_html_e( 'Import Deactivation Rules and Settings', 'dppp' ); ?></h2>
                <input type="button" class="dppp-close-button button" id="dppp-button-close-import-rules-reload"
                    value="<?php esc_attr_e( 'Close', 'dppp' ); ?>" />
                <div class="dppp-clear">
                    <?php
                        echo '<p>';
                        esc_html_e( 'Importing completed.', 'dppp' );
                        echo '</p>';
                        echo '<p>';
                        esc_html_e( 'If there were sites that did not have this plugin activated, they were skipped.', 'dppp' );
                        echo '</p>';
                        echo '<p>';
                        esc_html_e( 'If there were rules with Page URI selection, that affect pages that do not exist on both sites with '
                            . 'the same IDs and addresses (slugs), they were skipped.', 'dppp' );
                        echo '</p>';
                    ?>
                </div>

                <?php
            }
        } else {
            $skipped_array = dppp_import( $import );
            ?>

            <h2><?php esc_html_e( 'Import Deactivation Rules and Settings', 'dppp' ); ?></h2>
            <input type="button" class="dppp-close-button button" id="dppp-button-close-import-rules-reload" value="<?php esc_attr_e( 'Close', 'dppp' ); ?>" />
            <div class="dppp-clear">
                <?php
                if ( count( $skipped_array['skipped_rule_ids'] ) > 0 ) {
                    echo '<p>';
                    esc_html_e( sprintf( 'Importing completed. We had to skip %d deactivation rule(s) with Page URI selection type. '
                        . 'The affected pages either do not exist, or have a different address (slug) or ID.',
                        count( $skipped_array['skipped_rule_ids'] ) ), 'dppp' );
                    echo '</p>';
                    echo '<p>';
                    esc_html_e( 'IDs of skipped rules:', 'dppp' );
                    echo '</p>';
                    echo '<textarea id="dppp-skipped-rules-textarea">';
                    foreach ( $skipped_array['skipped_rule_ids'] as $rule_id ) {
                        echo intval( $rule_id ) . "\n";
                    }
                    echo '</textarea>';
                    echo '<p>';
                    esc_html_e( 'IDs of pages that either do not exist or have a different address (slug):', 'dppp' );
                    echo '</p>';
                    echo '<textarea id="dppp-skipped-pages-textarea">';
                    foreach ( $skipped_array['skipped_page_ids'] as $page_id ) {
                        echo intval( $page_id ) . "\n";
                    }
                    echo '</textarea>';
                } else {
                    echo '<p>';
                    esc_html_e( 'Importing completed.', 'dppp' );
                    echo '</p>';
                }
                ?>
            </div>

            <?php
        }

        wp_die();

    } elseif ( 'execute-export-rules' === $global_action ) {

        ?>

        <h2><?php esc_html_e( 'Export Deactivation Rules and Settings', 'dppp' ); ?></h2>
        <input type="button" class="dppp-close-button button" id="dppp-button-close-export-rules" value="<?php esc_attr_e( 'Close', 'dppp' ); ?>" />
        <div class="dppp-clear">
            <p>
                <?php
                printf( esc_html__( '%sSelect all the text%s in the box below and copy it. Then use the '
                    . '"Import Rules and Settings" global action on the destination website, and paste the text in the provided text box.', 'dppp' ),
                    '<a href="javascript:dpppSelectAllExported()">', '</a>' );
                ?>
            </p>
        </div>

        <?php

        $results_multi_array = $wpdb->get_results( "SELECT rule_id, status, plugin_file, deactivation_type, uri_type, "
            . "page_ids, uri_condition, uri_value, devices, custom_function, note FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE 1", ARRAY_A );

        if ( empty( $results_multi_array ) ) {
            $results_multi_array = Array();
        }

        $pages = Array();
        $affected_page_ids = Array();

        // We fill an array with the page IDs that are affected by rules with page URI selection type
        foreach ( $results_multi_array as $rule ) {
            if ( 'page' === $rule['uri_type'] ) {
                $page_ids_array = explode( ',', $rule['page_ids'] );
                foreach ( $page_ids_array as $page_id ) {
                    if ( ! in_array( intval( $page_id ), $affected_page_ids ) ) {
                        $affected_page_ids[] = intval( $page_id );
                    }
                }
            }
        }

        // We fill an array with the IDs and slugs of pages that are affected by rules with page URI selection type
        foreach ( $affected_page_ids as $affected_page_id ) {
            $pages[ $affected_page_id ] = get_post_field( 'post_name', $affected_page_id );
        }

        $information = Array(
            'unix-time' => time(),
            'version' => DPPP_DEACTIVATE_PLUGINS_PER_PAGE_VERSION,
            'rules-count' => count( $results_multi_array ),
        );

        $plugin_settings = Array();
        $settings_names = dppp_get_settings_names_to_export_import();
        foreach ( $settings_names as $current_name ) {
            $plugin_settings[ $current_name ] = dppp_get_option( $current_name );
        }

        $export_data = Array(
            'export-information' => $information,
            'deactivation-rules' => $results_multi_array,
            'plugin-groups' => dppp_get_option( 'dppp-plugin-groups' ),
            'plugin-settings' => $plugin_settings,
            'pages' => $pages,
        );

        echo '<textarea id="dppp-export-textarea">' . esc_textarea( base64_encode( json_encode( $export_data ) ) ) . '</textarea>';

        wp_die();

    } else {
        wp_die( 'invalid-data' );
    }
    wp_die( '[done]' );
}

/**
 * Generates a random string. We can choose the length and also which characters to exclude.
 * @param int $length
 * @param string $exclude
 * @return string
 */
function dppp_random_string( $length, $exclude = '' ) {

    // Available characters to choose from
    $characters = '0123456789abcdefghijklmnopqrstuvwxyz';

    // Exclude some characters (if any)
    $characters = str_replace( str_split( $exclude ), '', $characters );

    // The count of the availavle characters
    $available_character_length = strlen( $characters );

    // Start with an empty string
    $random_string = '';

    // Add a random character from the available ones
    // And do this $length times
    for ( $i = 0; $i < $length; $i++ ) {
        $random_string .= $characters[ rand( 0, $available_character_length - 1 ) ];
    }

    // Return the string we made
    return $random_string;
}

/**
 * Imports the plugin rules and settings in the current site (replacing the existing ones)
 * @param array $import
 * @return array
 */
function dppp_import( $import ) {
    $skip_page_ids = Array();

    // Go through all the affected pages
    foreach ( $import['pages'] as $page_id => $page_slug ) {
        $page_id = intval( $page_id );
        $post_type = get_post_field( 'post_type', $page_id );
        $post_name = get_post_field( 'post_name', $page_id );

        // If the page id does not exist or it is not a page, or it has a different slug, we will skip the rules about that page
        if ( 'page' !== $post_type || $post_name !== $page_slug ) {
            $skip_page_ids[] = $page_id;
        }
    }

    global $wpdb;

    // At this point we consider the data valid, and proceed with deleting all current deactivation rules
    $wpdb->query( "DELETE FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE 1" );

    // Import the plugin settings
    $settings_names = dppp_get_settings_names_to_export_import();
    foreach ( $settings_names as $current_name ) {
        if ( 'dppp-chosen-uri-parameters' === $current_name ) {
            update_option( $current_name, sanitize_textarea_field( $import['plugin-settings'][ $current_name ] ) );
        } else {
            update_option( $current_name, sanitize_html_class( $import['plugin-settings'][ $current_name ] ) );
        }
    }

    $skipped_rule_ids = Array();

    // Import deactivation rules
    foreach ( $import['deactivation-rules'] as $rule ) {

        $uri_type = sanitize_html_class( $rule['uri_type'] );
        $page_ids = dppp_sanitize_comma_separated_numbers( $rule['page_ids'] );
        $skip_rule = 'no';

        // If the page id does not exist or it is not a page, or it has a different slug, we will skip this rule. We have prepared an array of page IDs.
        if ( 'page' === $uri_type ) {
            $page_ids_array = explode( ',', $page_ids );
            foreach ( $page_ids_array as $page_id ) {
                if ( in_array( intval( $page_id ), $skip_page_ids ) ) {
                    $skip_rule = 'yes';
                    break;
                }
            }
        }

        // If the page id does not exist or it is not a page, or it has a different slug, we skip this rule
        if ( 'yes' === $skip_rule ) {
            $skipped_rule_ids[] = intval( $rule['rule_id'] );
            continue;
        }

        $status = sanitize_html_class( $rule['status'] );
        $deactivation_type = sanitize_html_class( $rule['deactivation_type'] );
        $devices = sanitize_html_class( $rule['devices'] );
        $custom_function = sanitize_text_field( $rule['custom_function'] );
        $note = sanitize_textarea_field( $rule['note'] );

        $plugin_file_new_separators = str_replace( ';;', 'dppp-semicolon-separator', $rule['plugin_file'] );
        $plugin_file_safe = dppp_sanitize_plugin_file( $plugin_file_new_separators );
        $plugin_file = str_replace( 'dppp-semicolon-separator', ';;', $plugin_file_safe );

        $uri_condition_new_separators = str_replace( '[dppp-or]', 'dppp-condition-separator', $rule['uri_condition'] );
        $uri_condition_safe = sanitize_html_class( $uri_condition_new_separators );
        $uri_condition = str_replace( 'dppp-condition-separator', '[dppp-or]', $uri_condition_safe );

        if ( $rule['uri_value'] !== '[dppp-front-end]' && $rule['uri_value'] !== '[dppp-back-end]' && strpos( $uri_condition, 'regular-expression' ) === false ) {
            $uri_value_new_separators = str_replace( '[dppp-or]', 'dppp-uri-value-separator', $rule['uri_value'] );
            $uri_value_safe = dppp_escape_uri( $uri_value_new_separators );
            $uri_value = str_replace( 'dppp-uri-value-separator', '[dppp-or]', $uri_value_safe );
        } else {
            $uri_value = $rule['uri_value'];
        }

        $status_insert = $wpdb->insert(
        	"{$wpdb->prefix}dppp_plugin_deactivation_rules",
        	array(
        		'status' => $status,
        		'plugin_file' => $plugin_file,
        		'deactivation_type' => $deactivation_type,
        		'uri_type' => $uri_type,
        		'page_ids' => $page_ids,
        		'uri_condition' => $uri_condition,
        		'uri_value' => $uri_value,
        		'devices' => $devices,
        		'custom_function' => $custom_function,
                'note' => $note,
        		'unix_time_added' => time(),
        	),
        	array(
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%s',
        		'%d',
        	)
        );
    }

    // Import plugin groups if any
    if ( is_array( $import['plugin-groups'] ) && count( $import['plugin-groups'] ) > 0 ) {
        update_option( 'dppp-plugin-groups', $import['plugin-groups'] );
    } else {
        delete_option( 'dppp-plugin-groups' );
    }

    // Refresh the page URI values since if we import from a site inside a different folder name, the URI is different for the same pages.
    dppp_refresh_page_uri_values();

    return Array( 'skipped_rule_ids' => $skipped_rule_ids, 'skipped_page_ids' => $skip_page_ids );
}

// Outputs the HTML code of the plugin groups section
function dppp_output_plugin_groups_section() {
    ?>
    <b class="dppp-subheading">
        <span class="dashicons dashicons-screenoptions"></span> <?php esc_html_e( 'Plugin Groups', 'dppp' ); ?>
    </b>
    <a id="dppp-show-hide-plugin-groups" title="<?php esc_attr_e( 'Show/Hide Section', 'dppp' ); ?>"
        class="dppp-show-hide-section" href="javascript:dpppShowHideSection( 'plugin-groups' )">
        <?php
        if ( dppp_get_option( 'dppp-section-show-hide-plugin-groups' ) !== 'hide' ) {
            $add_class_plugin_groups = '';
            echo '<span class="dashicons dashicons-arrow-up-alt2"></span>';
        } else {
            $add_class_plugin_groups = ' dppp-display-none';
            echo '<span class="dashicons dashicons-arrow-down-alt2"></span>';
        }
        ?>
    </a>
    <br class="dppp-hide-above-750">
    <a id="dppp-add-plugin-group-link" class="dppp-one-line dppp-hide-on-copy-rule" href="javascript:dpppAddNewPluginGroup()">
        <span class="dashicons dashicons-plus"></span><span class="dppp-add-new-plugin-group"><?php esc_html_e( 'Add New Plugin Group', 'dppp' ); ?></span>
    </a>
    <?php
    if ( dppp_get_option( 'dppp-hide-group-plugins' ) !== 'on' ) {
        ?>
        <a id="dppp-hide-group-plugins-link" class="dppp-one-line dppp-hide-on-copy-rule"
            href="javascript:dpppGlobalActions( 'hide-group-plugins', 'none' )">
            <span class="dashicons dashicons-admin-plugins"></span><span
            class="dppp-hide-group-plugins"><?php esc_html_e( 'Hide Group Plugins', 'dppp' ); ?></span>
        </a>
        <?php
    } else {
        ?>
        <a id="dppp-show-group-plugins-link" class="dppp-one-line dppp-hide-on-copy-rule"
            href="javascript:dpppGlobalActions( 'show-group-plugins', 'none' )">
            <span class="dashicons dashicons-admin-plugins"></span><span
            class="dppp-show-group-plugins"><?php esc_html_e( 'Show Group Plugins', 'dppp' ); ?></span>
        </a>
        <?php
    }
    $plugin_groups = dppp_get_option( 'dppp-plugin-groups' );
    if ( ! is_array( $plugin_groups ) || $plugin_groups === false || empty( $plugin_groups ) ) {
        ?>
        <div class="dppp-inner-box<?php echo esc_attr( $add_class_plugin_groups ); ?>">
            <div class="dppp-section-row">
                <span class="dppp-section-row-message">
                    <?php
                    echo esc_html__( 'You do not have any plugin groups right now. Add a plugin group in order to create '
                        . 'deactivation rules that affect multiple plugins.', 'dppp' );
                    ?>
                </span>
            </div>
        </div>
        <?php
    } else {
        $loop_count = 1;
        foreach ( $plugin_groups as $plugin_group ) {
            dppp_output_plugin_group_row( $loop_count, $plugin_group, $add_class_plugin_groups );
            $loop_count++;
        }
    }
}

/**
 * Strips all characters from a string except digits and dots
 * @param string $string
 * @return string
 */
function dppp_sanitize_version( $string ) {
    return preg_replace( "/[^0-9.]/", "", $string );
}

/**
 * Outputs the URI conditions and values for a rule
 * @param string $uri_type
 * @param string $uri_conditions
 * @param string $uri_values
 */
function dppp_output_uri_conditions_and_values( $uri_type, $uri_conditions, $uri_values ) {
    if ( 'page' === $uri_type ) {

        // Works for single or multiple selected pages
        $uri_values_array = explode( '[dppp-or]', $uri_values );
        for ( $loop = 0; $loop < count( $uri_values_array ); $loop++ ) {
            if ( 0 !== $loop ) {
                echo '<br>' . esc_html__( 'or', 'dppp' ) . ' ';
            }
            echo esc_html__( 'equals: ', 'dppp' );
            echo '<strong>' . esc_html( $uri_values_array[ $loop ] ) . '</strong>';
        }

    } elseif ( 'custom' === $uri_type ) {

        // Multiple custom conditions
        if ( strpos( $uri_conditions, '[dppp-or]' ) !== false ) {
            $uri_conditions_array = explode( '[dppp-or]', $uri_conditions );
            $uri_values_array = explode( '[dppp-or]', $uri_values );
            for ( $loop = 0; $loop < count( $uri_conditions_array ); $loop++ ) {
                if ( 0 !== $loop ) {
                    echo '<br>' . esc_html__( 'or', 'dppp' ) . ' ';
                }
                echo esc_html( dppp_get_rule_condition_text( $uri_conditions_array[ $loop ] ) );
                echo '<strong>' . esc_html( $uri_values_array[ $loop ] ) . '</strong>';
            }

        // Single custom condition
        } else {
            echo esc_html( dppp_get_rule_condition_text( $uri_conditions ) );
            echo '<strong>' . esc_html( $uri_values ) . '</strong>';
        }

    } else {
        echo esc_html__( 'Error', 'dppp' );
    }
}

/**
 * Check is the values for the more custom conditions are valid
 * @param string $more_custom_conditions
 * @param string $more_custom_values
 * @return bool
 */
function dppp_are_more_conditions_and_values_valid( $more_custom_conditions, $more_custom_values ) {
    if ( empty( $more_custom_conditions ) && empty( $more_custom_values ) ) {
        return true;
    }
    if ( strpos( $more_custom_conditions, '--dppp-or--' ) === false || strpos( $more_custom_values, '--dppp-or--' ) === false
        || substr_count( $more_custom_conditions, '--dppp-or--' ) !== substr_count( $more_custom_values, '--dppp-or--' ) ) {
        return false;
    }
    $conditions = explode( '--dppp-or--', $more_custom_conditions );
    foreach ( $conditions as $key => $condition ) {
        if ( 0 === $key ) {
            continue;
        }
        if ( ! in_array( $condition, Array( 'equals', 'does-not-equal', 'contains', 'does-not-contain', 'starts-with', 'does-not-start-with', 'ends-with',
            'does-not-end-with', 'regular-expression' ) ) ) {
            return false;
        }
    }
    $values = explode( '--dppp-or--', $more_custom_values );
    foreach ( $values as $key => $value ) {
        if ( 0 === $key ) {
            continue;
        }
        if ( empty( $value ) ) {
            return false;
        }
    }
    return true;
}

/**
 * Check is the values for the more custom conditions have a regular expression that is valid. If there are no regex conditions, it is also valid.
 * @param string $more_custom_conditions
 * @param string $more_custom_values
 * @return bool
 */
function dppp_are_more_conditions_and_values_regex_valid( $more_custom_conditions, $more_custom_values ) {
    if ( empty( $more_custom_conditions ) && empty( $more_custom_values ) ) {
        return true;
    }
    if ( strpos( $more_custom_conditions, '--dppp-or--' ) === false || strpos( $more_custom_values, '--dppp-or--' ) === false
        || substr_count( $more_custom_conditions, '--dppp-or--' ) !== substr_count( $more_custom_values, '--dppp-or--' ) ) {
        return false;
    }
    $conditions = explode( '--dppp-or--', $more_custom_conditions );
    $keys_with_regex = array();
    foreach ( $conditions as $key => $condition ) {
        if ( 0 === $key ) {
            continue;
        }
        if ( $condition === 'regular-expression' ) {
            $keys_with_regex[] = $key;
        }
    }
    if ( empty( $keys_with_regex ) ) {
        return true;
    }
    $values = explode( '--dppp-or--', $more_custom_values );
    foreach ( $values as $key => $value ) {
        if ( 0 === $key ) {
            continue;
        }
        if ( in_array( $key, $keys_with_regex ) && ! dppp_is_valid_regular_expression( $value ) ) {
            return false;
        }
    }
    return true;
}

// Handles the ajax request for displaying the box for editing an existing rule
function dppp_edit_rule_callback() {
    $rule_id = intval( $_POST['rule_id'] );
    echo '<h2>';
    echo esc_html( sprintf( __( 'Edit rule with ID %d', 'dppp' ), $rule_id ) );
    echo '</h2>';
    echo '<input type="button" class="dppp-close-button button" id="dppp-button-close-edit-rule" value="' . esc_attr__( 'Close', 'dppp' ) . '" />';
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( '<p class="dppp-clear">' . esc_html__( 'Error: Invalid security nonce.', 'dppp' ) . '</p>' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( '<p class="dppp-clear">' . esc_html__( 'Error: You do not have sufficient permissions to make this request.', 'dppp' ) . '</p>' );
    }
    global $wpdb;
    $result = $wpdb->get_row( $wpdb->prepare( "SELECT deactivation_type, uri_condition, uri_value, devices, custom_function, note, uri_type, page_ids FROM "
        . "{$wpdb->prefix}dppp_plugin_deactivation_rules WHERE rule_id = %d", $rule_id ), ARRAY_A );
    if ( ! empty( $result ) ) {
        if ( ! in_array( $result['deactivation_type'], Array( 'selected', 'except' ) ) || ! in_array( $result['uri_type'], Array( 'page', 'custom' ) )
            || ! in_array( $result['devices'], Array( 'all', 'desktop', 'mobile' ) )
            || ! dppp_is_comma_separated_numbers( $result['page_ids'] ) ) {
            wp_die( '<p class="dppp-clear">' . esc_html__( 'Error: Invalid data in the database', 'dppp' ) . '</p>' );
        }
    } else {
        wp_die( '<p class="dppp-clear">' . esc_html__( 'Error: Rule not found', 'dppp' ) . '</p>' );
    }
    ?>
    <div class="dppp-clear">
        <?php esc_html_e( '1. Deactivation type', 'dppp' ); ?><br>
        <select name="dppp-select-deactivation-type" id="dppp-select-deactivation-type">
            <option value="selected"<?php selected( $result['deactivation_type'], 'selected' ); ?>>
                <?php esc_html_e( 'Deactivate on selected URI', 'dppp' ); ?>
            </option>
            <option value="except"<?php selected( $result['deactivation_type'], 'except' ); ?>>
                <?php esc_html_e( 'Deactivate on all URI except selected', 'dppp' ); ?>
            </option>
        </select>
    </div>
    <div class="dppp-margin-bottom-15">
        <?php
        esc_html_e( '2. URI selection type', 'dppp' );
        echo '<br>';
        if ( 'page' === $result['uri_type'] ) {
            esc_html_e( 'Page', 'dppp' );
        } elseif ( 'custom' === $result['uri_type'] ) {
            esc_html_e( 'Custom', 'dppp' );
        } else {
            esc_html_e( 'Error', 'dppp' );
        }
        ?>
    </div>
    <?php
    if ( 'page' === $result['uri_type'] ) {
        echo '<div id="dppp-page-uri-type">';
        esc_html_e( '3. Select pages to get URI from (updates automatically if the page slug changes)', 'dppp' );

        // If the WPML plugin is active, we show a small menu to filter pages by language
        if ( is_plugin_active( 'sitepress-multilingual-cms/sitepress.php' ) ) {
            global $sitepress;
            $languages = apply_filters( 'wpml_active_languages', NULL );
            if ( ! empty( $languages ) ) {
                echo '<br>';
                esc_html_e( 'Filter by WPML language: ', 'dppp' );
                echo '<a id="dppp-wpml-filter-pages-all" class="dppp-display-none dppp-wpml-filter-pages" '
                    . 'href="javascript:dpppFilterPagesByLanguageWPML( \'dppp-all\' )">'
                    . esc_html__( 'All', 'dppp' ) . '</a><span id="dppp-wpml-filter-pages-all-active" class="dppp-wpml-filter-pages-active">'
                    . esc_html__( 'All', 'dppp' ) . '</span> | ';
                $number = 1;
                foreach( $languages as $language ) {
                    echo '<a id="dppp-wpml-filter-pages-' . esc_attr( sanitize_html_class( $language['language_code'] ) )
                        . '" class="dppp-wpml-filter-pages" href="javascript:dpppFilterPagesByLanguageWPML( \''
                        . esc_attr( esc_js( sanitize_html_class( $language['language_code'] ) ) ) . '\' )">'
                        . esc_html( strtoupper( $language['language_code'] ) ) . '</a><span id="dppp-wpml-filter-pages-'
                        . esc_attr( sanitize_html_class( $language['language_code'] ) ) . '-active" class="dppp-display-none dppp-wpml-filter-pages-active">'
                        . esc_html( strtoupper( $language['language_code'] ) ) . '</span>';
                    if ( $number !== count( $languages ) ) {
                        echo ' | ';
                    }
                    $number++;
                }
            }
        }

        // We will use this value to select initially the pages
        echo '<input id="dppp-edit-rule-initial-pages" type="hidden" value="' . esc_attr( $result['page_ids'] ) . '" />';

        $page_ids = explode( ',', $result['page_ids'] );

        echo '<div class="dppp-choose-elements" onscroll="dpppPositionFindSelectables()">';

        ?>
        <div class="dppp-find-selectables">
            <label>
                <span title="<?php esc_attr_e( 'Find', 'dppp' ); ?>" class="dashicons dashicons-search"></span>
                <input class="dppp-input-find-selectables" type="text" oninput="dpppFindSelectables()" />
                <a title="Clear" id="dppp-clear-find-selectables" class="dppp-display-none" href="javascript:dpppClearFindSelectables()">
                    <span class="dashicons dashicons-dismiss"></span>
                </a>
            </label>
        </div>
        <?php

        $args = array(
            'sort_order' => 'asc',
            'sort_column' => 'post_title',
            'hierarchical' => 1,
            'exclude' => '',
            'include' => '',
            'meta_key' => '',
            'meta_value' => '',
            'authors' => '',
            'child_of' => 0,
            'parent' => -1,
            'exclude_tree' => '',
            'number' => '',
            'offset' => 0,
            'post_type' => 'page',
            'post_status' => 'publish'
        );

        // If the WPML plugin is active, we show the pages in sections by language
        if ( is_plugin_active( 'sitepress-multilingual-cms/sitepress.php' ) ) {
            if ( ! empty( $languages ) ) {
                foreach ( $languages as $language ) {
                    $sitepress->switch_lang( $language['language_code'] );
                    echo '<div class="dppp-pages-language" id="dppp-pages-language-' . esc_attr( sanitize_html_class( $language['language_code'] ) ) . '">';
                    echo '<p>&nbsp;&nbsp; ' . esc_html( sprintf( __( 'WPML (%s)', 'dppp' ), $language['translated_name'] ) ) . '</p>';
                    $pages = get_pages( $args );
                    if ( is_array( $pages ) && count( $pages ) > 0 ) {
                        foreach ( $pages as $page ) {

                            $is_selected_page = in_array( $page->ID, $page_ids );

                            // We remove the page id from the array if we have found the page in the currently published pages
                            if ( true === $is_selected_page && ( $key = array_search( $page->ID, $page_ids ) ) !== false ) {
                                unset( $page_ids[ $key ] );
                            }

                            if ( true !== $is_selected_page ) {
                                echo '<p class="dppp-page-paragraphs" id="dppp-page-paragraph-' . esc_attr( $page->ID ) . '">';
                            } else {
                                echo '<p id="dppp-page-paragraph-' . esc_attr( $page->ID ) . '" class="dppp-selected-element dppp-page-paragraphs">';
                            }

                            echo '<label><input type="checkbox" class="dppp-page-checkbox" id="dppp-page-checkbox-' . esc_attr( $page->ID ) . '" '
                                . checked( $is_selected_page, true, false ) . ' /> ';
                            if ( $page->post_title === '' ) {
                                echo '<span class="dppp-page-title">';
                                esc_html_e( '(no title)', 'dppp' );
                                echo '</span></label></p>';
                            } else {
                                echo '<span class="dppp-page-title">' . esc_html( $page->post_title ) . '</span></label></p>';
                            }
                        }
                    } else {
                        echo '<p class="dppp-no-pages">' . esc_html__( 'There are no published pages at the moment.', 'dppp' ) . '</p>';
                    }
                    echo '</div>';
                }
            }

        // If the WPML plugin is not active, we just show all published pages
        } else {
            $pages = get_pages( $args );
            if ( is_array( $pages ) && count( $pages ) > 0 ) {
                foreach ( $pages as $page ) {
                    $is_selected_page = in_array( $page->ID, $page_ids );

                    // We remove the page id from the array if we have found the page in the currently published pages
                    if ( true === $is_selected_page && ( $key = array_search( $page->ID, $page_ids ) ) !== false ) {
                        unset( $page_ids[ $key ] );
                    }

                    if ( true !== $is_selected_page ) {
                        echo '<p class="dppp-page-paragraphs" id="dppp-page-paragraph-' . esc_attr( $page->ID ) . '">';
                    } else {
                        echo '<p id="dppp-page-paragraph-' . esc_attr( $page->ID ) . '" class="dppp-selected-element dppp-page-paragraphs">';
                    }

                    echo '<label><input type="checkbox" class="dppp-page-checkbox" id="dppp-page-checkbox-' . esc_attr( $page->ID ) . '" '
                        . checked( $is_selected_page, true, false ) . ' /> ';
                    if ( $page->post_title === '' ) {
                        echo '<span class="dppp-page-title">';
                        esc_html_e( '(no title)', 'dppp' );
                        echo '</span></label></p>';
                    } else {
                        echo '<span class="dppp-page-title">' . esc_html( $page->post_title ) . '</span></label></p>';
                    }
                }
            } else {
                echo '<p class="dppp-no-pages">' . esc_html__( 'There are no published pages at the moment.', 'dppp' ) . '</p>';
            }
        }

        // Now the $page_ids should contain only the selected pages that are not currently published (if there are any)
        if ( ! empty( $page_ids ) ) {
            if ( is_plugin_active( 'sitepress-multilingual-cms/sitepress.php' ) ) {
                echo '<p>&nbsp;&nbsp; ' . esc_html__( 'Currently not published pages (for all WPML languages)', 'dppp' ) . '</p>';
            } else {
                echo '<p>&nbsp;&nbsp; ' . esc_html__( 'Currently not published pages', 'dppp' ) . '</p>';
            }
            foreach ( $page_ids as $page_id ) {
                $status = get_post_status( $page_id );
                if ( 'pending' === $status ) {
                    $page_status = __( '[Pending]', 'dppp' );
                } elseif ( 'draft' === $status ) {
                    $page_status = __( '[Draft]', 'dppp' );
                } elseif ( 'future' === $status ) {
                    $page_status = __( '[Scheduled]', 'dppp' );
                } elseif ( 'private' === $status ) {
                    $page_status = __( '[Private]', 'dppp' );
                } elseif ( 'trash' === $status ) {
                    $page_status = __( '[Trashed]', 'dppp' );
                } elseif ( false === $status ) {
                    $page_status = __( '[Deleted]', 'dppp' );
                } else {
                    $page_status = __( '[Error]', 'dppp' );
                }
                echo '<p class="dppp-page-paragraphs dppp-selected-element" id="dppp-page-paragraph-' . esc_attr( $page_id ) . '">'
                    . '<label><input type="checkbox" class="dppp-page-checkbox" id="dppp-page-checkbox-' . esc_attr( $page_id ) . '" checked /> ';
                $page_title = get_the_title( $page_id );
                if ( empty( $page_title ) ) {
                    $page_title = sprintf( __( '[Page with ID %d]', 'dppp' ), intval( $page_id ) );
                }
                echo '<span class="dppp-page-title">' . esc_html( $page_title ) . ' ' . esc_html( $page_status ) . '</span></label></p>';
            }
        }
        echo '</div>';

        // If the WPML plugin is active, we show a checkbox option to also select translated pages of selected pages
        if ( is_plugin_active( 'sitepress-multilingual-cms/sitepress.php' ) ) {
            ?>
            <div class="dppp-select-wpml-translated-pages-contain">
                <label><input type="checkbox" id="dppp-select-wpml-translated-pages" name="dppp-select-wpml-translated-pages" />
                <?php esc_html_e( 'Also select all translated with WPML pages of selected pages', 'dppp' ); ?></label>
            </div>
            <?php
        }
        ?>
        <div class="dppp-under-pages-links">
            <div class="dppp-under-pages-links-left">
                <?php esc_html_e( 'Selected:', 'dppp' ); ?> <span></span>
            </div>
            <div class="dppp-under-pages-links-right">
                <a href="javascript:dpppSelectAllPages()"><?php esc_html_e( 'Select all', 'dppp' ); ?></a> /
                <a href="javascript:dpppDeselectAllPages()"><?php esc_html_e( 'Deselect all', 'dppp' ); ?></a>
            </div>
        </div>
        <?php
        echo '</div>';

    // It is a cutom URI type rule
    } elseif ( 'custom' === $result['uri_type'] ) {
        ?>
        <div id="dppp-custom-uri-type" class="dppp-display-block">
        <?php esc_html_e( '3. Custom URI condition', 'dppp' ); ?>
        <br>
        <?php
        $conditions = explode( '[dppp-or]', $result['uri_condition'] );
        $values = explode( '[dppp-or]', $result['uri_value'] );
        ?>
        <select name="dppp-custom-uri-select-condition" id="dppp-custom-uri-select-condition">
            <option value="equals"<?php selected( $conditions[0], 'equals' ); ?>>
                <?php esc_html_e( 'URI equals: ', 'dppp' ); ?>
            </option>
            <option value="does-not-equal"<?php selected( $conditions[0], 'does-not-equal' ); ?>>
                <?php esc_html_e( 'URI does not equal: ', 'dppp' ); ?>
            </option>
            <option value="contains"<?php selected( $conditions[0], 'contains' ); ?>>
                <?php esc_html_e( 'URI contains: ', 'dppp' ); ?>
            </option>
            <option value="does-not-contain"<?php selected( $conditions[0], 'does-not-contain' ); ?>>
                <?php esc_html_e( 'URI does not contain: ', 'dppp' ); ?>
            </option>
            <option value="starts-with"<?php selected( $conditions[0], 'starts-with' ); ?>>
                <?php esc_html_e( 'URI starts with: ', 'dppp' ); ?>
            </option>
            <option value="does-not-start-with"<?php selected( $conditions[0], 'does-not-start-with' ); ?>>
                <?php esc_html_e( 'URI does not start with: ', 'dppp' ); ?>
            </option>
            <option value="ends-with"<?php selected( $conditions[0], 'ends-with' ); ?>>
                <?php esc_html_e( 'URI ends with: ', 'dppp' ); ?>
            </option>
            <option value="does-not-end-with"<?php selected( $conditions[0], 'does-not-end-with' ); ?>>
                <?php esc_html_e( 'URI does not end with: ', 'dppp' ); ?>
            </option>
            <option value="regular-expression"<?php selected( $conditions[0], 'regular-expression' ); ?>>
                <?php esc_html_e( 'URI matches the regular expression: ', 'dppp' ); ?>
            </option>
        </select>
        <input type="text" name="dppp-custom-uri-value" id="dppp-custom-uri-value" value="<?php echo esc_attr( $values[0] ); ?>" />
        <?php

        // If there are more conditions we show them too
        if ( count( $conditions ) > 1 ) {
            ?>
            <div id="dppp-more-uri-conditions">
            <?php
            for ( $loop = 1; $loop < count( $conditions ); $loop++ ) {
            ?>
                <div id="dppp-<?php echo esc_attr( $loop + 1 ) ?>-uri-condition-container">
                    <div class="dppp-uri-conditions-inner">
                        <div class="dppp-uri-conditions-fields">
                            <select name="dppp-<?php echo esc_attr( $loop + 1 ) ?>-custom-uri-select-condition"
                                id="dppp-<?php echo esc_attr( $loop + 1 ) ?>-custom-uri-select-condition">
                                <option value="equals"<?php selected( $conditions[ $loop ], 'equals' ); ?>>
                                    <?php esc_html_e( 'or URI equals: ', 'dppp' ); ?>
                                </option>
                                <option value="does-not-equal"<?php selected( $conditions[ $loop ], 'does-not-equal' ); ?>>
                                    <?php esc_html_e( 'or URI does not equal: ', 'dppp' ); ?>
                                </option>
                                <option value="contains"<?php selected( $conditions[ $loop ], 'contains' ); ?>>
                                    <?php esc_html_e( 'or URI contains: ', 'dppp' ); ?>
                                </option>
                                <option value="does-not-contain"<?php selected( $conditions[ $loop ], 'does-not-contain' ); ?>>
                                    <?php esc_html_e( 'or URI does not contain: ', 'dppp' ); ?>
                                </option>
                                <option value="starts-with"<?php selected( $conditions[ $loop ], 'starts-with' ); ?>>
                                    <?php esc_html_e( 'or URI starts with: ', 'dppp' ); ?>
                                </option>
                                <option value="does-not-start-with"<?php selected( $conditions[ $loop ], 'does-not-start-with' ); ?>>
                                    <?php esc_html_e( 'or URI does not start with: ', 'dppp' ); ?>
                                </option>
                                <option value="ends-with"<?php selected( $conditions[ $loop ], 'ends-with' ); ?>>
                                    <?php esc_html_e( 'or URI ends with: ', 'dppp' ); ?>
                                </option>
                                <option value="does-not-end-with"<?php selected( $conditions[ $loop ], 'does-not-end-with' ); ?>>
                                    <?php esc_html_e( 'or URI does not end with: ', 'dppp' ); ?>
                                </option>
                                <option value="regular-expression"<?php selected( $conditions[ $loop ], 'regular-expression' ); ?>>
                                    <?php esc_html_e( 'or URI matches the regular expression: ', 'dppp' ); ?>
                                </option>
                            </select>
                            <input type="text" name="dppp-<?php echo esc_attr( $loop + 1 ) ?>-custom-uri-value"
                                id="dppp-<?php echo esc_attr( $loop + 1 ) ?>-custom-uri-value" value="<?php echo esc_attr( $values[ $loop ] ); ?>" />
                        </div>
                        <div class="dppp-contain-remove-condition">
                            <a title="<?php esc_html_e( 'Remove condition', 'dppp' ); ?>" class="dppp-remove-condition-link"
                                href="javascript:dpppRemoveCondition( '<?php echo esc_js( esc_attr( $loop + 1 ) ) ?>' )">
                                <span class="dashicons dashicons-no"></span>
                            </a>
                        </div>
                    </div>
                </div>
            <?php
            }
            ?>
            </div>
            <?php
        } else {
            ?>
            <div id="dppp-more-uri-conditions" class="dppp-display-none"></div>
            <?php
        }
        ?>
            <div id="dppp-uri-condition-template" class="dppp-display-none">
                <div id="dppp-template-uri-condition-container">
                    <div class="dppp-uri-conditions-inner">
                        <div class="dppp-uri-conditions-fields">
                            <select name="dppp-template-custom-uri-select-condition" id="dppp-template-custom-uri-select-condition">
                                <option value="equals" selected><?php esc_html_e( 'or URI equals: ', 'dppp' ); ?></option>
                                <option value="does-not-equal"><?php esc_html_e( 'or URI does not equal: ', 'dppp' ); ?></option>
                                <option value="contains"><?php esc_html_e( 'or URI contains: ', 'dppp' ); ?></option>
                                <option value="does-not-contain"><?php esc_html_e( 'or URI does not contain: ', 'dppp' ); ?></option>
                                <option value="starts-with"><?php esc_html_e( 'or URI starts with: ', 'dppp' ); ?></option>
                                <option value="does-not-start-with"><?php esc_html_e( 'or URI does not start with: ', 'dppp' ); ?></option>
                                <option value="ends-with"><?php esc_html_e( 'or URI ends with: ', 'dppp' ); ?></option>
                                <option value="does-not-end-with"><?php esc_html_e( 'or URI does not end with: ', 'dppp' ); ?></option>
                                <option value="regular-expression"><?php esc_html_e( 'or URI matches the regular expression: ', 'dppp' ); ?></option>
                            </select>
                            <input type="text" name="dppp-template-custom-uri-value" id="dppp-template-custom-uri-value" />
                        </div>
                        <div class="dppp-contain-remove-condition">
                            <a title="<?php esc_html_e( 'Remove condition', 'dppp' ); ?>" class="dppp-remove-condition-link"
                                href="javascript:dpppRemoveCondition( 'dppp-number-condition' )">
                                <span class="dashicons dashicons-no"></span>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
            <div class="dppp-align-right">
                <a id="dppp-add-more-conditions-link" href="javascript:void( 0 )"
                    onclick="dpppAddMoreConditions( <?php echo esc_js( esc_attr( count( $conditions ) + 1 ) ) ?> )">
                    <span class="dashicons dashicons-plus"></span><?php esc_html_e( 'Add more URI conditions', 'dppp' ); ?>
                </a>
            </div>
        </div>
    <?php
    } else {
        esc_html_e( 'Error', 'dppp' );
    }
    $visible_advanced = "no";
    if ( $result['devices'] !== 'all' || ( '' !== $result['note'] && null !== $result['note'] ) || ( '' !== $result['custom_function'] && null !== $result['custom_function'] ) ) {
        $visible_advanced = "yes";
    }
    if ( "no" === $visible_advanced ) {
        ?>
        <a id="dppp-show-advanced" href="javascript:dpppShowHidden('dppp-contain-advanced', 'dppp-show-advanced')"><?php esc_html_e( 'Show Advanced Settings', 'dppp' ); ?></a>
        <?php
    }
    ?>
    <div id="dppp-contain-advanced" class="<?php if ( "no" === $visible_advanced ) { echo 'dppp-display-none'; } ?>">
        <div class="dppp-contain-devices">
            <?php esc_html_e( '4. Affected devices', 'dppp' ); ?>
            <a id="dppp-read-more-devices"
                href="javascript:dpppShowHidden('dppp-notice-devices', 'dppp-read-more-devices')"><?php esc_html_e( 'Read more', 'dppp' ); ?></a>
            <br>
            <select name="dppp-select-devices" id="dppp-select-devices">
                <option value="all"<?php selected( $result['devices'], 'all' ); ?>><?php esc_html_e( 'All', 'dppp' ); ?></option>
                <option value="desktop"<?php selected( $result['devices'], 'desktop' ); ?>><?php esc_html_e( 'Desktop', 'dppp' ); ?></option>
                <option value="mobile"<?php selected( $result['devices'], 'mobile' ); ?>><?php esc_html_e( 'Mobile', 'dppp' ); ?></option>
            </select>
            <p id="dppp-notice-devices" class="dppp-display-none dppp-margin-top-5">
                <?php esc_html_e( 'Notice: If you are using a caching plugin, it needs to be able to generate different cache for different devices. '
                    . 'Otherwise the device of the first visitor to the page will determine the content of the cache and the deactivation rules for '
                    . 'that device would be applied to everyone. Also keep in mind that mobile includes tablets.', 'dppp' ); ?>
            </p>
        </div>
        <div class="dppp-contain-custom-function">
            <?php esc_html_e( '5. Custom condition function (for developers) (optional)', 'dppp' ); ?>
            <a id="dppp-read-more-custom-function"
                href="javascript:dpppShowHidden('dppp-hint-custom-function', 'dppp-read-more-custom-function')"><?php esc_html_e( 'Read more', 'dppp' ); ?></a>
            <input type="text" name="dppp-custom-condition-function" id="dppp-custom-condition-function"
                value="<?php if ( '' !== $result['custom_function'] && null !== $result['custom_function'] ) { echo esc_attr( $result['custom_function'] ); } ?>" />
            <p id="dppp-hint-custom-function" class="dppp-display-none dppp-margin-top-5">
                <?php printf( esc_html__( 'Add the name of your own PHP function (without brackets) for an additional condition check. The deactivation rule will only be applied if '
                    . 'your function returns true. This is an additional condition, so the regular conditions of the rule also have to be met. '
                    . 'Create your own %smust-use plugin file%s inside wp-content/mu-plugins to add your function, so it is available before the normal plugins are loaded. '
                    . 'Keep in mind that in a must-use plugin some WordPress functions will not work, because they are loaded at a later point of the loading process.', 'dppp' ),
                    '<a href="https://wordpress.org/support/article/must-use-plugins/" target="_blank">', '</a>' );
                ?>
            </p>
        </div>
        <div class="dppp-contain-add-note">
            <?php esc_html_e( '6. Note (optional)', 'dppp' ); ?>
            <?php
            if ( '' === $result['note'] || null === $result['note'] ) {
                ?>
                <textarea name="dppp-rule-note" id="dppp-rule-note"></textarea>
                <?php
            } else {
                ?>
                <textarea name="dppp-rule-note" id="dppp-rule-note"><?php echo esc_textarea( $result['note'] ); ?></textarea>
                <?php
            }
            ?>
        </div>
    </div>
    <div>
        <input type="button" id="dppp-button-save-rule"
            class="button button-primary dppp-full-width" value="<?php esc_attr_e( 'Save Rule', 'dppp' ); ?>" />
    </div>
    <?php
    wp_die();
}

// Handles the ajax request for saving the rule to the database after editing it
function dppp_save_edit_rule_data_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }
    $rule_id = intval( $_POST['rule_id'] );
    $deactivation_type = sanitize_html_class( $_POST['deactivation_type'] );
    $uri_type = sanitize_html_class( $_POST['uri_type'] );
    $custom_condition = sanitize_html_class( $_POST['custom_condition'] );
    if ( 'custom' === $uri_type ) {
        if ( 'regular-expression' === $custom_condition ) {
            $custom_value = stripslashes( $_POST['custom_value'] );
            if ( ! dppp_is_valid_regular_expression( $custom_value ) ) {
                wp_die( 'invalid-regex' );
            }
        } else {
            $custom_value = dppp_escape_uri( $_POST['custom_value'] );
            if ( $custom_value !== $_POST['custom_value'] ) {
                wp_die( 'invalid-data' );
            }
        }
        $more_custom_conditions = sanitize_html_class( $_POST['more_custom_conditions'] );
        if ( strpos( $more_custom_conditions, 'regular-expression' ) !== false ) {
            $more_custom_values = stripslashes( $_POST['more_custom_values'] );
        } else {
            $more_custom_values = dppp_escape_uri( $_POST['more_custom_values'] );
            if ( $more_custom_values !== $_POST['more_custom_values'] ) {
                wp_die( 'invalid-data' );
            }
        }
    }
    $devices = sanitize_html_class( $_POST['devices'] );
    $note = sanitize_textarea_field( stripslashes_deep( $_POST['note'] ) );
    $custom_function = sanitize_text_field( $_POST['custom_function'] );
    $select_translated_pages_wpml = sanitize_html_class( $_POST['select_translated_pages_wpml'] );
    $pages = dppp_sanitize_comma_separated_numbers( $_POST['pages'] );

    // If this is equal to 'dppp-group', then it is a group rule with a group hash in $_POST['hash']
    $plugin_file = dppp_sanitize_plugin_file( $_POST['plugin_file'] );

    if ( ! empty( $pages ) ) {
        $pages = substr_replace( $pages, "", -1 );
    }
    if ( ! in_array( $deactivation_type, Array( 'selected', 'except' ) ) || ! in_array( $uri_type, Array( 'page', 'custom' ) )
        || ! in_array( $devices, Array( 'all', 'desktop', 'mobile' ) )
        || ( ! empty( $pages ) && ! dppp_is_comma_separated_numbers( $pages ) )
        || ( ! dppp_is_valid_plugin_file( $plugin_file ) && 'dppp-group' !== $plugin_file )
        || dppp_strip_whitespace( $_POST['custom_function'] ) !== $_POST['custom_function']
        || ( ! empty( $_POST['custom_function'] ) && ! dppp_is_valid_php_function_name( $custom_function ) ) ) {
        wp_die( 'invalid-data' );
    }
    if ( 'custom' === $uri_type ) {
        if ( ! in_array( $custom_condition, Array( 'equals', 'does-not-equal', 'contains', 'does-not-contain', 'starts-with', 'does-not-start-with', 'ends-with',
            'does-not-end-with', 'regular-expression' ) ) || ! dppp_are_more_conditions_and_values_valid( $more_custom_conditions, $more_custom_values ) ) {
            wp_die( 'invalid-data' );
        }
        if ( ! dppp_are_more_conditions_and_values_regex_valid( $more_custom_conditions, $more_custom_values ) ) {
            wp_die( 'invalid-regex' );
        }
    }
    if ( 'page' === $uri_type && empty( $pages ) ) {
        wp_die( 'select-page' );
    }
    if ( 'custom' === $uri_type && empty( $custom_value ) ) {
        wp_die( 'enter-uri' );
    }

    // If we are adding a rule to a plugin group, we generate the plugin file string from the group hash
    if ( 'dppp-group' === $plugin_file ) {
        $group_hash = sanitize_html_class( $_POST['hash'] );
        $plugin_groups = dppp_get_option( 'dppp-plugin-groups' );
        if ( ! is_array( $plugin_groups ) || ! array_key_exists( $group_hash, $plugin_groups ) ) {
            wp_die( 'group-does-not-exist' );
        }
        $plugin_file = dppp_get_plugin_files_list_from_array( $plugin_groups[ $group_hash ]['plugins'] );
    }

    global $wpdb;

    // It is a custom rule
    if ( 'custom' === $uri_type ) {
        $more_custom_conditions = str_replace( '--dppp-or--', '[dppp-or]', $more_custom_conditions );
        $more_custom_values = str_replace( '--dppp-or--', '[dppp-or]', $more_custom_values );
        $custom_condition .= $more_custom_conditions;
        $custom_value .= $more_custom_values;
        $status = $wpdb->query( $wpdb->prepare( "SELECT rule_id FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE rule_id != %d AND plugin_file = %s "
            . "AND deactivation_type = %s AND uri_type = 'custom' AND page_ids = '0' AND uri_condition = %s AND uri_value = %s AND devices = %s",
            $rule_id, $plugin_file, $deactivation_type, $custom_condition, $custom_value, $devices ) );
        if ( $status === false ) {
            wp_die( 'could-not-check-if-exists' );
        }
        if ( $status !== 0 ) {
            wp_die( 'already-exists' );
        }
        if ( dppp_is_valid_url( $custom_value ) ) {
            wp_die( 'uri-not-url' );
        }
        $status_update = $wpdb->update(
            "{$wpdb->prefix}dppp_plugin_deactivation_rules",
            array(
        		'deactivation_type' => $deactivation_type,
        		'uri_condition' => $custom_condition,
        		'uri_value' => $custom_value,
        		'devices' => $devices,
        		'custom_function' => $custom_function,
                'note' => $note,
            ),
            array(
                'rule_id' => $rule_id,
            ),
            array( '%s', '%s', '%s', '%s', '%s', '%s' ),
            array( '%d' )
        );
        if ( $status_update === false ) {
            wp_die( 'could-not-edit' );
        }

    // It is a page rule
    } else {
        $page_uris = '';
        $page_ids = explode( ',', $pages );

        // If the WPML plugin is active, and the option to also select translated pages of selected pages was checked, we do that
        if ( is_plugin_active( 'sitepress-multilingual-cms/sitepress.php' ) && 'yes' === $select_translated_pages_wpml ) {
            global $sitepress;
            $add_page_ids = Array();
            foreach ( $page_ids as $page_id ) {
                $translation_group_id = apply_filters( 'wpml_element_trid', NULL, $page_id, 'post_page' );
                $translations = apply_filters( 'wpml_get_element_translations', NULL, $translation_group_id, 'post_page' );
                foreach ( $translations as $translation ) {
                    $add_page_ids[] = intval( $translation->element_id );
                }
            }
            foreach ( $add_page_ids as $add_page_id ) {
                if ( ! in_array( $add_page_id, $page_ids ) ) {
                    $page_ids[] = $add_page_id;
                }
            }
        }

        // We will delete the pages that are permanently deleted
        $to_unset = Array();
        foreach ( $page_ids as $key => $page_id ) {
            if ( get_post_status( $page_id ) === false ) {
                $to_unset[] = $key;
            }
        }
        foreach ( $to_unset as $unset_key ) {
            unset( $page_ids[ $unset_key ] );
        }

        // We sort the array and set the $pages var so the order is the same always and we can properly check if the rule exists
        sort( $page_ids );
        $pages = implode( ',', $page_ids );

        $loop = 1;
        foreach ( $page_ids as $page_id ) {
            $page_uri = dppp_escape_uri( dppp_get_page_uri_or_uris( $page_id ) );
            $page_uris .= $page_uri;
            if ( $loop !== count( $page_ids ) ) {
                $page_uris .= '[dppp-or]';
            }
            $loop++;
        }
        $status = $wpdb->query( $wpdb->prepare( "SELECT rule_id FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE rule_id != %d AND plugin_file = %s AND "
            . "deactivation_type = %s AND uri_type = 'page' AND page_ids = %s AND uri_condition = 'equals' AND uri_value = %s AND devices = %s",
            $rule_id, $plugin_file, $deactivation_type, $pages, $page_uris, $devices ) );
        if ( $status !== 0 ) {
            wp_die( 'already-exists' );
        }
        $status_update = $wpdb->update(
            "{$wpdb->prefix}dppp_plugin_deactivation_rules",
            array(
        		'deactivation_type' => $deactivation_type,
        		'page_ids' => $pages,
        		'uri_value' => $page_uris,
        		'devices' => $devices,
        		'custom_function' => $custom_function,
                'note' => $note,
            ),
            array(
                'rule_id' => $rule_id,
            ),
            array( '%s', '%s', '%s', '%s', '%s', '%s' ),
            array( '%d' )
        );
        if ( $status_update === false ) {
            wp_die( 'could-not-edit' );
        }
    }
    echo '[done]';
    wp_die();
}

// Handles the ajax request for editing a plugin group
function dppp_edit_plugin_group_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( '<h2>' . esc_html__( 'Edit Plugin Group', 'dppp' ) . '</h2>'
            . '<input type="button" class="dppp-close-button button" id="dppp-button-close-edit" value="' . esc_attr__( 'Close', 'dppp' ) . '" />'
            . '<p class="dppp-clear">' . esc_html__( 'Error: Invalid security nonce.', 'dppp' ) . '</p>' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( '<h2>' . esc_html__( 'Edit Plugin Group', 'dppp' ) . '</h2>'
            . '<input type="button" class="dppp-close-button button" id="dppp-button-close-edit" value="' . esc_attr__( 'Close', 'dppp' ) . '" />'
            . '<p class="dppp-clear">' . esc_html__( 'Error: You do not have sufficient permissions to make this request.', 'dppp' ) . '</p>' );
    }
    $group_hash = sanitize_html_class( $_POST['group_hash'] );
    $plugin_groups = dppp_get_option( 'dppp-plugin-groups' );
    if ( ( ! is_array( $plugin_groups ) && false === $plugin_groups ) || ! array_key_exists( $group_hash, $plugin_groups ) ) {
        wp_die( '<h2>' . esc_html__( 'Edit Plugin Group', 'dppp' ) . '</h2>'
            . '<input type="button" class="dppp-close-button button" id="dppp-button-close-edit" value="' . esc_attr__( 'Close', 'dppp' ) . '" />'
            . '<p class="dppp-clear">' . esc_html__( 'Error: The plugin group does not exist.', 'dppp' ) . '</p>' );
    }
    ?>
    <h2>
        <?php esc_html_e( 'Edit Plugin Group', 'dppp' ) ?>
    </h2>
    <input type="button" class="dppp-close-button button" id="dppp-button-close-edit" value="<?php esc_attr_e( 'Close', 'dppp' ); ?>" />
    <div class="dppp-clear">
        <?php esc_html_e( '1. Enter a name', 'dppp' ); ?><br>
        <input type="text" name="dppp-plugin-group-name" id="dppp-plugin-group-name"
            value="<?php echo esc_attr( stripslashes( $plugin_groups[ $group_hash ]['name'] ) ) ?>" />
    </div>
    <div id="dppp-select-plugins">
        <?php esc_html_e( '2. Select plugins', 'dppp' ); ?>
        (<?php esc_html_e( 'Active', 'dppp' ); ?> / <span class="dppp-red"><?php esc_html_e( 'Inactive', 'dppp' ); ?></span>)
        <div class="dppp-choose-elements">
            <?php
            $plugins = get_plugins();
            $plugin_hashes = '';
            foreach ( $plugins as $plugin_file => $plugin_data ) {
                if ( 'deactivate-plugins-per-page/deactivate-plugins-per-page.php' === $plugin_file ) {
                    continue;
                }
                $selected_plugin = 'no';
                $add_class = '';
                $plugin_hash = md5( $plugin_file );
                if ( in_array( $plugin_file, $plugin_groups[ $group_hash ]['plugins'] ) ) {
                    $selected_plugin = 'yes';
                    $add_class = 'dppp-selected-element';
                    $plugin_hashes .= $plugin_hash . '_';

                    // We remove the plugin from the array since we have found the plugin in the currently installed plugins earlier
                    $key = array_search( $plugin_file, $plugin_groups[ $group_hash ]['plugins'] );
                    unset( $plugin_groups[ $group_hash ]['plugins'][ $key ] );
                    unset( $plugin_groups[ $group_hash ]['plugin-names'][ $key ] );
                }
                echo '<p id="dppp-plugin-paragraph-' . esc_attr( $plugin_hash ) . '" class="' . esc_attr( $add_class )
                    . '"><label><span class="dppp-plugin-span-checkbox"><input type="checkbox" class="dppp-plugin-checkbox" '
                    . 'id="dppp-plugin-checkbox-' . esc_attr( $plugin_hash ) . '" ' . checked( $selected_plugin, 'yes', false ) . ' /> ';
                if ( is_plugin_active( $plugin_file ) ) {
                    echo esc_html( $plugin_data['Name'] );
                } else {
                    echo '<span class="dppp-red">' . esc_html( $plugin_data['Name'] ) . '</span>';
                }
                echo '</span>';

                // We show an icon if the plugin is in any group
                if ( 'yes' === $selected_plugin || dppp_plugin_in_any_group( $plugin_file, $plugin_groups ) ) {
                    echo '<span title="' . esc_attr__( 'The plugin is part of at least one group', 'dppp' )
                        . '" class="dashicons dashicons-screenoptions"></span>';
                }

                echo '</label></p>';
            }

            // If there are any plugins left, these are not installed plugins right now
            if ( count( $plugin_groups[ $group_hash ]['plugins'] ) > 0 ) {
                echo '<p>&nbsp;&nbsp; ' . esc_html__( 'Currently not installed plugins', 'dppp' ) . '</p>';
                foreach ( $plugin_groups[ $group_hash ]['plugins'] as $key => $plugin_file ) {
                    $plugin_hash = md5( $plugin_file );
                    $plugin_hashes .= $plugin_hash . '_';
                    echo '<p id="dppp-plugin-paragraph-' . esc_attr( $plugin_hash )
                        . '" class="dppp-selected-element"><label><span class="dppp-plugin-span-checkbox"><input type="checkbox" class="dppp-plugin-checkbox" '
                        . 'id="dppp-plugin-checkbox-' . esc_attr( $plugin_hash ) . '" checked /> '
                        . esc_html( $plugin_groups[ $group_hash ]['plugin-names'][ $key ] ) . '</span>';

                    // We show an icon since the plugin is in any group
                    echo '<span title="' . esc_attr__( 'The plugin is part of at least one group', 'dppp' )
                        . '" class="dashicons dashicons-screenoptions"></span>';

                    echo '</label></p>';
                }
            }
            ?>
        </div>
        <div class="dppp-under-plugins-links">
            <div class="dppp-under-plugins-links-left">
                <?php esc_html_e( 'Selected:', 'dppp' ); ?> <span></span>
            </div>
            <div class="dppp-under-plugins-links-right">
                <a href="javascript:dpppSelectAllPlugins()"><?php esc_html_e( 'Select all', 'dppp' ); ?></a> /
                <a href="javascript:dpppDeselectAllPlugins()"><?php esc_html_e( 'Deselect all', 'dppp' ); ?></a>
            </div>
        </div>
    </div>
    <input type="hidden" id="dppp-edit-group-initial-plugin-hashes" value="<?php echo esc_attr( $plugin_hashes ) ?>" />
    <div>
        <input type="button" id="dppp-button-save-selected-plugins"
            class="button button-primary dppp-full-width" value="<?php esc_attr_e( 'Save Plugin Group', 'dppp' ); ?>" />
    </div>
    <?php
    wp_die();
}

// Handles the ajax request for saving the group data after editing a plugin group
function dppp_save_edit_plugin_group_data_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }
    $plugin_hashes = sanitize_html_class( $_POST['plugins'] );
    $group_name = sanitize_text_field( $_POST['group_name'] );
    $group_hash = sanitize_html_class( $_POST['group_hash'] );
    if ( empty( $plugin_hashes ) || empty( $group_name ) || empty( $group_hash ) ) {
        wp_die( 'invalid-data' );
    }

    // We turn the hashes into an array of plugin files for the plugins that are installed
    $plugins = get_plugins();
    $selected_plugins = Array();
    foreach ( $plugins as $plugin_file => $plugin_data ) {
        $plugin_hash = md5( $plugin_file );
        if ( strpos( $plugin_hashes, $plugin_hash ) !== false ) {
            $selected_plugins[] = $plugin_file;
        }
    }
    $plugin_groups = dppp_get_option( 'dppp-plugin-groups' );

    // We also add to the selected plugins the ones that are deleted, but we have their data in the group data
    foreach ( $plugin_groups[ $group_hash ]['plugins'] as $plugin_file ) {
        $plugin_hash = md5( $plugin_file );
        if ( strpos( $plugin_hashes, $plugin_hash ) !== false && ! in_array( $plugin_file, $selected_plugins ) ) {
            $selected_plugins[] = $plugin_file;
        }
    }

    if ( count( $selected_plugins ) < 2 ) {
        wp_die( 'at-least-two-plugins' );
    }

    // We add the selected plugins as a new group in a multi-multi-array option along with the other groups already there
    if ( ! empty( $selected_plugins ) ) {
        sort( $selected_plugins );

        // If the selected plugins are the same as before, we just update the group name and we are done
        if ( $selected_plugins === $plugin_groups[ $group_hash ]['plugins'] ) {
            $plugin_groups[ $group_hash ]['name'] = $group_name;
            update_option( 'dppp-plugin-groups', $plugin_groups );
            wp_die( '[done]' );
        }

        $plugin_names = Array();
        foreach ( $selected_plugins as $group_plugin_file ) {

            // If the plugin is installed, we get the name from there, if it is not, from our previous data for the groups
            if ( array_key_exists( $group_plugin_file, $plugins ) ) {
                $plugin_data = get_plugin_data( WP_PLUGIN_DIR . '/' . $group_plugin_file );
                $plugin_names[] = $plugin_data['Name'];
            } else {
                $key = array_search( $group_plugin_file, $plugin_groups[ $group_hash ]['plugins'] );
                $plugin_names[] = $plugin_groups[ $group_hash ]['plugin-names'][ $key ];
            }
        }
        $new_group_hash = dppp_get_plugin_group_hash_from_array( $selected_plugins );
        if ( array_key_exists( $new_group_hash, $plugin_groups ) ) {
            wp_die( 'already-exists-group' );
        }

        // We change the group hash without changing the order of the groups, and then we change its value
        $new_plugin_groups = dppp_replace_array_key_same_order( $plugin_groups, $group_hash, $new_group_hash );
        $new_plugin_groups[ $new_group_hash ] = Array( 'name' => $group_name, 'plugins' => $selected_plugins, 'plugin-names' => $plugin_names );

        $old_plugins_list = dppp_get_plugin_files_list_from_array( $plugin_groups[ $group_hash ]['plugins'] );
        $new_plugins_list = dppp_get_plugin_files_list_from_array( $selected_plugins );

        // We update the rules for that group to use the new plugin list
        global $wpdb;
        $status_update = $wpdb->update(
            "{$wpdb->prefix}dppp_plugin_deactivation_rules",
            array(
        		'plugin_file' => $new_plugins_list,
            ),
            array(
                'plugin_file' => $old_plugins_list,
            ),
            array( '%s' ),
            array( '%s' )
        );
        if ( $status_update === false ) {
            wp_die( 'could-not-edit-group' );
        }

        // We finally update the plugin groups option
        update_option( 'dppp-plugin-groups', $new_plugin_groups );
    }
    wp_die( '[done]' );
}

// Handles the ajax request for plugin actions (delete rules, pause rules, start rules, deactivate on front-end, deactivate on back-end)
function dppp_plugin_or_group_actions_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }
    $plugin_action = sanitize_html_class( $_POST['plugin_action'] );
    $plugin_or_group = sanitize_html_class( $_POST['plugin_or_group'] );

    if ( 'group' === $plugin_or_group ) {
        $group_hash = sanitize_html_class( $_POST['group_hash'] );
        $plugin_groups = dppp_get_option( 'dppp-plugin-groups' );
        if ( ( ! is_array( $plugin_groups ) && false === $plugin_groups ) || ! array_key_exists( $group_hash, $plugin_groups ) ) {
            wp_die( 'group-does-not-exist' );
        } else {
            $plugin_file = dppp_get_plugin_files_list_from_array( $plugin_groups[ $group_hash ]['plugins'] );
        }
    } else {
        $plugin_file = dppp_sanitize_plugin_file( $_POST['plugin_file'] );
    }

    global $wpdb;
    if ( 'delete-rules' === $plugin_action ) {
        $wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE plugin_file = %s "
            . "AND uri_type != 'global'", $plugin_file ) );
    } elseif ( 'pause-rules' === $plugin_action ) {
        $wpdb->query( $wpdb->prepare( "UPDATE {$wpdb->prefix}dppp_plugin_deactivation_rules SET status = 'paused' WHERE plugin_file = %s "
            . "AND uri_type != 'global'", $plugin_file ) );
    } elseif ( 'start-rules' === $plugin_action ) {
        $wpdb->query( $wpdb->prepare( "UPDATE {$wpdb->prefix}dppp_plugin_deactivation_rules SET status = 'started' WHERE plugin_file = %s "
            . "AND uri_type != 'global'", $plugin_file ) );
    } elseif ( 'front-end-deactivate' === $plugin_action ) {
        $result = $wpdb->get_var( $wpdb->prepare( "SELECT rule_id FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE plugin_file = %s "
            . "AND uri_type = 'global' AND uri_value = '[dppp-front-end]'", $plugin_file ) );
        if ( empty( $result ) ) {
            $wpdb->insert(
            	"{$wpdb->prefix}dppp_plugin_deactivation_rules",
            	array(
            		'status' => 'started',
            		'plugin_file' => $plugin_file,
            		'deactivation_type' => 'selected',
            		'uri_type' => 'global',
            		'page_ids' => '0',
            		'uri_condition' => 'global',
            		'uri_value' => '[dppp-front-end]',
            		'devices' => 'all',
            		'unix_time_added' => time(),
            	),
            	array(
            		'%s',
            		'%s',
            		'%s',
            		'%s',
            		'%s',
            		'%s',
            		'%s',
            		'%s',
            		'%d',
            	)
            );
        } else {
            $wpdb->query( $wpdb->prepare( "UPDATE {$wpdb->prefix}dppp_plugin_deactivation_rules SET status = 'started' WHERE plugin_file = %s AND "
                . "uri_type = 'global' AND uri_value = '[dppp-front-end]'", $plugin_file ) );
        }
    } elseif ( 'front-end-activate' === $plugin_action ) {
        $result = $wpdb->get_var( $wpdb->prepare( "SELECT rule_id FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE plugin_file = %s "
            . "AND uri_type = 'global' AND uri_value = '[dppp-front-end]'", $plugin_file ) );
        if ( ! empty( $result ) ) {
            $wpdb->query( $wpdb->prepare( "UPDATE {$wpdb->prefix}dppp_plugin_deactivation_rules SET status = 'paused' WHERE plugin_file = %s AND "
                . "uri_type = 'global' AND uri_value = '[dppp-front-end]'", $plugin_file ) );
        }
    } elseif ( 'back-end-deactivate' === $plugin_action ) {
        $result = $wpdb->get_var( $wpdb->prepare( "SELECT rule_id FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE plugin_file = %s "
            . "AND uri_type = 'global' AND uri_value = '[dppp-back-end]'", $plugin_file ) );
        if ( empty( $result ) ) {
            $wpdb->insert(
            	"{$wpdb->prefix}dppp_plugin_deactivation_rules",
            	array(
            		'status' => 'started',
            		'plugin_file' => $plugin_file,
            		'deactivation_type' => 'selected',
            		'uri_type' => 'global',
            		'page_ids' => '0',
            		'uri_condition' => 'global',
            		'uri_value' => '[dppp-back-end]',
            		'devices' => 'all',
            		'unix_time_added' => time(),
            	),
            	array(
            		'%s',
            		'%s',
            		'%s',
            		'%s',
            		'%s',
            		'%s',
            		'%s',
            		'%s',
            		'%d',
            	)
            );
        } else {
            $wpdb->query( $wpdb->prepare( "UPDATE {$wpdb->prefix}dppp_plugin_deactivation_rules SET status = 'started' WHERE plugin_file = %s AND "
                . "uri_type = 'global' AND uri_value = '[dppp-back-end]'", $plugin_file ) );
        }
    } elseif ( 'back-end-activate' === $plugin_action ) {
        $result = $wpdb->get_var( $wpdb->prepare( "SELECT rule_id FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE plugin_file = %s "
            . "AND uri_type = 'global' AND uri_value = '[dppp-back-end]'", $plugin_file ) );
        if ( ! empty( $result ) ) {
            $wpdb->query( $wpdb->prepare( "UPDATE {$wpdb->prefix}dppp_plugin_deactivation_rules SET status = 'paused' WHERE plugin_file = %s AND "
                . "uri_type = 'global' AND uri_value = '[dppp-back-end]'", $plugin_file ) );
        }
    } else {
        wp_die( 'invalid-data' );
    }
    wp_die( '[done]' );
}

/**
 * Returns the global status of a plugin or group of plugins.
 * @param string $plugin_file
 * @return string
 */
function dppp_get_plugin_or_group_global_status( $plugin_file ) {
    global $wpdb;
    $back_end_deactivated = $front_end_deactivated = 'no';
    $results_multi_array = $wpdb->get_results( $wpdb->prepare( "SELECT uri_value FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE plugin_file = %s "
        . "AND uri_type = 'global' AND status = 'started'", $plugin_file ), ARRAY_A );
    if ( ! empty( $results_multi_array ) ) {
        foreach ( $results_multi_array as $results ) {
            if ( '[dppp-back-end]' === $results['uri_value'] ) {
                $back_end_deactivated = 'yes';
            }
            if ( '[dppp-front-end]' === $results['uri_value'] ) {
                $front_end_deactivated = 'yes';
            }
        }
        if ( 'yes' === $back_end_deactivated && 'yes' === $front_end_deactivated ) {
            return 'deactivated';
        }
        if ( 'yes' === $back_end_deactivated && 'no' === $front_end_deactivated ) {
            return 'back-end-deactivated';
        }
        if ( 'no' === $back_end_deactivated && 'yes' === $front_end_deactivated ) {
            return 'front-end-deactivated';
        }
    } else {
        return 'activated';
    }
}

/**
 * Returns a new array with one of its keys replaced while keeping the same order of elements
 * @param array $array
 * @param mixed $old_key
 * @param mixed $new_key
 * @return array
 */
function dppp_replace_array_key_same_order( $array, $old_key, $new_key ) {
    if ( array_key_exists( $old_key, $array ) ) {
        $all_old_keys = array_keys( $array );
        $key_of_key = array_search( $old_key, $all_old_keys );
        $all_old_keys[ $key_of_key ] = $new_key;
        return array_combine( $all_old_keys, $array );
    }
    return $array;
}

/**
 * Display an html select form element to use in a plugin settings page. It is automatically set to the current setting value
 * @param string $name
 * @param array $option_values
 * @param mixed $option_names
 */
function dppp_setting_select( $name, $option_values, $option_names ) {

    // Based on the $option_names argument we could use the option values as option names
    if ( ! is_array( $option_names ) && 'same-as-values' == $option_names ) {
        $option_names = $option_values;
    }

    $current_db_value = dppp_get_option( $name );
    echo '<select autocomplete="off" id="' . esc_attr( $name ) . '" name="' . esc_attr( $name ) . '" size="1">';
    for ( $i = 0; $i < count( $option_values ); $i++ ) {
        echo '<option value="' . esc_attr( $option_values[ $i ] ) . '" ' . selected( $option_values[ $i ], $current_db_value, false ) . ' >'
            . esc_html( $option_names[ $i ] ) . '</option>';
    }
    echo '</select>';
}

/**
 * Display an html textarea form element to use in a plugin settings page. It is automatically filled with the current setting value.
 * @param string $name
 */
function dppp_setting_textarea( $name ) {

    // Get the current value of the setting in the database
    $current_db_value = dppp_get_option( $name );

    // Output the textbox with the current value
    echo '<textarea autocomplete="off" id="' . esc_attr( $name ) . '" name="' . esc_attr( $name ) . '">' . esc_textarea( $current_db_value ) . '</textarea>';
}

/**
 * Returns true if the plugin is in any plugin group
 * @param string $plugin_file
 * @param mixed $plugin_groups
 * @return bool
 */
function dppp_plugin_in_any_group( $plugin_file, $plugin_groups ) {
    if ( is_array( $plugin_groups ) ) {
        foreach ( $plugin_groups as $plugin_group ) {
            if ( in_array( $plugin_file, $plugin_group['plugins'] ) ) {
                return true;
            }
        }
    }
    return false;
}

// Handles the ajax request for showing and hiding a section
function dppp_show_hide_section_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }
    $section = sanitize_html_class( $_POST['section'] );
    if ( dppp_get_option( 'dppp-section-show-hide-' . $section ) === 'hide' ) {
        update_option( 'dppp-section-show-hide-' . $section, 'show' );
    } else {
        update_option( 'dppp-section-show-hide-' . $section, 'hide' );
    }
    wp_die( 'done' );
}

// Handles the ajax request for performing quick plugin actions from the debug box
function dppp_debug_quick_action_callback() {
    if ( ! check_ajax_referer( 'dppp_debug_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }
    global $wpdb;
    $quick_action = sanitize_html_class( $_POST['quick_action'] );
    $plugin_file = dppp_sanitize_plugin_file( $_POST['plugin_file'] );
    $uri = dppp_escape_uri( $_POST['uri'] );
    $page_id = intval( $_POST['page_id'] );
    $group_hash = sanitize_html_class( $_POST['group_hash'] );

    if ( 'deactivate-current' === $quick_action || 'deactivate-except-current' === $quick_action ) {

        if ( 'deactivate-current' === $quick_action ) {
            $deactivation_type = 'selected';
        } else {
            $deactivation_type = 'except';
        }

        // It is a page rule (since it is a page and the URI matches the original page URI)
        if ( $page_id > 0 && dppp_get_original_page_uri( $page_id ) === $uri ) {
            $page_uri = dppp_escape_uri( dppp_get_page_uri_or_uris( $page_id ) );
            $status = $wpdb->query( $wpdb->prepare( "SELECT rule_id FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE plugin_file = %s AND "
                . "deactivation_type = %s AND uri_type = 'page' AND page_ids = %s AND uri_condition = 'equals' AND uri_value = %s "
                . "AND devices = 'all' AND status = 'started'", $plugin_file, $deactivation_type, $page_id, $page_uri ) );
            if ( $status !== 0 ) {
                wp_die( 'already-exists' );
            }
            $rule_id = $wpdb->get_var( $wpdb->prepare( "SELECT rule_id FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE plugin_file = %s AND "
                . "deactivation_type = %s AND uri_type = 'page' AND page_ids = %s AND uri_condition = 'equals' AND uri_value = %s "
                . "AND devices = 'all' AND status = 'paused'", $plugin_file, $deactivation_type, $page_id, $page_uri ) );

            // If the rule exists but it is paused, we start it
            if ( ! empty( $rule_id ) ) {
                $wpdb->update(
                    "{$wpdb->prefix}dppp_plugin_deactivation_rules",
                    array(
                        'status' => 'started',
                    ),
                    array(
                        'rule_id' => $rule_id,
                    ),
                    array( '%s' ),
                    array( '%d' )
                );

            // If the rule does not exist we create it
            } else {
                $status_insert = $wpdb->insert(
                	"{$wpdb->prefix}dppp_plugin_deactivation_rules",
                	array(
                		'status' => 'started',
                		'plugin_file' => $plugin_file,
                		'deactivation_type' => $deactivation_type,
                		'uri_type' => 'page',
                		'page_ids' => $page_id,
                		'uri_condition' => 'equals',
                		'uri_value' => $page_uri,
                		'devices' => 'all',
                		'unix_time_added' => time(),
                	),
                	array(
                		'%s',
                		'%s',
                		'%s',
                		'%s',
                		'%s',
                		'%s',
                		'%s',
                		'%s',
                		'%d',
                	)
                );
                if ( $status_insert === false ) {
                    wp_die( 'could-not-insert' );
                }
            }

            wp_die( '[done]' );

        // It is a custom rule
        } else {
            $status = $wpdb->query( $wpdb->prepare( "SELECT rule_id FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE plugin_file = %s AND "
                . "deactivation_type = %s AND uri_type = 'custom' AND uri_condition = 'equals' AND uri_value = %s "
                . "AND devices = 'all' AND status = 'started'", $plugin_file, $deactivation_type, $uri ) );
            if ( $status !== 0 ) {
                wp_die( 'already-exists' );
            }
            $rule_id = $wpdb->get_var( $wpdb->prepare( "SELECT rule_id FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE plugin_file = %s AND "
                . "deactivation_type = %s AND uri_type = 'custom' AND uri_condition = 'equals' AND uri_value = %s "
                . "AND devices = 'all' AND status = 'paused'", $plugin_file, $deactivation_type, $uri ) );

            // If the rule exists but it is paused, we start it
            if ( ! empty( $rule_id ) ) {
                $wpdb->update(
                    "{$wpdb->prefix}dppp_plugin_deactivation_rules",
                    array(
                        'status' => 'started',
                    ),
                    array(
                        'rule_id' => $rule_id,
                    ),
                    array( '%s' ),
                    array( '%d' )
                );

            // If the rule does not exist we create it
            } else {
                $status_insert = $wpdb->insert(
                	"{$wpdb->prefix}dppp_plugin_deactivation_rules",
                	array(
                		'status' => 'started',
                		'plugin_file' => $plugin_file,
                		'deactivation_type' => $deactivation_type,
                		'uri_type' => 'custom',
                		'page_ids' => '0',
                		'uri_condition' => 'equals',
                		'uri_value' => $uri,
                		'devices' => 'all',
                		'unix_time_added' => time(),
                	),
                	array(
                		'%s',
                		'%s',
                		'%s',
                		'%s',
                		'%s',
                		'%s',
                		'%s',
                		'%s',
                		'%d',
                	)
                );
                if ( $status_insert === false ) {
                    wp_die( 'could-not-insert' );
                }
            }
            wp_die('[done]');
        }
    }

    if ( 'add-to-group' === $quick_action ) {

        // We add to the group plugins the new plugin
        $group_plugins = Array( $plugin_file );

        $plugin_groups = dppp_get_option( 'dppp-plugin-groups' );

        // Show an error if there are no groups
        if ( ! is_array( $plugin_groups ) || empty( $plugin_groups ) ) {
            wp_die( 'no-groups' );
        }

        // Show an error if the chosen group does not exist
        if ( ! array_key_exists( $group_hash, $plugin_groups ) ) {
            wp_die( 'group-not-exist' );
        }

        // We also add to the group plugins the existing ones in the group
        foreach ( $plugin_groups[ $group_hash ]['plugins'] as $plugin_file ) {
            if ( ! in_array( $plugin_file, $group_plugins ) ) {
                $group_plugins[] = $plugin_file;
            } else {

                // Show an error if the plugin is already in this group
                wp_die( 'already-in-group' );
            }
        }

        // Show an error if somehow there are less than 2 plugins in the group after we add the new one
        if ( count( $group_plugins ) < 2 ) {
            wp_die( 'at-least-two-plugins' );
        }

        // We add the group plugins as a new group in a multi-multi-array option along with the other groups already there
        if ( ! empty( $group_plugins ) ) {
            sort( $group_plugins );

            // Check if get_plugins() function exists. This is required on the front end, since it is in a file that is normally only loaded in the admin.
            if ( ! function_exists( 'get_plugins' ) ) {
                require_once ABSPATH . 'wp-admin/includes/plugin.php';
            }

            $plugins = get_plugins();

            $plugin_names = Array();
            foreach ( $group_plugins as $group_plugin_file ) {

                // If the plugin is installed, we get the name from there, if it is not, from our previous data for the groups
                if ( array_key_exists( $group_plugin_file, $plugins ) ) {
                    $plugin_data = get_plugin_data( WP_PLUGIN_DIR . '/' . $group_plugin_file );
                    $plugin_names[] = $plugin_data['Name'];
                } else {
                    $key = array_search( $group_plugin_file, $plugin_groups[ $group_hash ]['plugins'] );
                    $plugin_names[] = $plugin_groups[ $group_hash ]['plugin-names'][ $key ];
                }
            }
            $new_group_hash = dppp_get_plugin_group_hash_from_array( $group_plugins );
            if ( array_key_exists( $new_group_hash, $plugin_groups ) ) {

                // The resulted group already exists
                wp_die( 'already-exists-group' );
            }

            // We change the group hash without changing the order of the groups, and then we change its value
            $new_plugin_groups = dppp_replace_array_key_same_order( $plugin_groups, $group_hash, $new_group_hash );
            $new_plugin_groups[ $new_group_hash ] = Array(
                'name' => $plugin_groups[ $group_hash ]['name'],
                'plugins' => $group_plugins,
                'plugin-names' => $plugin_names
            );

            $old_plugins_list = dppp_get_plugin_files_list_from_array( $plugin_groups[ $group_hash ]['plugins'] );
            $new_plugins_list = dppp_get_plugin_files_list_from_array( $group_plugins );

            // We update the rules for that group to use the new plugin list
            $status_update = $wpdb->update(
                "{$wpdb->prefix}dppp_plugin_deactivation_rules",
                array(
            		'plugin_file' => $new_plugins_list,
                ),
                array(
                    'plugin_file' => $old_plugins_list,
                ),
                array( '%s' ),
                array( '%s' )
            );
            if ( $status_update === false ) {
                wp_die( 'could-not-edit-group' );
            }

            // We finally update the plugin groups option
            update_option( 'dppp-plugin-groups', $new_plugin_groups );
        }
        wp_die( '[done]' . esc_html( $new_group_hash ) );

    }
    wp_die();
}

/**
 * Checks if the string is a URI parameter name.
 * @param string $string
 * @return bool
 */
function dppp_is_valid_uri_parameter_name( $parameter ) {
    if ( dppp_strip_whitespace( $parameter ) !== $parameter || ! dppp_is_valid_url( 'http://example.com/page.php?' . $parameter . '=123&test=456' )
        || ! dppp_is_valid_url( 'http://example.com/page.php?test=123&' . $parameter . '=456' ) ) {
        return false;
    }
    $url_parts = parse_url( 'http://example.com/page.php?' . $parameter . '=123&test=456' );
    $query = Array();
    if ( ! is_array( $url_parts ) || ! array_key_exists( 'query', $url_parts ) ) {
        return false;
    }
    parse_str( $url_parts['query'], $query );
    if ( ! is_array( $query ) || ! array_key_exists( $parameter, $query ) || ! array_key_exists( 'test', $query ) || count( $query ) !== 2 ) {
        return false;
    }
    if ( intval( $query[ $parameter ] ) !== 123 || intval( $query['test'] ) !== 456 ) {
        return false;
    }
    return true;
}

/**
 * Checks if the string is a comma-separated list of URI parameters (also only one parameter is allowed too).
 * @param string $string
 * @return bool
 */
function dppp_is_comma_separated_uri_parameters( $string ) {
    if ( strpos( $string, "\n" ) !== false || strpos( $string, "\r" ) !== false
        || ( strpos( $string, ' ' ) !== false && substr_count( $string, ' ' ) !== substr_count( $string, ',' ) ) ) {
        return false;
    }
    if ( strpos( $string, ',' ) === false ) {
        return dppp_is_valid_uri_parameter_name( $string );
    } else {
        $parameters = explode( ',', $string );
        if ( ! is_array( $parameters ) ) {
            return false;
        }
        foreach ( $parameters as $parameter ) {
            if ( ! dppp_is_valid_uri_parameter_name( dppp_strip_whitespace( $parameter ) ) ) {
                return false;
            }
        }
    }
    return true;
}

/**
 * Removes all spaces, tabs, new lines from a string
 * @param string $string
 * @return string
 */
function dppp_strip_whitespace( $string ) {
    return preg_replace( '/\s+/', '', $string );
}

// An ajax request that makes this device local (creates a cookie)
function dppp_make_device_local_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }
    $cookie = dppp_get_option( 'dppp-local-mode-cookie' );
    $returned = setcookie( $cookie, $cookie, time() + (86400 * 365), COOKIEPATH, COOKIE_DOMAIN );
    if ( ! $returned ) {
        wp_die( 'failed-cookie' );
    }
    wp_die( '[done]' );
}

// An ajax request that removes this local device (removes a cookie)
function dppp_remove_local_device_action_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }
    $cookie = dppp_get_option( 'dppp-local-mode-cookie' );
    $returned = setcookie( $cookie, $cookie, time() - (86400 * 365), COOKIEPATH, COOKIE_DOMAIN );
    if ( ! $returned ) {
        wp_die( 'failed-remove-cookie' );
    }
    wp_die( '[done]' );
}

// An ajax request that gets the plugin icon address and stores it in the database
function dppp_get_plugin_icon_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }
    $plugin_slug = sanitize_html_class( $_POST['plugin_slug'] );
    $plugin_file = dppp_sanitize_plugin_file( $_POST['plugin_file'] );
    $icon_url = dppp_get_plugin_icon_url( $plugin_slug );
    if ( dppp_is_valid_url( $icon_url ) && dppp_get_host_from_url( $icon_url ) === 'ps.w.org' ) {
        update_option( 'dppp-plugin-icon-' . md5( $plugin_file ), esc_url_raw( $icon_url ), false );
        wp_die( '[done]' . esc_url( $icon_url ) );
    } else {
        update_option( 'dppp-plugin-icon-' . md5( $plugin_file ), 'not-found', false );
        wp_die( '[done]not-found' );
    }
}

// An ajax request that pastes the copied rules adding them to the chosen plugin or group
function dppp_paste_rules_callback() {
    if ( ! check_ajax_referer( 'dppp_options_ajax_nonce', 'security', false ) ) {
        wp_die( 'invalid-nonce' );
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'no-access' );
    }

    $plugin_file = dppp_sanitize_plugin_file( $_POST['plugin_file'] );
    $rule_ids_string = dppp_strip_whitespace( $_POST['rule_ids'] );

    if ( ! dppp_is_comma_separated_numbers( $rule_ids_string ) ) {
        wp_die( 'invalid-data' );
    }
    if ( ! dppp_is_valid_plugin_file( $plugin_file ) && 'dppp-group' !== $plugin_file ) {
        wp_die( 'invalid-data' );
    }

    if ( 'dppp-group' === $plugin_file ) {
        $group_hash = sanitize_html_class( $_POST['hash'] );
        $plugin_groups = dppp_get_option( 'dppp-plugin-groups' );
        if ( ! is_array( $plugin_groups ) || ! array_key_exists( $group_hash, $plugin_groups ) ) {
            wp_die( 'group-does-not-exist' );
        }
        $plugin_file = dppp_get_plugin_files_list_from_array( $plugin_groups[ $group_hash ]['plugins'] );
    }

    $rule_ids = explode( ',', $rule_ids_string );

    global $wpdb;
    $inserted_ids = Array();
    $failed = 0;
    $skipped = 0;

    foreach ( $rule_ids as $rule_id ) {
        $result = $wpdb->get_row( $wpdb->prepare( "SELECT status, plugin_file, deactivation_type, uri_type, page_ids, uri_condition, uri_value, devices, custom_function, note "
            . "FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE rule_id = %d", $rule_id ), ARRAY_A );
        if ( ! empty( $result ) ) {
            if ( $result['plugin_file'] === $plugin_file ) {
                $skipped++;
                continue;
            } else {
                $already_exists = $wpdb->query( $wpdb->prepare( "SELECT rule_id FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE plugin_file = %s "
                    . "AND deactivation_type = %s AND uri_type = %s AND page_ids = %s AND uri_condition = %s AND uri_value = %s AND devices = %s AND custom_function = %s",
                    $plugin_file, $result['deactivation_type'], $result['uri_type'], $result['page_ids'], $result['uri_condition'],
                    $result['uri_value'], $result['devices'], $result['custom_function'] ) );
                if ( $already_exists !== 0 ) {
                    $skipped++;
                    continue;
                } else {
                    $status_insert = $wpdb->insert(
                    	"{$wpdb->prefix}dppp_plugin_deactivation_rules",
                    	array(
                    		'status' => $result['status'],
                    		'plugin_file' => $plugin_file,
                    		'deactivation_type' => $result['deactivation_type'],
                    		'uri_type' => $result['uri_type'],
                    		'page_ids' => $result['page_ids'],
                    		'uri_condition' => $result['uri_condition'],
                    		'uri_value' => $result['uri_value'],
                    		'devices' => $result['devices'],
                    		'custom_function' => $result['custom_function'],
                            'note' => $result['note'],
                    		'unix_time_added' => time(),
                    	),
                    	array(
                    		'%s',
                    		'%s',
                    		'%s',
                    		'%s',
                    		'%s',
                    		'%s',
                    		'%s',
                    		'%s',
                    		'%s',
                    		'%s',
                    		'%d',
                    	)
                    );
                    if ( $status_insert !== false ) {
                        $inserted_ids[] = $wpdb->insert_id;
                    } else {
                        $failed++;
                    }
                }
            }
        }
    }

    if ( empty( $inserted_ids ) ) {
        if ( $skipped > 0 && $failed > 0 ) {
            wp_die( 'skipped-failed' );
        } else if ( $skipped > 0 && 0 === $failed ) {
            wp_die( 'skipped' );
        } else if ( $failed > 0 && 0 === $skipped ) {
            wp_die( 'failed' );
        } else {
            wp_die( 'error' );
        }
    } else {
        if ( $skipped > 0 && $failed > 0 ) {
            wp_die( 'done-skipped-failed' );
        } else if ( $skipped > 0 && 0 === $failed ) {
            wp_die( 'done-skipped' );
        } else if ( $failed > 0 && 0 === $skipped ) {
            wp_die( 'done-failed' );
        } else {
            wp_die( 'done' );
        }
    }
}

/**
 * Uses the plugin API to get the plugin icon URL address and returns it
 * @param string $plugin_slug
 * @return string
 */
function dppp_get_plugin_icon_url( $plugin_slug ) {
    if ( ! function_exists( 'plugins_api' ) ) {
        require_once ABSPATH . 'wp-admin/includes/plugin-install.php';
    }
    $icon_url = 'not-found';
    $arguments = Array(
        'slug' => $plugin_slug,
        'fields' => 'icons',
    );
    $results = plugins_api( 'plugin_information', $arguments );
    if ( is_object( $results ) && isset( $results->icons ) && is_array( $results->icons ) ) {
        if ( array_key_exists( '1x', $results->icons ) ) {
            $icon_url = $results->icons['1x'];
        } elseif ( array_key_exists( '2x', $results->icons ) ) {
            $icon_url = $results->icons['2x'];
        }
    }
    return $icon_url;
}

/**
 * Returns the plugin slug from the provided plugin file string
 * @param string $url
 * @return string
 */
function dppp_get_slug_from_plugin_file( $plugin_file ) {
    return substr( $plugin_file, 0, strpos( $plugin_file, '/' ) );
}

/**
 * Returns the host of the provided URL
 * @param string $url
 * @return string
 */
function dppp_get_host_from_url( $url ) {
    return parse_url( $url, PHP_URL_HOST );
}

/**
 * Deletes the admin menu cache on plugin activation, since this adds menus usually
 * @param string $plugin
 * @param bool $network_wide
 */
function dppp_delete_cache_on_plugin_activate( $plugin, $network_wide ) {
    if ( dppp_get_option( 'dppp-affect-back-end' ) === 'cache' && current_user_can( 'manage_options' ) ) {
        dppp_delete_admin_menus_cache();

        // On network activation we save the last time we did that for later when on subsites we may force a cache deletion
        if ( true === $network_wide ) {
            update_site_option( 'dppp-network-cache-plugin-status-changed', time() );
        }
    }
}

/**
 * Deletes the admin menu cache on plugin deactivation, since this removes menus usually
 * @param string $plugin
 * @param bool $network_deactivating
 */
function dppp_delete_cache_on_plugin_deactivate( $plugin, $network_deactivating ) {
    if ( dppp_get_option( 'dppp-affect-back-end' ) === 'cache' && current_user_can( 'manage_options' ) ) {
        dppp_delete_admin_menus_cache();

        // On network deactivation we save the last time we did that for later when on subsites we may force a cache deletion
        if ( true === $network_deactivating ) {
            update_site_option( 'dppp-network-cache-plugin-status-changed', time() );
        }
    }
}

/**
 * Deletes the admin menu cache on theme switch, since this creates or removes menus usually
 * @param string $old_name
 * @param object $old_theme
 */
function dppp_delete_cache_on_theme_switch( $old_name, $old_theme ) {
    if ( dppp_get_option( 'dppp-affect-back-end' ) === 'cache' && current_user_can( 'manage_options' ) ) {
        dppp_delete_admin_menus_cache();
    }
}

// Loads the styles and scripts needed for the admin menu cache feature
function dppp_register_admin_menus_cache_assets() {
    if ( is_array( $GLOBALS ) && array_key_exists( 'dppp_affected_by_custom_rules', $GLOBALS ) && $GLOBALS['dppp_affected_by_custom_rules'] === 'yes'
        && dppp_get_option( 'dppp-affect-back-end' ) === 'cache' && current_user_can( 'manage_options' ) ) {
        $is_network_admin = is_network_admin();
        $current_user_id = get_current_user_id();
        if ( ( true === $is_network_admin && get_site_option( 'dppp-cache-network-submenu-' . $current_user_id ) !== false
            && get_site_option( 'dppp-cache-network-menu-' . $current_user_id ) !== false
            && get_site_option( 'dppp-cache-network-menu-html-' . $current_user_id ) !== false )
            || ( false === $is_network_admin && get_option( 'dppp-cache-submenu-' . $current_user_id ) !== false
            && get_option( 'dppp-cache-menu-' . $current_user_id ) !== false && get_option( 'dppp-cache-menu-html-' . $current_user_id ) !== false ) ) {

            // This script loads on pages that have the admin menu cached. It removes the non-working collapse button and replaces it with the working one.
            wp_register_script( 'dppp-admin-menus-cache-script', dppp_plugin_dir_url() . 'scripts/admin-menus-cache.js', array(),
                DPPP_DEACTIVATE_PLUGINS_PER_PAGE_VERSION, false );
            wp_enqueue_script( 'dppp-admin-menus-cache-script', '', array(), false, true );

            wp_register_style( 'dppp-admin-menus-cache-style', dppp_plugin_dir_url() . 'styles/admin-menus-cache.css',
                false, DPPP_DEACTIVATE_PLUGINS_PER_PAGE_VERSION );
            wp_enqueue_style( 'dppp-admin-menus-cache-style' );
        } else {
            wp_register_style( 'dppp-admin-menus-no-cache-style', dppp_plugin_dir_url() . 'styles/admin-menus-no-cache.css',
                false, DPPP_DEACTIVATE_PLUGINS_PER_PAGE_VERSION );
            wp_enqueue_style( 'dppp-admin-menus-no-cache-style' );
        }

        if ( true === $is_network_admin ) {
            $cache_age = time() - intval( get_site_option( 'dppp-cache-network-menu-age-' . $current_user_id ) );
            $network_recreate_url = esc_url( network_admin_url( 'settings.php?dppp-ignore-custom-rules=yes' ) );
        } else {
            $cache_age = time() - intval( get_option( 'dppp-cache-menu-age-' . $current_user_id ) );
            $network_recreate_url = '';
        }

        $expire_time = intval( dppp_get_option( 'dppp-admin-menu-cache-time' ) );
        if ( $expire_time < 300 ) {
            $expire_time = 1800;
        }

        // When this script is loaded, is the cache is old enough, a hidden iframe will load automatically that recreates the admin menu cache
        wp_register_script( 'dppp-recreate-cache-script', dppp_plugin_dir_url() . 'scripts/recreate-cache.js', array(),
            DPPP_DEACTIVATE_PLUGINS_PER_PAGE_VERSION, false );
        $localize = array(
            'recreateCacheURL' => esc_url( admin_url( 'options-general.php?dppp-ignore-custom-rules=yes' ) ),
            'recreateNetworkCacheURL' => $network_recreate_url,
            'isNetworkAdmin' => intval( $is_network_admin ),
            'cacheAge' => intval( $cache_age ),
            'cacheExpireTime' => $expire_time,
        );
        wp_localize_script( 'dppp-recreate-cache-script', 'dpppLocalizedRecreate', $localize );
        wp_enqueue_script( 'dppp-recreate-cache-script', '', array(), false, true );
    }
}

// Here we decide if we should output the admin menu cache later or a message or do nothing
function dppp_prepare_cached_admin_menus() {
    if ( is_array( $GLOBALS ) && array_key_exists( 'dppp_affected_by_custom_rules', $GLOBALS ) && $GLOBALS['dppp_affected_by_custom_rules'] === 'yes'
        && dppp_get_option( 'dppp-affect-back-end' ) === 'cache' && current_user_can( 'manage_options' ) ) {
        $current_user_id = get_current_user_id();
        if ( is_network_admin() ) {
            if ( get_site_option( 'dppp-cache-network-submenu-' . $current_user_id ) !== false
                && get_site_option( 'dppp-cache-network-menu-' . $current_user_id ) !== false
                && get_site_option( 'dppp-cache-network-menu-html-' . $current_user_id ) !== false ) {
                add_action( 'adminmenu', 'dppp_output_cached_admin_menus' );
            } else {
                add_action( 'adminmenu', 'dppp_output_no_admin_menus_cache' );
            }
        } else {
            if ( get_option( 'dppp-cache-submenu-' . $current_user_id ) !== false && get_option( 'dppp-cache-menu-' . $current_user_id ) !== false
                && get_option( 'dppp-cache-menu-html-' . $current_user_id ) !== false ) {

                /*
                 * If this is a multisite and not the main site, and this site has not seen the last time a plugin was network activated/deactivated
                 * we delete the admin menu cache for this site, and change the time of the last seen network activation/deactivation.
                 * Otherwise we show the cached admin menus.
                 */
                if ( is_multisite() && dppp_get_main_site_id() !== get_current_blog_id()
                    && intval( get_option( 'dppp-cache-seen-network-plugin-status-changed' ) )
                    < intval( get_site_option( 'dppp-network-cache-plugin-status-changed' ) ) ) {
                    dppp_delete_admin_menus_cache();
                    update_option( 'dppp-cache-seen-network-plugin-status-changed', time() );
                    add_action( 'adminmenu', 'dppp_output_no_admin_menus_cache' );
                } else {
                    add_action( 'adminmenu', 'dppp_output_cached_admin_menus' );
                }
            } else {
                add_action( 'adminmenu', 'dppp_output_no_admin_menus_cache' );
            }
        }
    }
}

// Outputs a message that the admin menu cache was not found on this request
function dppp_output_no_admin_menus_cache() {
    if ( is_network_admin() ) {
        echo '<a class="dppp-admin-menu-message" href="#">' . esc_html__( 'No network admin menus cache found.', 'dppp' ) . '</a>'
            . '<a class="dppp-admin-menu-cached-link" href="'
            . esc_url( admin_url( 'plugins.php?page=deactivate-plugins-per-page#dppp-settings-section' ) ) . '">- '
            . esc_html__( 'Go to settings', 'dppp' ) . '</a>'
            . '<a class="dppp-admin-menu-cached-link" href="javascript:dpppRecreateNetworkAdminMenuCacheAndReload()">- '
            . esc_html__( 'Recreate cache and reload the page', 'dppp' ) . '</a>';
    } else {
        echo '<a class="dppp-admin-menu-message" href="#">' . esc_html__( 'No admin menus cache found.', 'dppp' ) . '</a>'
            . '<a class="dppp-admin-menu-cached-link" href="'
            . esc_url( admin_url( 'plugins.php?page=deactivate-plugins-per-page#dppp-settings-section' ) ) . '">- '
            . esc_html__( 'Go to settings', 'dppp' ) . '</a>'
            . '<a class="dppp-admin-menu-cached-link" href="javascript:dpppRecreateAdminMenuCacheAndReload()">- '
            . esc_html__( 'Recreate cache and reload the page', 'dppp' ) . '</a>';
    }
}

// We autput the HTML of the cached admin menus (we will hide the original admin menu with CSS)
function dppp_output_cached_admin_menus() {

    // These variables will hold the HTML for the little numbers in some menus that indicate number of updates or pending comments, so they are not cached.
    $submenu_updates = $menu_comments = $menu_plugins = $menu_themes = false;

    // These are the not cached menu and submenu arrays
    global $submenu, $menu;

    // We find and get the HTML of the number of updates in the updates submenu
    if ( is_array( $submenu ) && array_key_exists( 'index.php', $submenu ) && is_array( $submenu['index.php'] ) ) {
        foreach ( $submenu['index.php'] as $key => $value ) {
            if ( is_array( $value ) ) {
                foreach ( $value as $inner_key => $inner_value ) {
                    if ( false === $submenu_updates && ! is_array( $inner_value ) && 'update-core.php' === $inner_value ) {
                        $submenu_updates = $submenu['index.php'][ $key ][0];
                        break 2;
                    }
                }
            }
        }
    }

    // We find and get the HTML of the number of updates in the plugins and themes menus and the number of pending comments in the comments menu
    if ( is_array( $menu ) ) {
        foreach ( $menu as $key => $value ) {
            if ( is_array( $value ) ) {
                foreach ( $value as $inner_key => $inner_value ) {
                    if ( false === $menu_comments && ! is_array( $inner_value ) && 'edit-comments.php' === $inner_value ) {
                        $menu_comments = $menu[ $key ][0];
                    }
                    if ( false === $menu_plugins && ! is_array( $inner_value ) && 'plugins.php' === $inner_value ) {
                        $menu_plugins = $menu[ $key ][0];
                    }
                    if ( false === $menu_themes && ! is_array( $inner_value ) && 'themes.php' === $inner_value ) {
                        $menu_themes = $menu[ $key ][0];
                    }
                }
            }
        }
    }

    $current_user_id = get_current_user_id();
    $is_network_admin = is_network_admin();

    if ( true === $is_network_admin ) {
        $cache_submenu = get_site_option( 'dppp-cache-network-submenu-' . $current_user_id );

        // Here we will set the number of network updates in the updates submenu, so it is not cached
        if ( array_key_exists( 'index.php', $cache_submenu ) ) {
            $submenu_updates_index = dppp_get_array_index_if_inner_is( $cache_submenu['index.php'], 'update-core.php' );
            if ( false !== $submenu_updates && false !== $submenu_updates_index ) {
                $cache_submenu['index.php'][ $submenu_updates_index ][ 0 ] = $submenu_updates;
            }
        }
        $cache_menu =  get_site_option( 'dppp-cache-network-menu-' . $current_user_id );
        $menu_themes_index = dppp_get_array_index_if_inner_is( $cache_menu, 'themes.php' );
        $menu_plugins_index = dppp_get_array_index_if_inner_is( $cache_menu, 'plugins.php' );

        // Here we will set the number of plugin updates in the network plugins menu, so it is not cached
        if ( false !== $menu_plugins && false !== $menu_plugins_index && array_key_exists( $menu_plugins_index, $cache_menu ) ) {
            $cache_menu[ $menu_plugins_index ][0] = $menu_plugins;
        }

        // Here we will set the number of theme updates in the network themes menu, so it is not cached
        if ( false !== $menu_themes && false !== $menu_themes_index && array_key_exists( $menu_themes_index, $cache_menu ) ) {
            $cache_menu[ $menu_themes_index ][0] = $menu_themes;
        }
        $menu_html = strip_tags( wp_kses_post( get_site_option( 'dppp-cache-network-menu-html-' . $current_user_id ) ), '<a>' );
        $cache_age = time() - intval( get_site_option( 'dppp-cache-network-menu-age-' . $current_user_id ) );
    } else {
        $cache_submenu = get_option( 'dppp-cache-submenu-' . $current_user_id );

        // Here we will set the number of updates in the updates submenu, so it is not cached
        if ( array_key_exists( 'index.php', $cache_submenu ) ) {
            $submenu_updates_index = dppp_get_array_index_if_inner_is( $cache_submenu['index.php'], 'update-core.php' );
            if ( false !== $submenu_updates && false !== $submenu_updates_index ) {
                $cache_submenu['index.php'][ $submenu_updates_index ][ 0 ] = $submenu_updates;
            }
        }
        $cache_menu =  get_option( 'dppp-cache-menu-' . $current_user_id );
        $menu_comments_index = dppp_get_array_index_if_inner_is( $cache_menu, 'edit-comments.php' );
        $menu_plugins_index = dppp_get_array_index_if_inner_is( $cache_menu, 'plugins.php' );
        $menu_themes_index = dppp_get_array_index_if_inner_is( $cache_menu, 'themes.php' );

        // Here we will set the number of pending comments in the comments menu, so it is not cached
        if ( false !== $menu_comments && false !== $menu_comments_index ) {
            $cache_menu[ $menu_comments_index ][0] = $menu_comments;
        }

        // Here we will set the number of plugin updates in the plugins menu, so it is not cached
        if ( false !== $menu_plugins && false !== $menu_plugins_index ) {
            $cache_menu[ $menu_plugins_index ][0] = $menu_plugins;
        }

        // Here we will set the number of theme updates in the themes menu, so it is not cached
        if ( false !== $menu_themes && false !== $menu_themes_index ) {
            $cache_menu[ $menu_themes_index ][0] = $menu_themes;
        }
        $menu_html = strip_tags( wp_kses_post( get_option( 'dppp-cache-menu-html-' . $current_user_id ) ), '<a>' );
        $cache_age = time() - intval( get_option( 'dppp-cache-menu-age-' . $current_user_id ) );
    }

    // We get the output HTML of the admin menus with our cached menu and submenu arrays
    ob_start();
    _wp_menu_output( $cache_menu, $cache_submenu );
    $output = ob_get_contents();
    ob_end_clean();

    // We find all the href attribute values of the links in the cached admin menus output
    preg_match_all("/href='(.*?)'/s", $output, $broken_matches);
    $broken_href = $broken_matches[1];

    // We find all the href attribute values of the links in the admin menu HTML code we saved earlier for this purpose
    preg_match_all("/href='(.*?)'/s", $menu_html, $original_matches);
    $original_href = $original_matches[1];

    $count_broken = count( $broken_href );
    $count_original = count( $original_href );

    /*
     * We will fix all the href values that are different from the original, otherwise they are broken for the deactivated links
     * In case there are some menus missing (th ecounts are different) we fix the links in a different way by making sure the string
     * in the broken version is contained in the fixed version to avoid problems with incorrect places for links.
     */
    if ( $count_broken < $count_original ) {
        $offset_broken = 0;
        for ( $index = 0; $index < $count_original; $index++ ) {
            $index_broken = $index + $offset_broken;
            $index_original = $index;
            if ( isset( $broken_href[ $index_broken ] ) && isset( $original_href[ $index_original ] )
                && $broken_href[ $index_broken ] !== $original_href[ $index_original ] ) {
                if ( strpos( $original_href[ $index_original ], $broken_href[ $index_broken ] ) !== false ) {
                    $output = str_replace( "href='" . $broken_href[ $index_broken ], "href='" . esc_attr( $original_href[ $index_original ] ), $output );
                } else {
                    $offset_broken--;
                }
            }
        }
    } else {
        for ( $index = 0; $index < $count_broken; $index++ ) {
            if ( $broken_href[ $index ] !== $original_href[ $index ] ) {
                $output = str_replace( "href='" . $broken_href[ $index ], "href='" . esc_attr( $original_href[ $index ] ), $output );
            }
        }
    }

    // We replace the id of the collapse menu button. We will remove this button later with jQuery and put the working one in its place.
    $output = str_replace( "collapse-menu", "dppp-collapse-menu", $output );

    // We output the cached admin menus
    echo '<div id="dppp">' . $output . '</div>';

    // We calculate the cache age string so it says minutes, hours or days from seconds given
    if ( $cache_age > ( 3600 * 24 * 3 ) ) {
        $cache_age_number = round( $cache_age / ( 24 * 3600 ) );
        $cache_age_string = esc_html__( 'days', 'dppp' );
    } elseif ( $cache_age > ( 3600 * 2 ) ) {
        $cache_age_number = round( $cache_age / 3600 );
        $cache_age_string = esc_html__( 'hours', 'dppp' );
    } elseif ( $cache_age > 120 ) {
        $cache_age_number = round( $cache_age / 60 );
        $cache_age_string = esc_html__( 'min', 'dppp' );
    } else {
        $cache_age_number = $cache_age;
        $cache_age_string = esc_html__( 'sec', 'dppp' );
    }

    // Below we output information about the admin menu cache and links to recreate it or to the plugin settings
    echo '<a href="#" class="dppp-admin-menu-message">';
    if ( true === $is_network_admin ) {
        echo esc_html__( 'The network admin menus are cached.', 'dppp' );
    } else {
        echo esc_html__( 'The admin menus are cached.', 'dppp' );
    }
    echo ' ' . sprintf( esc_html__( 'Cache age: %d %s.', 'dppp' ), $cache_age_number, $cache_age_string ) . '</a>'
        . '<a class="dppp-admin-menu-cached-link" href="'
        . esc_url( admin_url( 'plugins.php?page=deactivate-plugins-per-page#dppp-settings-section' ) ) . '">- '
        . esc_html__( 'Go to settings', 'dppp' ) . '</a>';
    if ( true === $is_network_admin ) {
        echo '<a class="dppp-admin-menu-cached-link" href="javascript:dpppRecreateNetworkAdminMenuCacheAndReload()">- '
            . esc_html__( 'Recreate cache and reload the page', 'dppp' ) . '</a>';
    } else {
        echo '<a class="dppp-admin-menu-cached-link" href="javascript:dpppRecreateAdminMenuCacheAndReload()">- '
            . esc_html__( 'Recreate cache and reload the page', 'dppp' ) . '</a>';
    }
}

// Creates or recreates the admin menu cache
function dppp_create_admin_menu_cache() {

    /*
     * Recreate the cache if we haven't deactivated plugins on this admin page with custom rules and also if the page is not from the ones
     * that we skip deactivation on, since globally deactivated plugins on the back-end will have menus there and they will be added to the cache.
     * But we do create the cache on this plugin settings page if a special parameter is present, indicating that we just enabled
     * the admin menu cache feature (when we do that we also apply the global deactivation rules just for a little while and then reload).
     */
    if ( current_user_can( 'manage_options' ) && is_array( $GLOBALS ) && ! array_key_exists( 'dppp_affected_by_custom_rules', $GLOBALS )
        && dppp_get_option( 'dppp-affect-back-end' ) === 'cache' && ( ! dppp_skip_this_admin_page()
        || strpos( $_SERVER['REQUEST_URI'], 'plugins.php?page=deactivate-plugins-per-page&dppp-recreate-cache=yes' ) !== false ) ) {

        $expire_time = intval( dppp_get_option( 'dppp-admin-menu-cache-time' ) );
        if ( $expire_time < 300 ) {
            $expire_time = 1800;
        }

        $current_user_id = get_current_user_id();
        $is_network_admin = is_network_admin();
        if ( true === $is_network_admin ) {
            $cache_age = time() - intval( get_site_option( 'dppp-cache-network-menu-age-' . $current_user_id ) );
        } else {
            $cache_age = time() - intval( get_option( 'dppp-cache-menu-age-' . $current_user_id ) );
        }

        // We recreate the cache if it is too old or if this is the URL for manual recreation
        if ( $cache_age > $expire_time || strpos( $_SERVER['REQUEST_URI'], '?dppp-ignore-custom-rules=yes' ) !== false ) {
            global $submenu, $menu;
            if ( is_array( $submenu ) && is_array( $menu ) && function_exists( '_wp_menu_output' ) ) {

                ob_start();
                _wp_menu_output( $menu, $submenu );
                $output = ob_get_contents();
                ob_end_clean();

                if ( true === $is_network_admin ) {
                    update_site_option( 'dppp-cache-network-submenu-' . $current_user_id, $submenu );
                    update_site_option( 'dppp-cache-network-menu-' . $current_user_id, $menu );
                    update_site_option( 'dppp-cache-network-menu-age-' . $current_user_id, time() );

                    // We will use this HTML code to see the correct links and fix them in our output HTML, they seem to break when the plugin is deactivated
                    update_site_option( 'dppp-cache-network-menu-html-' . $current_user_id, $output );
                } else {
                    update_option( 'dppp-cache-submenu-' . $current_user_id, $submenu, false );
                    update_option( 'dppp-cache-menu-' . $current_user_id, $menu, false );
                    update_option( 'dppp-cache-menu-age-' . $current_user_id, time(), false );

                    // We will use this HTML code to see the correct links and fix them in our output HTML, they seem to break when the plugin is deactivated
                    update_option( 'dppp-cache-menu-html-' . $current_user_id, $output, false );
                }
            }
        }
    }
}

// Deletes the admin menu cache and network admin menu cache
function dppp_delete_admin_menus_cache() {
    global $wpdb;
    $results = $wpdb->get_results( "SELECT option_name FROM {$wpdb->prefix}options WHERE option_name LIKE '%dppp-cache-%'", ARRAY_A );
    if ( ! empty( $results ) ) {
        foreach ( $results as $result ) {
            if ( strpos( $result['option_name'], 'dppp-cache-' ) !== false ) {
                delete_option( $result['option_name'] );
            }
        }
    }

    /*
     * If it is a multisite and we are in the main site we also delete the network admin menu cache, since plugins on the main site
     * can affect the menus in the network dashboard sometimes.
     */
    if ( is_multisite() && dppp_get_main_site_id() == get_current_blog_id() ) {

        // The database table prefix for the main network site
        $main_blog_prefix = $wpdb->get_blog_prefix( dppp_get_main_site_id() );

        $results = $wpdb->get_results( "SELECT meta_key FROM {$main_blog_prefix}sitemeta WHERE meta_key LIKE '%dppp-cache-%'", ARRAY_A );
        if ( ! empty( $results ) ) {
            foreach ( $results as $result ) {
                if ( strpos( $result['meta_key'], 'dppp-cache-' ) !== false ) {
                    delete_site_option( $result['meta_key'] );
                }
            }
        }
    }
}

/**
 * Returns the id of the main site of the network (which is usually 1)
 * @return int
 */
function dppp_get_main_site_id() {
    if ( dppp_wordpress_version_is( '>=', '4.9' ) ) {
        return get_main_site_id();
    } else {
        global $current_site;
        return intval( $current_site->blog_id );
    }
}

/**
 * Checks if the statement put together with the two parameters is true or false for the WordPress version.
 * @param string $operator
 * @param string $version
 * @return bool
 */
function dppp_wordpress_version_is( $operator, $version ) {
    global $wp_version;
    return version_compare( $wp_version, $version, $operator );
}

/**
 * Returns the index of an array inside an array if any of the values of the inner array are equal to a string.
 * @param string $array
 * @param string $string
 * @return mixed
 */
function dppp_get_array_index_if_inner_is( $array, $string ) {
    if ( is_array( $array ) ) {
        foreach ( $array as $key => $value ) {
            if ( is_array( $value ) ) {
                foreach ( $value as $inner_value ) {
                    if ( is_string( $inner_value ) && $string === $inner_value ) {
                        return $key;
                    }
                }
            }
        }
    }
    return false;
}

/**
 * Checks if a string is a valid function name
 * @param string $function_name
 * @return bool
 */
function dppp_is_valid_php_function_name( $function_name ) {
    if ( ! preg_match( '/^[a-zA-Z_]/', $function_name ) ) {
        return false;
    }
    if ( ! preg_match( '/^[a-zA-Z0-9_]*$/', $function_name ) ) {
        return false;
    }
    $reserved_words = array(
        '__halt_compiler', 'abstract', 'and', 'array', 'as', 'break',
        'callable', 'case', 'catch', 'class', 'clone', 'const', 'continue',
        'declare', 'default', 'die', 'do', 'echo', 'else', 'elseif', 'empty',
        'enddeclare', 'endfor', 'endforeach', 'endif', 'endswitch', 'endwhile',
        'eval', 'exit', 'extends', 'final', 'for', 'foreach', 'function',
        'global', 'goto', 'if', 'implements', 'include', 'include_once',
        'instanceof', 'insteadof', 'interface', 'isset', 'list', 'namespace',
        'new', 'or', 'print', 'private', 'protected', 'public', 'require',
        'require_once', 'return', 'static', 'switch', 'throw', 'trait', 'try',
        'unset', 'use', 'var', 'while', 'xor'
    );
    if ( in_array( $function_name, $reserved_words ) ) {
        return false;
    }
    return true;
}

/**
 * Checks if a string is a valid regular expression
 * @param string $regular_expression
 * @return bool
 */
function dppp_is_valid_regular_expression( $regular_expression ) {
    if( @preg_match( $regular_expression, null ) === false ) {
        return false;
    }
    return true;
}

/**
 * The pluigin WP-Optimize will show the database table of this plugin as belonging to another free plugin that have probably copied code from this plugin
 * and that is why they have the save table name. This is why WP-Optimize will offer to delete the database table of this plugin, thinking it is for a not installed plugin,
 * which is a mistake. We hide this option and message to avoid unwanted deleteion by mistake.
 * @param string $content
 * @return string
 */
function dppp_fix_wp_optimize_table_detection( $content ) {
    global $wpdb;
    $the_prefix = $wpdb->prefix;
    $content = str_replace( $the_prefix . 'dppp_plugin_deactivation_rules<div class="table-plugins">', esc_html( $the_prefix )
        . 'dppp_plugin_deactivation_rules<div class="table-plugins" style="display:none">', $content );
    $content = str_replace( '<div><button class="button button-secondary run-single-table-delete" data-table="' . esc_attr( $the_prefix )
        . 'dppp_plugin_deactivation_rules"', '<div style="display:none"><button class="button button-secondary run-single-table-delete" data-table="'
        . esc_attr( $the_prefix ) . 'dppp_plugin_deactivation_rules"', $content );
    return $content;
}

/**
 * Returns the current server URI to use for rules with Page URI selection (we could ignore parameters or not)
 * IMPORTANT: This function is declared in both the normal plugin file and the mu-plugin file. This way we can use it in the mu-plugin before the normal
 * plugin is loaded, and also use it in the normal plugin in case the mu-plugin is missing. When changes are made to it, both places need to be updated.
 * @param string
 * @return string
 */
if ( ! function_exists( 'dppp_get_uri_for_page_rules' ) ) {
    function dppp_get_uri_for_page_rules( $decoded_uri ) {
        $uri_parameters_page = dppp_get_option( 'dppp-uri-parameters' );
        if ( get_option( 'permalink_structure' ) !== '' && 'ignore-all' === $uri_parameters_page ) {
            return strtok( $decoded_uri, '?' );
        } elseif ( 'ignore-chosen' === $uri_parameters_page ) {
            $chosen_parameters_string = dppp_get_option( 'dppp-chosen-uri-parameters' );
            $chosen_parameters_string = preg_replace( '/\s+/', '', $chosen_parameters_string );
            $chosen_parameters_array = explode( ',', $chosen_parameters_string );
            return rawurldecode( remove_query_arg( $chosen_parameters_array, $decoded_uri ) );
        } else {
            return $decoded_uri;
        }

    }
}

/**
 * Returns the current server URI to use for rules with Custom URI selection (we could ignore parameters or some or none)
 * IMPORTANT: This function is declared in both the normal plugin file and the mu-plugin file. This way we can use it in the mu-plugin before the normal
 * plugin is loaded, and also use it in the normal plugin in case the mu-plugin is missing. When changes are made to it, both places need to be updated.
 * @param string
 * @return string
 */
if ( ! function_exists( 'dppp_get_uri_for_custom_rules' ) ) {
    function dppp_get_uri_for_custom_rules( $decoded_uri ) {
        $uri_parameters_custom = dppp_get_option( 'dppp-uri-parameters-custom' );
        if ( get_option( 'permalink_structure' ) !== '' && 'ignore-all' === $uri_parameters_custom ) {
            return strtok( $decoded_uri, '?' );
        } elseif ( 'ignore-chosen' === $uri_parameters_custom ) {
            $chosen_parameters_string = dppp_get_option( 'dppp-chosen-uri-parameters' );
            $chosen_parameters_string = preg_replace( '/\s+/', '', $chosen_parameters_string );
            $chosen_parameters_array = explode( ',', $chosen_parameters_string );
            return rawurldecode( remove_query_arg( $chosen_parameters_array, $decoded_uri ) );
        } else {
            return $decoded_uri;
        }
    }
}

/**
 * Returns true if we are on an admin page that we need to not affect by rules.
 * IMPORTANT: This function is declared in both the normal plugin file and the mu-plugin file. This way we can use it in the mu-plugin before the normal
 * plugin is loaded, and also use it in the normal plugin in case the mu-plugin is missing. When changes are made to it, both places need to be updated.
 * @return bool
 */
if ( ! function_exists( 'dppp_skip_this_admin_page' ) ) {
    function dppp_skip_this_admin_page() {
        if ( is_admin() && ( ( isset( $_GET['page'] ) && 'deactivate-plugins-per-page' === $_GET['page'] )
            || ( strpos( $_SERVER['REQUEST_URI'], '/plugins.php' ) !== false && ! isset( $_GET['page'] ) )
            || strpos( $_SERVER['REQUEST_URI'], '/plugin-install.php' ) !== false
            || strpos( $_SERVER['REQUEST_URI'], '/plugin-editor.php' ) !== false
            || strpos( $_SERVER['REQUEST_URI'], '/update-core.php' ) !== false
            || strpos( $_SERVER['REQUEST_URI'], '/update.php' ) !== false
            || strpos( $_SERVER['REQUEST_URI'], '/widgets.php?legacy-widget-preview' ) !== false ) ) {
            return true;
        }
        return false;
    }
}

/**
 * Returns true if we are on a network admin page that we need to not affect by rules
 * IMPORTANT: This function is declared in both the normal plugin file and the mu-plugin file. This way we can use it in the mu-plugin before the normal
 * plugin is loaded, and also use it in the normal plugin in case the mu-plugin is missing. When changes are made to it, both places need to be updated.
 * @return bool
 */
if ( ! function_exists( 'dppp_skip_this_network_admin_page' ) ) {
    function dppp_skip_this_network_admin_page() {
        if ( is_network_admin() && ( ( strpos( $_SERVER['REQUEST_URI'], '/plugins.php' ) !== false && ! isset( $_GET['page'] ) )
            || strpos( $_SERVER['REQUEST_URI'], '/plugin-install.php' ) !== false
            || strpos( $_SERVER['REQUEST_URI'], '/plugin-editor.php' ) !== false
            || strpos( $_SERVER['REQUEST_URI'], '/update-core.php' ) !== false
            || strpos( $_SERVER['REQUEST_URI'], '/update.php' ) !== false ) ) {
            return true;
        }
        return false;
    }
}

/**
 * Gets an option value by name. If it does not exist, and we have a default value for it, returns the default value.
 * IMPORTANT: This function is declared in both the normal plugin file and the mu-plugin file. This way we can use it in the mu-plugin before the normal
 * plugin is loaded, and also use it in the normal plugin in case the mu-plugin is missing. When changes are made to it, both places need to be updated.
 * @param string $name
 * @return mixed
 */
if ( ! function_exists( 'dppp_get_option' ) ) {
    function dppp_get_option( $name ) {
        $value = get_option( $name );
        if ( is_bool( $value ) && false === $value ) {
            $default_options = Array(
                'dppp-database-table-created' => 'no',
                'dppp-hide-group-plugins' => 'off',
                'dppp-debug-mode' => 'off',
                'dppp-inactive-plugins' => 'off',
                'dppp-completely-remove' => 'off',
                'dppp-groups-first' => 'off',
                'dppp-trailing-slash' => 'original',
                'dppp-affect-back-end' => 'off',
                'dppp-uri-parameters' => 'ignore-chosen',
                'dppp-uri-parameters-custom' => 'ignore-chosen',
                'dppp-chosen-uri-parameters' => 'fbclid, fb_action_ids, fb_action_types, fb_source, gclid, _ga, age-verified, ao_noptimize, usqp, '
                    . 'cn-reloaded, utm_medium, utm_expid, utm_term, utm_source, utm_campaign, utm_content, msclkid, dclid, ignorenitro, v',
                'dppp-section-show-hide-settings' => 'show',
                'dppp-section-show-hide-active-plugins' => 'show',
                'dppp-section-show-hide-inactive-plugins' => 'show',
                'dppp-section-show-hide-plugin-groups' => 'show',
                'dppp-plugin-icons' => 'on',
                'dppp-local-mode' => 'off',
                'dppp-rules-first' => 'off',
                'dppp-hide-plugins-in-group' => 'off',
                'dppp-priority' => 10,
                'dppp-affect-visual-editors' => 'off',
                'dppp-edit-post-parameter' => 'off',
                'dppp-admin-menu-cache-time' => 1800,
            );
            if ( array_key_exists( $name, $default_options ) ) {
                return $default_options[ $name ];
            } else {
                return false;
            }
        } else {
            return $value;
        }
    }
}

/**
 * Checks if the string is a valid URL
 * IMPORTANT: This function is declared in both the normal plugin file and the mu-plugin file. This way we can use it in the mu-plugin before the normal
 * plugin is loaded, and also use it in the normal plugin in case the mu-plugin is missing. When changes are made to it, both places need to be updated.
 * @param string $string
 * @return bool
 */
if ( ! function_exists( 'dppp_is_valid_url' ) ) {
    function dppp_is_valid_url( $string ) {
        if ( filter_var( $string, FILTER_VALIDATE_URL ) === false ) {
            return false;
        }
        return true;
    }
}

/**
 * Checks if the current device is a local device (has the cookie for local mode).
 * IMPORTANT: This function is declared in both the normal plugin file and the mu-plugin file. This way we can use it in the mu-plugin before the normal
 * plugin is loaded, and also use it in the normal plugin in case the mu-plugin is missing. When changes are made to it, both places need to be updated.
 * @return bool
 */
if ( ! function_exists( 'dppp_is_local_device' ) ) {
    function dppp_is_local_device() {
        $cookie = dppp_get_option( 'dppp-local-mode-cookie' );
        if ( false !== $cookie && isset( $_COOKIE[ $cookie ] ) && $_COOKIE[ $cookie ] === $cookie ) {
            return true;
        }
        return false;
    }
}

/**
 * @author abzarwp
 */
require_once plugin_dir_path( __FILE__ ) . "abzarwp/abzarwp.php";